package cn.com.duiba.tuia.remoteservice.engine.impl;

import cn.com.duiba.boot.utils.WarningUtils;
import cn.com.duiba.tuia.constants.ABTestConstant;
import cn.com.duiba.tuia.constants.ErrorCode;
import cn.com.duiba.tuia.domain.model.AdvertFilter;
import cn.com.duiba.tuia.domain.model.CatMonitorWarnThreshold;
import cn.com.duiba.tuia.domain.model.FilterResult;
import cn.com.duiba.tuia.domain.model.ShieldStrategyVO;
import cn.com.duiba.tuia.domain.model.engine.BuildParametersRtn;
import cn.com.duiba.tuia.domain.vo.AdvertFilterVO;
import cn.com.duiba.tuia.enums.CatGroupEnum;
import cn.com.duiba.tuia.enums.ReqSourceTypeEnum;
import cn.com.duiba.tuia.exception.TuiaException;
import cn.com.duiba.tuia.log.AdvertFilterLog;
import cn.com.duiba.tuia.log.FilterResultLog;
import cn.com.duiba.tuia.service.AdvertLaunchMonitorService;
import cn.com.duiba.tuia.service.BaseService;
import cn.com.duiba.tuia.service.ImitateAdvertService;
import cn.com.duiba.tuia.service.LianTongIntegralConsumeService;
import cn.com.duiba.tuia.service.engine.EngineABTestService;
import cn.com.duiba.tuia.service.engine.EngineFilterService;
import cn.com.duiba.tuia.service.engine.EngineRiskCheckService;
import cn.com.duiba.tuia.service.engine.EngineSceneService;
import cn.com.duiba.tuia.service.impl.EngineServiceImpl;
import cn.com.duiba.tuia.tool.CatUtil;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.tuia.advert.exception.ReadableMessageException;
import cn.com.tuia.advert.model.*;
import cn.com.tuia.advert.model.dsp.AdxAdvertPriceDto;
import cn.com.tuia.advert.service.IEngineService;
import com.alibaba.fastjson.JSON;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Primary;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.FutureTask;

/**
 * 广告引擎服务 ImpL
 *
 * @author lijicong
 * @see cn.com.duiba.tuia.service.impl.EngineServiceImpl
 * @since 2020-09-29
 */
@RestController
@Primary
public class RemoteEngineServiceImpl extends BaseService implements IEngineService {

    @Resource
    private EngineServiceImpl engineServiceImpl;
    @Resource
    private EngineSceneService engineSendSceneService;
    @Resource
    private EngineRiskCheckService engineRiskCheckService;
    @Resource
    private ImitateAdvertService imitateAdvertService;
    @Resource
    private CatMonitorWarnThreshold catMonitorWarnThreshold;
    @Resource
    private AdvertLaunchMonitorService advertLaunchMonitorService;
    @Resource
    private EngineABTestService engineABTestService;

    @Autowired
    private EngineFilterService engineFilterService;

    @Autowired
    private LianTongIntegralConsumeService lianTongIntegralConsumeService;

    /**
     * 兑吧请求一个广告
     *
     * @param req
     * @return
     */
    @Override
    public ObtainAdvertRsp obtainAdvert(ObtainAdvertReq req) {
        return doObtainAdvert(req);
    }

    private ObtainAdvertRsp doObtainAdvert(ObtainAdvertReq req) {
        DBTimeProfile.enter("RemoteEngineServiceImpl.doObtainAdvert");
        ObtainAdvertRsp rsp = new ObtainAdvertRsp();
        rsp.setSlotId(req.getSlotId());

        AdvertFilter advertFilter = new AdvertFilter(req.getAppId(), req.getSlotId(), req.getOrderId(), req.getActivityId());
        //互动大盘类型
        advertFilter.setReqSourceType(ReqSourceTypeEnum.HD_ACTIVITY.getType());
        FilterResult filterResult = null;
        ShieldStrategyVO shieldStrategyVO = null;
        Map<Long, AdvertFilterVO> validAdverts = null;

        try {
            // 1. 构建参数
            BuildParametersRtn parametersRtn = CatUtil.executeInCatTransaction(() -> engineServiceImpl.buildParameters(req, rsp), "obtainAdvert", "buildParameters");
            filterResult = parametersRtn.getFilterResult();
            shieldStrategyVO = parametersRtn.getShieldStrategyVO();

            //设置活动类型 临时用
            advertFilter.setReqSourceType(null == req.getActivitySceneType()?ReqSourceTypeEnum.HD_ACTIVITY.getType():(10+req.getActivitySceneType()));

            logIfNecessary(req, parametersRtn);

            // 2.1 风控反作弊接口，命中则走特权库发券
            FutureTask<Map<String, Integer>> futureTask = engineRiskCheckService.asynRiskCheckAndSendIfHit(req, rsp, parametersRtn, advertFilter);

            // 3. 过滤得到有效的广告
            validAdverts = engineFilterService.getValidAdverts(parametersRtn.getAdvQueryParamTmp(), req, parametersRtn.getFilterResult(), advertFilter);

            // 3. 查询过滤广告
//            validAdverts = engineServiceImpl.preFilterAdvertWithEsHystrix(parametersRtn.getAdvQueryParamTmp(), req, parametersRtn.getFilterResult(), advertFilter);

            // 2.2 风控反作弊接口，异步返回结果
            boolean isHitRisk = engineRiskCheckService.isHitRisk(futureTask);
            if (isHitRisk) {
                return rsp;
            }

            // 4. 应用策略发券
            engineSendSceneService.applyScene(req, rsp, advertFilter, parametersRtn, validAdverts);

        } catch (Throwable ex) {
            // 异常处理
            handleException(ex, filterResult);
        } finally {
            // 记录日志与监控
            logAndMonitor(rsp, filterResult, advertFilter, shieldStrategyVO, validAdverts);
        }
        return rsp;
    }

    private void logIfNecessary(ObtainAdvertReq req, BuildParametersRtn parametersRtn) {
        // 添加advQueryParam 请求日志
        if (logConfig.getInfoEnable()) {
            logger.info("请求参数日志，orderId ={}，consumerId={}, param={}，appDetail={}",
                    req.getOrderId(), req.getConsumerId(), JSON.toJSONString(parametersRtn.getAdvQueryParamTmp()), JSON.toJSONString(parametersRtn.getAppDetail()));
        }
    }

    /**
     * 处理obtainAdvert的异常
     *
     * @param ex
     * @param filterResult
     */
    private void handleException(Throwable ex, FilterResult filterResult) {
        if (filterResult == null) {
            filterResult = new FilterResult();
        }
        if (ex instanceof TuiaException) {
            // 发券失败监控
            TuiaException e = (TuiaException) ex;
            CatUtil.catLog("obtainAdvertError" + e.getResultCode());
            WarningUtils.markThresholdWarning("obtainAdvertError", catMonitorWarnThreshold.getObtainAdvertError());
            filterResult.setResultCode(e.getResultCode());
        } else {
            // 发券异常监控
            CatUtil.catLog(CatGroupEnum.CAT_102009.getCode());
            filterResult.setResultCode(ErrorCode.E9999999.getErrorCode());
            WarningUtils.markThresholdWarning("obtainAdvertException", catMonitorWarnThreshold.getObtainAdvertExc());
            logger.error("obtainAdvert is exception ", ex);
        }
    }

    /**
     * 记录日志与监控
     *
     * @param rsp
     * @param filterResult
     * @param advertFilter
     * @param shieldStrategyVO
     * @param validAdverts
     */
    private void logAndMonitor(ObtainAdvertRsp rsp, FilterResult filterResult, AdvertFilter advertFilter, ShieldStrategyVO shieldStrategyVO, Map<Long, AdvertFilterVO> validAdverts) {
        if (filterResult == null) {
            filterResult = new FilterResult();
        }
        filterResult.setStrategyType(rsp.getStrategyType());
        // 组装广告过滤日志
        try {
            engineServiceImpl.packAdvertFilterLog(advertFilter, filterResult, validAdverts);
        } catch (Exception e) {
            logger.error("packAdvertFilterLog is exception ", e);
        }

        // 清空重复发券与nezha过滤
        filterResult.setNezhaFilterMap(null);

        AdvertFilterLog.log(advertFilter);
        FilterResultLog.innerLog(filterResult);
        DBTimeProfile.release();

        filterResult.setShieldStrategyVO(shieldStrategyVO);

        // 发送监控消息
        advertLaunchMonitorService.monitorHandle(filterResult);
    }

    @Override
    public ObtainAdvertRsp imitateObtainAdvert(ObtainAdvertReq req) {
        return imitateAdvertService.imitateObtainAdvert(req);
    }

    @Override
    public QueryAdvertRsp queryAdvert(QueryAdvertReq req) {
        return engineServiceImpl.queryAdvert(req);
    }

    @Override
    public QueryPreloadingAdvertRsp queryPreloadingAdvert(ObtainAdvertReq req) {
        QueryPreloadingAdvertRsp rsp = new QueryPreloadingAdvertRsp();
        // 请求广告
        req.setPreloadingMaterial(Boolean.TRUE);
        ObtainAdvertRsp obtainAdvertRsp = this.obtainAdvert(req);
        long advertId = obtainAdvertRsp.getAdvertId();
        if (advertId == 0) {
            rsp.setListMaterialUrl(Collections.emptyList());
            return rsp;
        }
        rsp.setAdvertId(advertId);
        rsp.setListMaterialUrl(engineServiceImpl.listMaterialUrl(advertId));
        return rsp;
    }

    @Override
    public void insertDspAdvertOrder(ObtainAdvertReq req, AdxAdvertPriceDto adxAdvertPrice) {
        engineServiceImpl.insertDspAdvertOrder(req, adxAdvertPrice);
    }

    @Override
    public void dspLaunchLog(ObtainAdvertReq req, ObtainAdvertRsp rsp, AdxAdvertPriceDto adxAdvertPrice) {
        engineServiceImpl.dspLaunchLog(req, rsp, adxAdvertPrice);
    }

    /**
     * 互动广告补打发券日志
     *
     * @param req
     */
    @Override
    public void reLog(SpmlogReq req) {
        engineServiceImpl.reLog(req);
    }

    @Override
    public ObtainAdvertRsp lianTongIntegralConsumeObtainAdvert(ObtainAdvertReq req) throws ReadableMessageException {
        return lianTongIntegralConsumeService.lianTongIntegralConsumeObtainAdvert(req);
    }

}
