package cn.com.duiba.tuia.repository;

import com.google.common.collect.Maps;
import com.netflix.hystrix.contrib.javanica.annotation.DefaultProperties;
import com.netflix.hystrix.contrib.javanica.annotation.HystrixCommand;
import com.netflix.hystrix.contrib.javanica.annotation.HystrixProperty;
import com.netflix.hystrix.contrib.javanica.conf.HystrixPropertiesManager;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.CellUtil;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.util.Bytes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.data.hadoop.hbase.HbaseTemplate;
import org.springframework.stereotype.Repository;

/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2018年02月08日 16:47
 * @descript:
 * @version: 1.0
 */
@DefaultProperties(groupKey = "Engine", threadPoolKey = "QueryDeviceIdDmpRepository",
    threadPoolProperties = {
        @HystrixProperty(name = HystrixPropertiesManager.CORE_SIZE, value = "20"),
        @HystrixProperty(name = HystrixPropertiesManager.MAX_QUEUE_SIZE, value = "20")
    },
        commandProperties = {@HystrixProperty(name = HystrixPropertiesManager.EXECUTION_ISOLATION_THREAD_TIMEOUT_IN_MILLISECONDS, value = "200")})
@Repository
public class QueryDeviceIdDmpRepository implements ApplicationContextAware {

    private static final Logger logger = LoggerFactory.getLogger(QueryDeviceIdDmpRepository.class);

    private final String TABLE_NAME = "tuia_dmp_tag";

    private final byte[] FAMILY = Bytes.toBytes("cf");

    //dmp 人群包 命中
    public static final String HIT = "1";
    //dmp 人群包 没有 命中
    public static final String NOT_HIT = "2";

    //dmp(行业人群包)标签值为1的key("命中")
    public static final String DMP_VALUE_HIT_KEY = "tags+1";
    //dmp(行业人群包)标签值为-1的key（不命中）
    public static final String DMP_VALUE_NOT_HIT_KEY = "tags-1";


    private ApplicationContext applicationContext;


    /**
     * 查询缓存的用户社会属性
     *
     * @param deviceId
     * @param tagTimestamp 存放标签更新时间戳
     * @return
     */
    @HystrixCommand(commandKey = "DeviceIdDmp", fallbackMethod = "fallback")
    public Map<String, String> getDeviceIdDmpData(String deviceId, StringBuilder sb, List<Long> tagTimestamp) {
        Map<String, String> deviceIdScoreData = findDeviceIdDmpData(deviceId, sb, tagTimestamp);
        return deviceIdScoreData == null ? Collections.emptyMap() : deviceIdScoreData;
    }

    /**
     * 查询hbase用户社会属性
     *
     * @param key deviceKey
     * @param sb
     * @param tagTimestamp 存放标签更新时间戳
     * @return
     */

    public Map<String, String> findDeviceIdDmpData(String key, StringBuilder sb, List<Long> tagTimestamp) {

        HbaseTemplate hbaseTemplate = (HbaseTemplate) applicationContext.getBean("hbaseTemplate");
        if (hbaseTemplate == null) {
            throw new RuntimeException("hbaseTemplate 没有初始化");
        }

        return hbaseTemplate.execute(TABLE_NAME, table -> {

            Get get = new Get(Bytes.toBytes(key));
            get.addFamily(FAMILY);
            Result result = table.get(get);

            if (result.isEmpty()) {
                return  Collections.emptyMap();
            }

            Cell[] cells = result.rawCells();
            Map<String, String> allTags = Maps.newHashMapWithExpectedSize(cells.length);

            for (Cell cell : cells) {
                byte[] cellBytes = CellUtil.cloneQualifier(cell);
                String columnName = Bytes.toString(cellBytes);
                String columnValue = Bytes.toString(CellUtil.cloneValue(cell));

                tagTimestamp.add(cell.getTimestamp());
                //获取所有设备下的标签
                allTags.put(columnName, columnValue);
            }

            return allTags;
        });
    }

    /**
     * 构建请求参数
     *
     * @param key
     * @param sb
     * @return
     */
    private Get buildRequest(String key, StringBuilder sb) {

        String logString = this.getRawKey(key);
        sb.append(logString);
        Get get = new Get(Bytes.toBytes(logString));
        get.addFamily(FAMILY);
        return get;
    }

    /**
     * 获取rawKey
     *
     * @param deviceId
     * @return
     */
    private String getRawKey(String deviceId) {
        int num = Math.abs(deviceId.hashCode() % 100);
        String prexfix = num < 10 ? "0" + num : String.valueOf(num);
        StringBuilder stringBuilder = new StringBuilder(prexfix);
        return stringBuilder.append("-").append(deviceId).toString();
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    public Map<String, String> fallback(String deviceId, StringBuilder sb, List<Long> tagTimestamp, Throwable e) {

        logger.info("QueryDeviceIdDmpRepository Error,设备ID:{},tableName:{},rowKey:{},family:{},column:{}",
                deviceId, TABLE_NAME, sb, "cf", "部分列", e);
        return Collections.emptyMap();
    }
}
