package cn.com.duiba.tuia.service;

import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.nezha.engine.api.enums.TagTopNTypeEnum;
import cn.com.duiba.nezha.engine.api.support.RecommendEngineException;
import cn.com.duiba.tuia.cache.RedisUtil;
import cn.com.duiba.tuia.dao.advert_tag.AdvertRcmdTagPreferenceTopDao;
import cn.com.duiba.tuia.domain.dataobject.AdvertRcmdTagPreferenceTopEntity;

/**
 * Created by pc on 2016/10/18.
 */

@Service
public class AdvertRcmdTagPreferenceTopBo {

    private static final Logger   logger                     = LoggerFactory.getLogger(AdvertRcmdTagPreferenceTopBo.class);
    private static final Long     EXPIRE_TIME                = 20 * 60L;
    private static final String   AD_RCMD_TAG_TOP_KEY_PREFIX = "nz_ad_rcmd_tag_";

    @Autowired
    AdvertRcmdTagPreferenceTopDao advertRcmdTagPreferenceTopDao;

    @Autowired
    private RedisUtil             redisUtil;

    /**
     * 根据应用ID，读取偏好标签topN
     *
     * @param appId 应用ID
     * @param statType 数据统计类型
     * @param topN 约束,取Top
     * @return
     */
    public List<AdvertRcmdTagPreferenceTopEntity> getTagByAppIdTop(Long appId, Long statType, long topN, int topNType) {

        List<AdvertRcmdTagPreferenceTopEntity> advertRcmdTagPreferenceTopEntityList = null;
        try {
            String dimAppId = appId + "";
            String dimActivityTopicId = "all";
            advertRcmdTagPreferenceTopEntityList = getTagTop(dimAppId, dimActivityTopicId, statType, topN, topNType);
        } catch (Exception e) {
            logger.warn("AdvertRCMDTagPreferenceTopBo.getTagByAppIdTop happen error", e);
            throw new RecommendEngineException("AdvertRCMDTagPreferenceTopBo.getTagByAppIdTop happen error", e);
        }
        return advertRcmdTagPreferenceTopEntityList;

    }

    /**
     * 根据活动主题ID，读取偏好标签topN
     *
     * @param activityTopicId
     * @param statType
     * @param topN
     * @return
     */
    public List<AdvertRcmdTagPreferenceTopEntity> getTagByActivityTopicIdTop(Long activityTopicId, Long statType,
                                                                             long topN, int topNType) {

        List<AdvertRcmdTagPreferenceTopEntity> advertRcmdTagPreferenceTopEntityList = null;
        try {
            String dimAppId = "all";
            String dimActivityTopicId = activityTopicId + "";
            advertRcmdTagPreferenceTopEntityList = getTagTop(dimAppId, dimActivityTopicId, statType, topN, topNType);
        } catch (Exception e) {
            logger.warn("AdvertRCMDTagPreferenceTopBo.getTagByActivityTopicIdTop happen error", e);
            throw new RecommendEngineException("AdvertRCMDTagPreferenceTopBo.getTagByActivityTopicIdTop happen error",
                                               e);
        }
        return advertRcmdTagPreferenceTopEntityList;

    }

    /**
     * 根据应用ID和活动主题ID，读取偏好标签topN
     *
     * @param appId
     * @param activityTopicId
     * @param statType
     * @param topN
     * @return
     */
    public List<AdvertRcmdTagPreferenceTopEntity> getTagByAppIdAndActivityTopicIdTop(Long appId, Long activityTopicId,
                                                                                     Long statType, long topN,
                                                                                     int topNType) {

        List<AdvertRcmdTagPreferenceTopEntity> advertRcmdTagPreferenceTopEntityList = null;
        try {
            String dimAppId = appId + "";
            String dimActivityTopicId = activityTopicId + "";
            advertRcmdTagPreferenceTopEntityList = getTagTop(dimAppId, dimActivityTopicId, statType, topN, topNType);
        } catch (Exception e) {
            logger.warn("AdvertRCMDTagPreferenceTopBo.getTagByAppIdAndActivityTopicIdTop happen error", e);
            throw new RecommendEngineException("AdvertRCMDTagPreferenceTopBo.getTagByAppIdAndActivityTopicIdTop happen error",
                                               e);
        }
        return advertRcmdTagPreferenceTopEntityList;

    }

    /**
     * 读取全局偏好标签topN
     *
     * @param statType
     * @param topN
     * @return
     */
    public List<AdvertRcmdTagPreferenceTopEntity> getTagByGlobalTop(Long statType, long topN, int topNType) {

        List<AdvertRcmdTagPreferenceTopEntity> advertRcmdTagPreferenceTopEntityList = null;
        try {
            String dimAppId = "all";
            String dimActivityTopicId = "all";
            advertRcmdTagPreferenceTopEntityList = getTagTop(dimAppId, dimActivityTopicId, statType, topN, topNType);
        } catch (Exception e) {
            logger.error("AdvertRCMDTagPreferenceTopBo.getTagByGlobalTop happen error", e);
            throw new RecommendEngineException("AdvertRCMDTagPreferenceTopBo.getTagByGlobalTop happen error", e);
        }
        return advertRcmdTagPreferenceTopEntityList;
    }

    /**
     * 根据统计维度应用ID和统计维度活动主题ID，读取偏好标签topN
     *
     * @param dimAppId
     * @param dimActivityTopicId
     * @return
     */
    private List<AdvertRcmdTagPreferenceTopEntity> getTagTop(String dimAppId, String dimActivityTopicId, Long statType,
                                                             long topN, int topNType) {

        List<AdvertRcmdTagPreferenceTopEntity> advertRcmdTagPreferenceTopEntityList = null;

        try {
            // 参数检验
            if (StringUtils.isBlank(dimAppId) || StringUtils.isBlank(dimActivityTopicId)) {
                logger.warn("getTagTop param is null", "params invalid");
                return advertRcmdTagPreferenceTopEntityList;
            }

            // 获取缓存Key
            String key = buildTagTopKey(dimAppId, dimActivityTopicId, statType, topNType);

            // 从缓存获取推荐
            logger.debug("getTagTop cache exits,key = {} read redis", key);
            advertRcmdTagPreferenceTopEntityList = redisUtil.getList(key, AdvertRcmdTagPreferenceTopEntity.class);

            // 如果缓存没有数据，则从数据库加载
            if (advertRcmdTagPreferenceTopEntityList == null) {
                // 从数据库获取推荐
                logger.debug("getTagTop cache not exits,key = {} read mysql", key);

                if (topNType == TagTopNTypeEnum.CTR_TYPE.getType()) {
                    advertRcmdTagPreferenceTopEntityList = advertRcmdTagPreferenceTopDao.findByDimAppAndActivityTopicAndStatType(dimAppId,
                                                                                                                                 dimActivityTopicId,
                                                                                                                                 statType,
                                                                                                                                 topN);
                }

                if (topNType == TagTopNTypeEnum.FINAL_TYPE.getType()) {
                    advertRcmdTagPreferenceTopEntityList = advertRcmdTagPreferenceTopDao.findByDimAppAndActivityTopicAndStatTypeWithLimitedByFianlRank(dimAppId,
                                                                                                                                                       dimActivityTopicId,
                                                                                                                                                       statType,
                                                                                                                                                       topN);
                }

                // 保存缓存
                redisUtil.setList(key, advertRcmdTagPreferenceTopEntityList, EXPIRE_TIME);
            }

        } catch (Exception e) {
            logger.error("AdvertRCMDTagPreferenceTopBo.getTagTop happen error", e);
            throw new RecommendEngineException("AdvertRCMDTagPreferenceTopBo.getTagTop happen error", e);
        }

        // 返回结果
        return advertRcmdTagPreferenceTopEntityList;
    }

    private String buildTagTopKey(String dimAppId, String dimActivityTopicId, Long statType, int topNType) {
        String key;
        try {
            key = AD_RCMD_TAG_TOP_KEY_PREFIX + dimAppId + "_" + dimActivityTopicId + "_" + statType + "_" + topNType;
        } catch (Exception e) {
            logger.error("buildTagTopKey happend error", e);
            throw new RecommendEngineException("buildTagTopKey happend error", e);
        }
        return key;
    }

}
