package cn.com.duiba.tuia.service.buildparam.impl;

import cn.com.duiba.tuia.cache.AdvertTargetAppCacheService;
import cn.com.duiba.tuia.dao.advert.SimilarAdvertDAO;
import cn.com.duiba.tuia.domain.dataobject.SimilarAdvertDO;
import cn.com.duiba.tuia.domain.vo.AdvertTargetAppVO;
import cn.com.duiba.tuia.pangea.center.api.localservice.apollopangu.ApolloPanGuService;
import cn.com.duiba.tuia.service.buildparam.NezhaAdvertExploreService;
import cn.com.duibaboot.ext.autoconfigure.cat.annotation.CatTransaction;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author lijicong
 * @since 2020-11-20
 */
@Service
@Slf4j
public class NezhaAdvertExploreServiceImpl implements NezhaAdvertExploreService {

    @Autowired
    private AdvertTargetAppCacheService advertTargetAppCacheService;

    @Autowired
    private SimilarAdvertDAO similarAdvertDAO;

    public static final Integer DEFAULT = 1;

    private static final String APOLLO_EXPLORE_KEY = "tuia-engine.explore.whitelist";

    private LoadingCache<String, Map<String, List<String>>> cache = Caffeine.newBuilder()
            .expireAfterWrite(5, TimeUnit.MINUTES)
            .build(this::loadConfig);

    @Override
    public Map<String, List<String>> getWhitelist() {
        return cache.get(APOLLO_EXPLORE_KEY);
    }

    private Map<String, List<String>> loadConfig(String key) {

        try {
            //数据库查询所有的 配置
            List<SimilarAdvertDO> similarAdvertDOS = similarAdvertDAO.queryAllData();
            return parseData(similarAdvertDOS);
        } catch (Exception e) {
            log.warn("相似广告数据查询失败", e);
        }

//        try {
//            String configStr = apolloPanGuService.getIdMapStrByKeyStr(key);
//            return parseConfig(configStr);
//        } catch (Exception e) {
//            log.warn("获取盘古配置失败 configKey={}, error", key, e);
//        }
        return Collections.emptyMap();
    }

    private Map<String, List<String>> parseData(List<SimilarAdvertDO> similarAdvertDOS) {

        if (CollectionUtils.isEmpty(similarAdvertDOS)) {
            return Collections.emptyMap();
        }

        Map<String, List<String>> rtnMap = new HashMap<>();

        for (SimilarAdvertDO similarAdvertDO : similarAdvertDOS) {
            String advertId = String.valueOf(similarAdvertDO.getAdvertId());

            List<String> advertOrientIdList = rtnMap.get(advertId);

            if (null == advertOrientIdList) {
                advertOrientIdList = new LinkedList<>();
                rtnMap.put(advertId, advertOrientIdList);
            }

            advertOrientIdList.add(buildAdvertOrientId(similarAdvertDO));
        }

        return rtnMap;

    }

    private String buildAdvertOrientId(SimilarAdvertDO similarAdvertDO) {

        return new StringBuilder().append(similarAdvertDO.getSimilarAdvertId()).
                append("|").
                append(DEFAULT.equals(similarAdvertDO.getIsDefault()) ? 0 : similarAdvertDO.getSimilarOrientId()).
                toString();
    }

    /**
     * 配置格式
     * <pre>
     *     1-1:1-2,1-3,1-4\n2-1:2-2,2-3,2-4
     * </pre>
     * 转为Map，key: advertId-packetId
     *
     * @param configStr
     * @return
     */
    private static Map<String, List<String>> parseConfig(String configStr) {
        if (StringUtils.isBlank(configStr)) {
            return Collections.emptyMap();
        }
        Map<String, List<String>> map = new HashMap<>();
        try {
            String[] raws = configStr.split("\\n");
            for (String raw : raws) {
                String[] config = raw.split(":");
                String similarAdvert = config[1];
                List<String> advertOrients = Arrays.stream(similarAdvert.split(","))
                        .map(item -> item.replace("-", "|")) // 方便后续从缓存中获取targetApps
                        .collect(Collectors.toList());
                map.put(config[0], advertOrients);
            }
        } catch (Exception e) {
            log.warn("探索算法 盘古配置相似广告格式错误", e);
        }
        return map;
    }

    @Override
    @CatTransaction(type = "obtainAdvert", name = "enabledExplore")
    public boolean enabledExplore(Map<String, List<String>> config, Long advertId, Long appId) {
        if (MapUtils.isEmpty(config)) {
            return false;
        }
        List<String> advertPackageIds = config.get(String.valueOf(advertId));
        if (CollectionUtils.isEmpty(advertPackageIds)) {
            return false;
        }
        Collection<Long> appIds = null;
        try {
            appIds = getAdvertMediaAppIds(advertPackageIds);
        } catch (Exception e) {
            log.warn("探索算法 获取定向媒体失败", e);
        }
        return CollectionUtils.isNotEmpty(appIds) && appIds.contains(appId);
    }

    private Collection<Long> getAdvertMediaAppIds(List<String> advertPackageIds) {
        Map<String, AdvertTargetAppVO> cachedMap = advertTargetAppCacheService.getAllCachedAdvertTargetAppVO(advertPackageIds);
        return cachedMap.values().stream()
                .filter(item -> CollectionUtils.isNotEmpty(item.getTargetApps()))
                .map(item -> item.getTargetApps())
                .flatMap(Collection::stream)
                .collect(Collectors.toSet());
    }

    public static void main(String[] args) {
        String value = "5601-6685:5534-6487,5603-6688\n5607-6695:5534-6487,5603-6688\n";
        System.out.println(parseConfig(value));
    }
}
