package cn.com.duiba.tuia.service.buildparam.impl;

import cn.com.duiba.tuia.dao.media.SlotChooseAdvertDAO;
import cn.com.duiba.tuia.domain.dataobject.SlotChooseAdvertDto;
import cn.com.duiba.tuia.domain.flow.SlotChooseAdverttVO;
import cn.com.duiba.tuia.domain.enums.SlotChooseAdvertTypeEnum;
import cn.com.duiba.tuia.service.buildparam.SlotChooseAdvertService;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

@Slf4j
@Service
public class SlotChooseAdvertServiceImpl implements SlotChooseAdvertService {

    @Autowired
    private ExecutorService executorService;

    @Autowired
    private SlotChooseAdvertDAO sLotChooseAdvertDAO;

    /**
     * 不生效
     */
    private SlotChooseAdverttVO NO_EFFECT = SlotChooseAdverttVO.builder().effect(false).build();
    private SlotChooseAdverttVO ALL_EFFECT = SlotChooseAdverttVO.builder().effect(true).build();


    private final LoadingCache<Long, SlotChooseAdverttVO> SLOT_CHOOSE_ADVERT_LIST_CACHE = CacheBuilder.newBuilder().initialCapacity(1).maximumSize(5000)
            .refreshAfterWrite(5, TimeUnit.MINUTES).expireAfterWrite(1, TimeUnit.HOURS).build(new CacheLoader<Long, SlotChooseAdverttVO>() {
                @Override
                public SlotChooseAdverttVO load(Long slotId) {
                    try {

                        List<SlotChooseAdvertDto> slotChooseAdvertDtoList = sLotChooseAdvertDAO.selectBySlotId(slotId);
                        if (CollectionUtils.isEmpty(slotChooseAdvertDtoList)) {
                            return NO_EFFECT;
                        }

                        Set<Long> advertIds = new HashSet<>();
                        Set<Long> accountIds = new HashSet<>();
                        Set<String> shieldTagsSet = new HashSet<>();

                        for (SlotChooseAdvertDto slotChooseAdvertDto : slotChooseAdvertDtoList) {

                            Integer resourceType = slotChooseAdvertDto.getResourceType();
                            String resourceId = slotChooseAdvertDto.getResourceId();
                            if (SlotChooseAdvertTypeEnum.ADVERT_ID.getCode().equals(resourceType)) {
                                advertIds.add(Long.valueOf(resourceId));
                            }

                            if (SlotChooseAdvertTypeEnum.ACCOUNT_ID.getCode().equals(resourceType)) {
                                accountIds.add(Long.valueOf(resourceId));
                            }

                            if (SlotChooseAdvertTypeEnum.INDUSTRY_TAG.getCode().equals(resourceType)
                                    || SlotChooseAdvertTypeEnum.RESOURCE_TAG.getCode().equals(resourceType)) {
                                shieldTagsSet.add(resourceId);
                            }
                        }

                        return SlotChooseAdverttVO.builder().effect(true).
                                shieldTags(shieldTagsSet).
                                advertIds(advertIds).
                                accountIds(accountIds).
                                build();
                    }catch (Exception e){
                        log.error("SLOT_CHOOSE_ADVERT_LIST_CACHE key:{} exception",slotId,e);
                        return ALL_EFFECT;
                    }
                }

                @Override
                public ListenableFuture<SlotChooseAdverttVO> reload(final Long slotId, SlotChooseAdverttVO old) {
                    ListenableFutureTask<SlotChooseAdverttVO> task = ListenableFutureTask.create(()->load(slotId));
                    executorService.submit(task);
                    return task;
                }
            });


    /**
     * 刷新缓存
     */
    @Override
    public void refreshData(Long slotId) {
        this.SLOT_CHOOSE_ADVERT_LIST_CACHE.refresh(slotId);
    }

    /**
     * 根据广告位 id 查询 定制广告配置信息
     *
     * @param slotId
     * @return
     */
    @Override
    public SlotChooseAdverttVO queryChooseAdvertList(Long slotId) {
        try {
            return SLOT_CHOOSE_ADVERT_LIST_CACHE.getUnchecked(slotId);
        } catch (Exception e) {
            log.error("缓存查询异常，slotId="+slotId,e);
            return NO_EFFECT;
        }
    }
}
