package cn.com.duiba.tuia.service.condition.impl;

import cn.com.duiba.tuia.cache.MakeTagCacheService;
import cn.com.duiba.tuia.domain.flow.MediaList;
import cn.com.duiba.tuia.domain.model.*;
import cn.com.duiba.tuia.enums.AdvertConditionFilterOrderEnum;
import cn.com.duiba.tuia.service.condition.AdvertConditionFilter;
import cn.com.tuia.advert.enums.AdvertFilterTypeEnum;
import cn.com.tuia.advert.model.ObtainAdvertReq;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.List;
import java.util.Set;

/**
 * 屏蔽标签
 */
@Component
@Slf4j
public class BannedTagConditionFilter implements AdvertConditionFilter {

    @Autowired
    private MakeTagCacheService makeTagCacheService;

    /**
     * 过滤接口
     *
     *
     * @param advertConditionContext@return
     */
    @Override
    public boolean match(AdvertConditionContext advertConditionContext) {

        AdvOrientationItem item = advertConditionContext.getAdvOrientationItem();
        AdvQueryParam advQueryParam = advertConditionContext.getAdvQueryParam();
        ObtainAdvertReq req = advertConditionContext.getReq();
        Set<AdvertFilterType> filterTypeSets = advertConditionContext.getFilterTypeSets();

        boolean result = checkBannedTags(item, advQueryParam, filterTypeSets);

        return !result;
    }

    /**
     * 过滤类型
     *
     * @return
     */
    @Override
    public AdvertFilterTypeEnum getFilterType() {
        return AdvertFilterTypeEnum.BANNED_TAG;
    }

    /**
     * 执行顺序
     *
     * @return
     */
    @Override
    public int getOrder() {
        return AdvertConditionFilterOrderEnum.BANNED_TAG_ORDER.getOrder();
    }

    /**
     * checkBannedTags:(是否命中屏蔽标签). <br/>
     */
    private Boolean checkBannedTags(AdvOrientationItem item,  AdvQueryParam advQueryParam, Set<AdvertFilterType> filterTypeSets) {
        // 广告白名单不过滤
        MediaList mediaWhiteList = advQueryParam.getMediaWhiteList();
        if(mediaWhiteList != null && mediaWhiteList.isAdvertInList(item.getAdvertId(),item.getAccountId())){
            return false;
        }

        AdvBannedTag advBannedTag = item.getAdvBannedTag();
        AdvBannedTag advQueryBannedTag = advQueryParam.getAdvBannedTag();

        // 按照 行业-资源-属性-落地页 顺序匹配过滤
        boolean industryTagsMatch = isAnyMatch(advBannedTag.getIndustryTags(), advQueryBannedTag.getIndustryTags());
        if (industryTagsMatch) {
            filterTypeSets.add(new AdvertFilterType(item.getAdvertId(), item.getOrientationId(), AdvertFilterTypeEnum.BANNED_INDUSTRY_TAG.getCode()));
            return true;
        }
        boolean resourceTagsMatch = isAnyMatch(advBannedTag.getResourceTags(), advQueryBannedTag.getResourceTags());
        if (resourceTagsMatch) {
            filterTypeSets.add(new AdvertFilterType(item.getAdvertId(), item.getOrientationId(), AdvertFilterTypeEnum.BANNED_RESOURCE_TAG.getCode()));
            return true;
        }
        boolean attributeTagsMatch = isAnyMatch(advBannedTag.getAttributeTags(), advQueryBannedTag.getAttributeTags());
        if (attributeTagsMatch) {
            filterTypeSets.add(new AdvertFilterType(item.getAdvertId(), item.getOrientationId(), AdvertFilterTypeEnum.BANNED_ATTRIBUTE_TAG.getCode()));
            return true;
        }

        //如果是百奇或积木域名，添加独有的落地页标签
        Set<String> promoteTagSet = advBannedTag.getPromoteUrlTags();
        if (item.getSourceId() != null && item.getSourceType() != null) {
            List<String> makeTags = makeTagCacheService.getMakeTagByKey(item.getSourceId(), item.getSourceType());
            if (CollectionUtils.isNotEmpty(makeTags)) {
                promoteTagSet.addAll(makeTags);
            }
        }
        boolean promoteUrlTagsMatch = isAnyMatch(promoteTagSet, advQueryBannedTag.getPromoteUrlTags());
        if (promoteUrlTagsMatch) {
            filterTypeSets.add(new AdvertFilterType(item.getAdvertId(), item.getOrientationId(), AdvertFilterTypeEnum.BANNED_PROMOTEURL_TAG.getCode()));
            return true;
        }
        return false;
    }

    private static boolean isAnyMatch(Collection<String> collection1, Collection<String> collection2) {
        if (CollectionUtils.isEmpty(collection1) || CollectionUtils.isEmpty(collection2)) {
            return false;
        }
        return collection1.stream().anyMatch(collection2::contains);
    }
}
