package cn.com.duiba.tuia.service.condition.impl;

import cn.com.duiba.tuia.cache.AdvertMapCacheManager;
import cn.com.duiba.tuia.domain.dataobject.AdvertTagDO;
import cn.com.duiba.tuia.domain.model.*;
import cn.com.duiba.tuia.domain.vo.AdvertVO;
import cn.com.duiba.tuia.enums.AdvertConditionFilterOrderEnum;
import cn.com.duiba.tuia.service.condition.AdvertConditionFilter;
import cn.com.duiba.tuia.tool.StringTool;
import cn.com.tuia.advert.enums.AdvertFilterTypeEnum;
import cn.com.tuia.advert.model.ObtainAdvertReq;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;
import java.util.Set;

@Component
@Slf4j
public class SelectAdvertConditionFilter implements AdvertConditionFilter {

    @Autowired
    private AdvertMapCacheManager advertMapCacheManager;

    /**
     * 过滤接口
     *
     *
     * @param advertConditionContext@return
     */
    @Override
    public boolean match(AdvertConditionContext advertConditionContext) {
        AdvOrientationItem item = advertConditionContext.getAdvOrientationItem();
        AdvQueryParam advQueryParam = advertConditionContext.getAdvQueryParam();
        ObtainAdvertReq req = advertConditionContext.getReq();
        Set<AdvertFilterType> filterTypeSets = advertConditionContext.getFilterTypeSets();

        boolean result = appSelectAdvertCheck(advQueryParam, item.getLeftMaterial(),item.getAdvertId(), advertMapCacheManager.getAdvertCache(item.getAdvertId()));
        if(!result){
            filterTypeSets.add(new AdvertFilterType(item.getAdvertId(), item.getOrientationId(), getFilterType().getCode()));
        }

        return result;
    }

    /**
     * 过滤类型
     *
     * @return
     */
    @Override
    public AdvertFilterTypeEnum getFilterType() {
        return AdvertFilterTypeEnum.SELECT_ADVERT;
    }

    /**
     * 执行顺序
     *
     * @return
     */
    @Override
    public int getOrder() {
        return AdvertConditionFilterOrderEnum.SELECT_ADVERT_ORDER.getOrder();
    }

    /**
     * @return false-媒体选择的行业或者广告和我们的广告池没有交集，过滤、true-媒体选择的行业或者广告和我们的广告池有交集，可以出券
     */
    @SuppressWarnings("squid:S3776")
    public boolean appSelectAdvertCheck(AdvQueryParam advQueryParam,Set<Long> leftMaterial,Long advertId, AdvertVO advertVO) {
        boolean isElectApp = advQueryParam.isElectApp();

        //只有开启了互选功能的媒体才需要做下面的判断
        if (isElectApp) {
            if (advertVO == null) {
                return false;
            }
            Set<Long> appSelectAdvertIds = advQueryParam.getAppSelectAdvertIds();
            Set<String> appSelectTags = advQueryParam.getAppSelectTags();

            //如果媒体既没有选择行业也没有选择广告，则该媒体不能出券
            if (CollectionUtils.isEmpty(appSelectAdvertIds) && CollectionUtils.isEmpty(appSelectTags)) {
                return false;
            }
            //媒体既选择了广告又选择了标签，取合集，两个一起判断
            if (CollectionUtils.isNotEmpty(appSelectAdvertIds) && CollectionUtils.isNotEmpty(appSelectTags)) {
                Boolean result = appSelectAdvertIds.contains(advertId) || checkSelectIndustryTag(appSelectTags, advertVO);
                if(!result){
                    return result;
                }
            }
            //媒体只选择了行业标签，只用判断行业标签
            else if (CollectionUtils.isNotEmpty(appSelectTags)) {
                Boolean result = checkSelectIndustryTag(appSelectTags, advertVO);
                if(!result){
                    return result;
                }
            }
            //媒体只选择了广告，只用判断广告
            else if (CollectionUtils.isNotEmpty(appSelectAdvertIds)) {
                Boolean result = appSelectAdvertIds.contains(advertId);
                if(!result){
                    return result;
                }
            }

            //互选广告需优化-增加素材维度选择新增逻辑  不用考虑熔断情况
            Map<Long, Set<Long>> selectedAdvertAndMaterialIds = advQueryParam.getSelectedAdvertAndMaterialIds();
            if(null == selectedAdvertAndMaterialIds){
                //媒体后台 传过来的值是空的
                return false;
            }
            Set<Long> materialIds = selectedAdvertAndMaterialIds.get(advertId);
            if(null == materialIds||materialIds.isEmpty()||null == leftMaterial||leftMaterial.isEmpty()){
                //没有选中该广告，或者广告下可用素材是空的
                return false;
            }
            //素材取交集
            leftMaterial.retainAll(materialIds);
            return !leftMaterial.isEmpty();
        }
        return true;
    }

    private boolean checkSelectIndustryTag(Set<String> selectIndustryTags, AdvertVO advertVO) {
        AdvertTagDO advertTagDO = advertVO.getAdvertTagDO();
        if (advertTagDO == null) {
            return false;
        }
        //行业标签
        List<String> advertIndustryTags = StringTool.getStringListByStr(advertTagDO.getMatchTagNums());

        //不为空就有交集(true)，有交集就可以出券
        return CollectionUtils.isNotEmpty(CollectionUtils.intersection(selectIndustryTags, advertIndustryTags));
    }
}
