package cn.com.duiba.tuia.service.condition.impl;

import cn.com.duiba.tuia.cache.AdvertMapCacheManager;
import cn.com.duiba.tuia.domain.model.*;
import cn.com.duiba.tuia.domain.vo.AdvertVO;
import cn.com.duiba.tuia.enums.AdvertConditionFilterOrderEnum;
import cn.com.duiba.tuia.service.condition.AdvertConditionFilter;
import cn.com.tuia.advert.enums.AdvertFilterTypeEnum;
import cn.com.tuia.advert.model.ObtainAdvertReq;
import cn.com.tuia.advert.model.Period;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.Set;

@Component
@Slf4j
public class SpecialAppConditionFilter implements AdvertConditionFilter {

    @Autowired
    private AdvertMapCacheManager advertMapCacheManager;

    private static final Integer CHARGE_TYPE_CPC = 1;

    /**
     * 过滤接口
     *
     *
     * @param advertConditionContext@return
     */
    @Override
    public boolean match(AdvertConditionContext advertConditionContext) {
        AdvOrientationItem item = advertConditionContext.getAdvOrientationItem();
        AdvQueryParam advQueryParam = advertConditionContext.getAdvQueryParam();
        ObtainAdvertReq req = advertConditionContext.getReq();
        Set<AdvertFilterType> filterTypeSets = advertConditionContext.getFilterTypeSets();

        boolean result = checkSpecialApp(item,advertMapCacheManager.getAdvertCache(item.getAdvertId()), advQueryParam.getAppId());
        if(!result){
            filterTypeSets.add(new AdvertFilterType(item.getAdvertId(), item.getOrientationId(), getFilterType().getCode()));
        }
        return result;
    }

    /**
     * 过滤类型
     *
     * @return
     */
    @Override
    public AdvertFilterTypeEnum getFilterType() {
        return AdvertFilterTypeEnum.SPECIAL_APP;
    }

    /**
     * 执行顺序
     *
     * @return
     */
    @Override
    public int getOrder() {
        return AdvertConditionFilterOrderEnum.SPECIAL_APP_ORDER.getOrder();
    }

    private boolean checkSpecialApp(AdvOrientationItem item, AdvertVO advertVO, Long appId) {
        if (advertVO == null) {
            return false;
        }
        //为空则没有设置过可投的特殊媒体，不用过滤
        if (MapUtils.isEmpty(advertVO.getSpecialApps())) {
            return true;
        }

        List<Period> periods = advertVO.getSpecialApps().get(appId);

        //periods为空就说明广告设置了可投的特殊媒体，但是该请求媒体不在特殊媒体内，该媒体不能出该广告
        //空包appId写死了-1，拿出时就为空，不能出
        if (CollectionUtils.isEmpty(periods)) {
            return false;
        }
        //现在开始校验时段

        if(!periodsCheck(periods)){
            return false;
        }

        //TODO 此处逻辑有问题，cpa(ocpc)怎么过滤
        if(!CHARGE_TYPE_CPC.equals(item.getChargeType())){
            return true;
        }

        SpecialAppBo specialAppBo= Optional.ofNullable(advertVO.getSpecialAppsOtherData()).map(data->data.get(appId)).orElse(null);
        if(null == specialAppBo){
            return false;
        }

        //特殊广告 最低出价过滤(互动广告 需要 对 cpc 计费的 广告主 和 媒体进行 过滤)
        if(!lowestPriceCheck(specialAppBo.getLowestPrice(),item.getCpcPrice())){
            return false;
        }

        return true;
    }

    private boolean lowestPriceCheck(Long lowestPrice,Long cpcPrice){
        if(null == lowestPrice||null == cpcPrice){
            return false;
        }

        /**
         * 最低出价 要求 大于等于0，=-1时为没有最低出价(数据合理性校验)
         */
        if(lowestPrice == -1){
            return true;
        }

        if(lowestPrice > cpcPrice){
            return false;
        }

        return true;
    }

    private boolean periodsCheck(List<Period> periods) {
        DateTime dateTime = new DateTime(new Date());

        int curHourInt = Integer.parseInt(dateTime.toString("HHmm"));

        for (Period period : periods) {
            //时段为空则当不限处理，可以出
            if (StringUtils.isEmpty(period.getEndHour()) || StringUtils.isEmpty(period.getStartHour())) {
                return true;
            }

            String startHour = period.getStartHour().replaceAll(":", "");
            String endHour = period.getEndHour().replaceAll(":", "");

            int startHourInt = Integer.parseInt(startHour);
            int endHourInt = Integer.parseInt(endHour);

            //当前时段在配置的时间区间内则可投
            if (curHourInt >= startHourInt && curHourInt < endHourInt) {
                return true;
            }
        }

        //不在配置的时间区间内则不可投
        return false;
    }
}
