package cn.com.duiba.tuia.service.filter.impl;

import cn.com.duiba.tuia.dao.flowback.DataFlowbackPlanDao;
import cn.com.duiba.tuia.domain.dataobject.DataFlowbackPlanDTO;
import cn.com.duiba.tuia.domain.dataobject.SpecialSlotWhiteListDTO;
import cn.com.duiba.tuia.domain.vo.AdvertFilterVO;
import cn.com.duiba.tuia.service.filter.AutoFlowbackFilter;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.tuia.advert.model.ObtainAdvertReq;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.PostConstruct;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

@Service
public class AutoFlowbackFilterImpl implements AutoFlowbackFilter {

    private static final Logger logger = LoggerFactory.getLogger(AutoFlowbackFilterImpl.class);

    public static final String SPLIT = "#";

    @Autowired
    private DataFlowbackPlanDao dataFlowbackPlanDao;

    @Autowired
    private ExecutorService executorService;

    //有消息同步，不用设置过期时间来保证避免读取旧值
    private LoadingCache<String, Optional<Long>> cache = CacheBuilder.newBuilder().initialCapacity(5000).
            recordStats().refreshAfterWrite(5, TimeUnit.MINUTES)
            .build(
                    new CacheLoader<String, Optional<Long>>() {
                        @Override
                        public Optional<Long> load(String key) {
                            String[] split = key.split(SPLIT);
                            Long slotId = Long.valueOf(split[0]);
                            String dateStr = split[1];

                            DataFlowbackPlanDTO dataFlowbackPlanDTO = dataFlowbackPlanDao.queryBySlotIdAndDate(slotId, dateStr);

                            return null == dataFlowbackPlanDTO ? Optional.empty() : Optional.ofNullable(dataFlowbackPlanDTO.getAdvertId());
                        }

                        @Override
                        public ListenableFuture<Optional<Long>> reload(String key, Optional<Long> oldValue) throws Exception {
                            ListenableFutureTask<Optional<Long>> task = ListenableFutureTask.create(() -> load(key));
                            executorService.submit(task);
                            return task;
                        }
                    });


    /**
     * 返回是否是 自动回流流量
     * @param logExtExpMap
     * @return
     */
    public static Boolean isAutoFlowback(Map<String, String> logExtExpMap){
        return null != logExtExpMap && "1".equals(logExtExpMap.get("isOutside"));
    }


    //缓存初始化
    @PostConstruct
    public void init() {
        //从数据库中 获取 本日的 符合投放条件的数据
        List<DataFlowbackPlanDTO> effectPlans = dataFlowbackPlanDao.queryAllEffectPlan();
        effectPlans.forEach(effectPlan -> {
            Long slotId = effectPlan.getSlotId();
            Date date = effectPlan.getDate();
            String key = buildKey(slotId, date);
            cache.put(key, Optional.ofNullable(effectPlan.getAdvertId()));
        });
    }

    private String buildKey(Long slotId, Date date) {
        return slotId + SPLIT + DateUtils.getDayStr(date);
    }

    private String buildKey(Long slotId, String dateStr) {
        return slotId + SPLIT + dateStr;
    }

    @Override
    public void addCache(String message) {

        List<SpecialSlotWhiteListDTO> dtos = buildSpecialSlotWhiteListDTO(message);
        if(CollectionUtils.isEmpty(dtos)){
            return;
        }

        for (SpecialSlotWhiteListDTO dto : dtos) {
            try {
                cache.put(buildKey(dto.getSlotId(),dto.getDateStr()),Optional.ofNullable(dto.getAdvertId()));
            } catch (Exception e) {
                logger.info("缓存添加异常,dto="+ JSON.toJSONString(dto),e);
            }
        }

    }


    @Override
    public void rmCache(String message) {
        List<SpecialSlotWhiteListDTO> dtos = buildSpecialSlotWhiteListDTO(message);
        if(CollectionUtils.isEmpty(dtos)){
            return;
        }

        for (SpecialSlotWhiteListDTO dto : dtos) {
            try {
                cache.invalidate(buildKey(dto.getSlotId(),dto.getDateStr()));
            } catch (Exception e) {
                logger.info("缓存失效异常,dto="+ JSON.toJSONString(dto),e);
            }
        }

    }

    private List<SpecialSlotWhiteListDTO> buildSpecialSlotWhiteListDTO(String message) {
        try {
            return JSONArray.parseArray(message, SpecialSlotWhiteListDTO.class);
        } catch (Exception e) {
            logger.info("消息解析异常，msg="+message,e);
            return null;
        }
    }

    /**
     * 广告主自动回流数据过滤
     *
     * @param validAdverts
     */
    @Override
    public void doFilter(Map<Long, AdvertFilterVO> validAdverts, ObtainAdvertReq req) {

        String flag = null;
        Map<String, String> logExtMap = req.getLogExtExpMap();
        if (null != logExtMap) {
            flag = logExtMap.get("isOutside");
        }

        Long slotId = req.getSlotId();

        Long advertId = cache.getUnchecked(buildKey(slotId,new Date())).orElse(null);

        //自动回流流量
        if ("1".equals(flag) || null != advertId) {
//            Long advertId = cache.getUnchecked(buildKey(slotId,new Date())).orElse(null);
            if (null == advertId) {
                //不在 投放白名单中 则过滤
                validAdverts.clear();
            } else {
                //在投放白名单中 则 取交集
                validAdverts.keySet().retainAll(Arrays.asList(advertId));
            }
        }
    }
}
