package cn.com.duiba.tuia.service.filter.impl;

import cn.com.duiba.bigdata.online.service.api.dto.ResourceDataDto;
import cn.com.duiba.bigdata.online.service.api.form.DeviceResourceForm;
import cn.com.duiba.bigdata.online.service.api.remoteservice.RemoteDeviceResourceService;
import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duiba.tuia.domain.dataobject.LowNeedsDataFilterBaseRuleDTO;
import cn.com.duiba.tuia.domain.dataobject.LowNeedsDataFilterRuleDTO;
import cn.com.duiba.tuia.domain.enums.LowNeedsCalTypeEnum;
import cn.com.duiba.tuia.domain.enums.LowNeedsDataTypeEnum;
import cn.com.duiba.tuia.domain.model.*;
import cn.com.duiba.tuia.domain.vo.AdvertFilterVO;
import cn.com.duiba.tuia.domain.vo.AdvertPriceVO;
import cn.com.tuia.advert.enums.AdvertFilterTypeEnum;
import cn.com.duiba.tuia.enums.RandomServiceEnum;
import cn.com.duiba.tuia.service.RandomService;
import cn.com.duiba.tuia.service.filter.LowNeedsAdFilter;
import cn.com.tuia.advert.model.ObtainAdvertReq;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.cloud.context.scope.refresh.RefreshScopeRefreshedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;

@RefreshScope
@Service
public class LowNeedsAdFilterImpl implements LowNeedsAdFilter {

    private static final Logger logger = LoggerFactory.getLogger(LowNeedsAdFilterImpl.class);

    @Autowired
    private RandomService randomService;

    @Autowired
    private RemoteDeviceResourceService remoteDeviceResourceService;

    @Value("${tuia.engine.filter.lowNeedsAd.rate:0}")
    private String lowNeedsRate;

    @Value("${tuia.engine.filter.lowNeedsAd.config:[]}")
    private String lowNeedsFilterConfig;

    private List<LowNeedsDataFilterRule> lowNeedsDataFilterRuleList;

    /**
     * 启动监听
     * @param contextStartedEvent
     */
    @EventListener
    public void listenOnStart(MainContextRefreshedEvent contextStartedEvent){
        refreshLowNeedsDataFilterRule();
    }

    /**
     * 刷新配置监听
     * @param event
     */
    @EventListener
    public void listenOnRefresh(RefreshScopeRefreshedEvent event){
        refreshLowNeedsDataFilterRule();
    }

    private void refreshLowNeedsDataFilterRule() {
        List<LowNeedsDataFilterRuleDTO> objects;
        try {
            objects = JSONObject.parseArray(lowNeedsFilterConfig,LowNeedsDataFilterRuleDTO.class);
        } catch (Exception e) {
            objects = null;
            logger.error("低意向人群资源过滤器配置错误，错误配置：tuia.engine.filter.lowNeedsAd.config",e);
        }

        if(null == objects){
            return;
        }

        List<LowNeedsDataFilterRule> newRules = new LinkedList<>();
        objects.forEach(obj->{
            LowNeedsDataFilterRule lowNeedsDataFilterRule;
            try {
                lowNeedsDataFilterRule = buildLowNeedsDataFilterRule(obj);
            } catch (Exception e) {
                lowNeedsDataFilterRule = null;
            }

            if(null == lowNeedsDataFilterRule){
                logger.error("低意向人群过滤，过滤资源规则解析失败，失败规则=" + JSON.toJSONString(obj));
                return;
            }

            newRules.add(lowNeedsDataFilterRule);
            logger.info("低意向人群过滤，过滤资源规则刷新成功，生效规则=" + JSON.toJSONString(obj));
        });

        //给过滤服务添加数据筛选器，原子操作
        this.lowNeedsDataFilterRuleList = newRules;
    }

    //构建低意向过滤过滤资源规则
    private LowNeedsDataFilterRule buildLowNeedsDataFilterRule(LowNeedsDataFilterRuleDTO dto) {

        List<LogicCaler> logicCalers = new ArrayList<>();

        List<LowNeedsDataFilterBaseRuleDTO> rules = dto.getRules();
        //校验参数
        Assert.notEmpty(rules,"配置的低意向人群过滤，规则不能为空");

        for (LowNeedsDataFilterBaseRuleDTO rule : rules) {
            //
            ResourceDataGetter resourceDataGetter = LowNeedsDataTypeEnum.getResourceDataGetterByName(rule.getDataType());
            Assert.notNull(resourceDataGetter,"配置的数据获取器有问题");

            LogicDataCaler logicDataCaler = LowNeedsCalTypeEnum.getLogicDataCalerByName(rule.getCalType());
            Assert.notNull(logicDataCaler,"配置的逻辑运算器有问题");

            BigDecimal threshold = new BigDecimal(rule.getThreshold());
            Assert.notNull(threshold,"配置的阈值有问题");

            LogicCaler logicCaler = new LogicCaler(resourceDataGetter,logicDataCaler,threshold);
            logicCalers.add(logicCaler);

        }

        return new LowNeedsDataFilterRule(logicCalers);
    }


    @SuppressWarnings("squid:S3776")
    @Override
    public void doFilter(Map<Long, AdvertFilterVO> validAdverts, FilterResult filterResult, ObtainAdvertReq req, AdvertFilter advertFilter) {

        //全局切量校验
        if(null == filterResult.getRandomServiceTag()){
            logger.warn("全局切量初始值为null，影响了用户低意向广告过滤，导致过滤失效。");
            return;
        }

        //全局切量
        Boolean aBoolean = randomService.decodeRandomTag(filterResult.getRandomServiceTag(), RandomServiceEnum.LOW_NEEDS_AD_RANDOM_SERVICE);
        filterResult.setRsHitCut(aBoolean);
        if(!aBoolean){
            return;
        }

        //查询数据
        List<String> deviceResourceData = getdeviceResourceData(req.getDeviceId());

        //如果没有返回 数据 则不过滤
        if(CollectionUtils.isEmpty(deviceResourceData)){
            return;
        }

        Set<Long> filterKeySet = new HashSet<>();
        Set<Long> leftPayedKeySet = new HashSet<>();
        Set<String> filterTagSet = new HashSet<>(deviceResourceData);

        validAdverts.forEach((key,value)->{
            String tag = value.getResourceTag();
            if(StringUtils.isEmpty(tag)){
                //判断当前广告下是否有 付费券
                if(checkHasPayedPkg(value)) {
                    leftPayedKeySet.add(key);
                }
                return;
            }

            if(filterTagSet.contains(tag)){
                filterKeySet.add(key);
            }else{
                //付费券检测
                //日志打印
                if(checkHasPayedPkg(value)) {
                    leftPayedKeySet.add(key);
                }
            }

        });

        //存在要过滤的券
        if(filterKeySet.size()>0){
            //用户低意向广告用户切量
            ThreadLocalRandom threadLocalRandom = ThreadLocalRandom.current();
            Boolean isLucky = threadLocalRandom.nextInt(100) < Integer.valueOf(lowNeedsRate);
            if(isLucky) {
                filterResult.setRsFilted(true);
                //是否需要释放
                if (leftPayedKeySet.size() > 0) {
                    Set<AdvertFilterType> reason = advertFilter.getReason();
                    if (null != reason) {
                        for (Long filertKey : filterKeySet) {
                            AdvertFilterVO advertFilterVO = validAdverts.get(filertKey);
                            if (null != advertFilterVO) {
                                Set<AdvertPriceVO> orientationSets = advertFilterVO.getAdvertPriceVOSortedSet();
                                if(null != orientationSets) {
                                    orientationSets.forEach(vo -> {
                                        reason.add(new AdvertFilterType(vo.getAdvertId(), vo.getAdvertOrientationPackageId(), AdvertFilterTypeEnum.LOW_NEEDS_AD.getCode()));
                                    });
                                }
                            }
                        }
                    }

                    validAdverts.keySet().removeAll(filterKeySet);
                } else {
                    filterResult.setRsFiltFree(true);
                }
            }
        }
    }

    private List<String> getdeviceResourceData(String deviceId) {

        //没有过滤规则 则不筛选数据（兼容没有配置规则的情况）
        List<LowNeedsDataFilterRule> lowNeedsDataFilterRulesTmp = lowNeedsDataFilterRuleList;
        if(CollectionUtils.isEmpty(lowNeedsDataFilterRulesTmp)){
            return Collections.emptyList();
        }

        DeviceResourceForm deviceResourceForm = new DeviceResourceForm();
        deviceResourceForm.setDeviceId(deviceId);
        List<ResourceDataDto> resourceData;
        try {
            resourceData = remoteDeviceResourceService.getResourceData(deviceResourceForm);
        } catch (Exception e) {
            resourceData = null;
        }

        return CollectionUtils.isEmpty(resourceData) ?
                Collections.emptyList() :
                resourceData.stream()
                        .filter(data ->isInRule(data,lowNeedsDataFilterRulesTmp))
                        .map(ResourceDataDto::getResourceId)
                        .collect(Collectors.toList());
    }


    private boolean isInRule(ResourceDataDto data,List<LowNeedsDataFilterRule> lowNeedsDataFilterRulesTmp) {
        for (LowNeedsDataFilterRule lowNeedsDataFilterRule : lowNeedsDataFilterRulesTmp) {
            //规则之间是或的关系 只要命中一条规则就过滤
            if(lowNeedsDataFilterRule != null && lowNeedsDataFilterRule.inRule(data)){
                return true;
            }
        }

        //没有命中任何规则，则此条资源 不需要被过滤，所以返回false，在需要过滤的资源中去除此资源
        return false;
    }

    /**
     * 判断当前广告下是否有付费券
     * @return
     */
    private boolean checkHasPayedPkg(AdvertFilterVO vo) {

        SortedSet<AdvertPriceVO> advertPriceVOSortedSet = vo.getAdvertPriceVOSortedSet();
        if(CollectionUtils.isEmpty(advertPriceVOSortedSet)){
            return false;
        }

        return advertPriceVOSortedSet.stream().anyMatch(advertPriceVO->advertPriceVO.getFee()!=null && advertPriceVO.getFee()>0);
    }
}
