package cn.com.duiba.tuia.service.filter.impl;

import cn.com.duiba.bigdata.online.service.api.dto.DeviceFilterDto;
import cn.com.duiba.tuia.domain.model.TfUserFilter;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.CollectionUtils;

import java.util.Collections;
import java.util.List;
import java.util.Optional;

@Slf4j
public class TfUserFilterResourceImpl implements TfUserFilter {

    private static final String LAND_PAGE = "1";

    /**
     * 广告主id
     */
    private final String resourceTag;

    /**
     * 转化指标
     */
    private final List<String> targets;

    /**
     * 转化指标
     */
    private final List<String> targetPeriods;

    /**
     * 转化指标
     */
    private final Boolean isInfoLog;

    public TfUserFilterResourceImpl(String resourceTag, List<String> targets, List<String> targetPeriods, Boolean isInfoLog) {
        this.resourceTag = resourceTag;
        this.targets = targets;
        this.targetPeriods = targetPeriods;
        this.isInfoLog = isInfoLog;
    }

    @Override
    public Boolean doFilter(DeviceFilterDto deviceFilterDto) {
        //没有转化记录
        if(null == deviceFilterDto){
            return true;
        }

        // 没有选择指标
        if (CollectionUtils.isEmpty(targets)) {
            return true;
        }

        List<String> tfTargets = getMergedTfTargets(deviceFilterDto);
        if (isInfoLog) {
            log.info("MergedTfTargets {} targets={} targetPeriods={} tfTargets={}",
                    this.resourceTag, targets, targetPeriods, tfTargets);
        }

        /**
         * 若有交集，则已有转化，需要过滤，false
         */
        return !CollectionUtils.containsAny(targets, tfTargets);
    }

    /**
     * 获取合并后的数据
     *  落地页转化分为 当天和15天内
     * @return
     */
    private List<String> getMergedTfTargets(DeviceFilterDto deviceFilterDto) {
        // 1天数据
        List<String> tfTarget1 = Optional.ofNullable(deviceFilterDto)
                .map(dto -> dto.getDayResourceEffect())
                .orElse(Collections.emptyList());

        // 15天内数据
        List<String> tfTarget15 = Optional.ofNullable(deviceFilterDto)
                .map(dto -> dto.getResourceEffect())
                .map(map-> map.get(this.resourceTag))
                .orElse(Collections.emptyList());
        if (isInfoLog) {
            log.info("TfUserFilterResourceImpl {} targets={} targetPeriods={} tfTargets1={} tfTargets15={}",
                    this.resourceTag, targets, targetPeriods, tfTarget1, tfTarget15);
        }

        // 不包含落地页 或者 没有配置时间段
        if (!targets.contains(LAND_PAGE)) {
            return tfTarget15;
        }

        // 包含落地页
        List<String> tfTarget = Lists.newArrayList(tfTarget15);
        // 移除落地页数据, 落地页由指标时间段决定
        tfTarget.remove(LAND_PAGE);

        // 落地页 当日
        if (targetPeriods.contains("1-1") && tfTarget1.contains(resourceTag)) {
            tfTarget.add(LAND_PAGE);
        }
        // 落地页 15日内
        else if (targetPeriods.contains("1-15") && tfTarget15.contains(LAND_PAGE)) {
            tfTarget.add(LAND_PAGE);
        }

        return tfTarget;
    }
}
