package cn.com.duiba.tuia.service.impl;

import lombok.extern.slf4j.Slf4j;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.data.redis.core.HashOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Maps;

import cn.com.duiba.tuia.constants.ErrorCode;
import cn.com.duiba.tuia.dao.slot.SlotWhiteListDAO;
import cn.com.duiba.tuia.domain.dataobject.AdvertMaterialDto;
import cn.com.duiba.tuia.domain.dataobject.ResoureTagsDO;
import cn.com.duiba.tuia.domain.model.AppTagQueryTime;
import cn.com.duiba.tuia.domain.model.FilterResult;
import cn.com.duiba.tuia.domain.model.ShieldStrategyVO;
import cn.com.duiba.tuia.domain.model.SlotTagQueryTime;
import cn.com.duiba.tuia.message.rocketmq.AdvertLaunchMonitorProducer;
import cn.com.duiba.tuia.service.AdvertLaunchMonitorService;
import cn.com.duiba.tuia.service.ResourceTagsService;
import cn.com.duiba.tuia.ssp.center.api.dto.ActivityAdvertDto;
import cn.com.duiba.tuia.ssp.center.api.dto.advertmonitor.ActivityAdvert4MonitorDto;
import cn.com.duiba.tuia.ssp.center.api.dto.advertmonitor.ActivityDirectMode4MonitorDto;
import cn.com.duiba.tuia.ssp.center.api.dto.advertselect.AdvertSelectedDto;
import cn.com.duiba.tuia.ssp.center.api.remote.advertselect.dto.media.req.AuditedAdvertAndMaterial4AdDto;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.tuia.advert.constants.AdvertLaunchMonitorConstant;

/**
 * 广告发券监控
 *
 * @author peanut.huang
 * @date 2019/8/9
 * @since JDK 1.8
 */
@Service
@RefreshScope
@Slf4j
public class AdvertLaunchMonitorServiceImpl implements AdvertLaunchMonitorService {

    @Value("${advert.launch.monitor.app:-1}")
    private String monitorApps;
    @Autowired
    private ExecutorService                     executorService;
    @Autowired
    private AdvertLaunchMonitorProducer         advertLaunchMonitorProducer;
    @Autowired
    private SlotWhiteListDAO                    slotWhiteListDAO;
    @Autowired
    private RedisTemplate<Object, Object>       redisTemplate;
    @Autowired
    private ResourceTagsService                 resourceTagsService;

    @Override
    public void monitorHandle(FilterResult filterResult) {

        // 异步处理
        executorService.execute(() -> execMonitor(filterResult));
    }

    /**
     * 执行监控
     *
     * @param filterResult
     */
    private void execMonitor(FilterResult filterResult){
        if(!ErrorCode.E0000000.getErrorCode().equals(filterResult.getResultCode())){
           log.info("发券失败，不进行监控，code={}, appId={}, slotId={}, orderId={}", filterResult.getResultCode(), filterResult.getAppId(), filterResult.getSlotId(), filterResult.getOrderId());
           return;
        }

        try {
            // 1、当前发券的媒体是否要监控
            boolean isNeedMonitor = needMonitor(filterResult.getAppId());
            if(!isNeedMonitor){
                return;
            }

            // 2、发券计划相关的信息存储于redis，供监控系统查询
            saveAdvertInfo2Redis(filterResult);

            // 3、订单号 + 媒体id + 广告位id通过rocketMq发送至监控系统
            boolean msgResult = sendRocketMqMsg(filterResult);
            if(!msgResult){
                log.error("发券监控消息发送失败, orderId={}", filterResult.getOrderId());
            }
        }catch (Exception e){
            log.error("monitorHandle error orderId={}", filterResult.getOrderId(), e);
        }
    }

    /**
     * 当前媒体是否要监控发券
     *
     * @param appId
     * @return
     */
    private boolean needMonitor(Long appId) {

        // 监控列表为空的情况监控大盘所有媒体，发券媒体在监控列表中时才要监控
        return StringUtils.isBlank(monitorApps) || "-1".equals(monitorApps) || Stream.of(monitorApps.split(",")).map(Long::valueOf)
                                                        .anyMatch(e -> Objects.deepEquals(e, appId));
    }

    /**
     * 保存广告信息至redis
     *
     * @param filterResult
     */
    private void saveAdvertInfo2Redis(FilterResult filterResult) {

        /**
         * key: orderId
         * hk:各种场景，如：应用标签，广告位标签
         * value: 时间
         */
        // 1、订单号为redis key
        String redisKey = filterResult.getOrderId();
        HashOperations<Object, Object, Object> hashOperations = redisTemplate.opsForHash();

        // 2、构建发券信息
        Map<String, String> advertInfoMap = buildLaunchInfo(filterResult);

        // 3、放入缓存，5分钟过期
        hashOperations.putAll(redisKey, advertInfoMap);
        hashOperations.getOperations().expire(redisKey, 5, TimeUnit.MINUTES);
    }

    /**
     * 构建存入redis的发券信息
     *
     * @param filterResult
     * @return
     */
    private Map<String,String> buildLaunchInfo(FilterResult filterResult) {

        // 结果map
        Map<String, String> resultMap = Maps.newHashMap();
        resultMap.put("advertId", filterResult.getSuccessId() + "");
        resultMap.put("orientationId", filterResult.getPlanId() + "");
        resultMap.put("accountId", filterResult.getAccountId() + "");
        resultMap.put("materialId", filterResult.getMaterialId() + "");
        resultMap.put("launchType", filterResult.getType() + "");
        //发券时间
        resultMap.put("launchTime", LocalDateTime.now().toInstant(ZoneOffset.of("+8")).toEpochMilli() +"");

        ShieldStrategyVO shieldStrategyVO = filterResult.getShieldStrategyVO();
        if(shieldStrategyVO != null){
            resultMap.put("flowType", shieldStrategyVO.getFlowType() + "");
            resultMap.put("handledSlot", shieldStrategyVO.getHandledSlot() ? "1" : "0");
            resultMap.put("luckyBag", shieldStrategyVO.getLuckyBag() ? "1" : "0");
        }

        // 1、处理媒体维度
        handleAppResult(resultMap, filterResult);

        // 2、处理广告位维度
        handleSlotResult(resultMap, filterResult);

        // 3、处理活动白名单维度
        handleActivityWhiteResult(resultMap, filterResult);

        // 5、处理互选维度
        handleAppSelectResult(resultMap, filterResult);

        // 6、处理广告维度
        handleAdvertResult(resultMap, filterResult);



        return resultMap;
    }

    /**
     * 处理广告维度
     *
     * 暂时只处理素材
     *
     * @param resultMap
     * @param filterResult
     */
    private void handleAdvertResult(Map<String, String> resultMap, FilterResult filterResult) {


        List<AdvertMaterialDto> validAdvertMaterialSet = filterResult.getValidAdvertMaterialSet();
        if(CollectionUtils.isEmpty(validAdvertMaterialSet)){
            return;
        }

        Long materialId = Long.valueOf(filterResult.getMaterialId());
        Long advertId = filterResult.getSuccessId();

        // 过滤出发券的素材
        AdvertMaterialDto advertMaterialDto = validAdvertMaterialSet.stream().filter(e -> e.getAdvertId().equals(advertId) && e.getId().equals(materialId)).findFirst().orElse(null);
        if(advertMaterialDto == null){
            return;
        }

        // 广告素材修改时间
        resultMap.put("advertMaterialDate", DateUtils.getSecondStr(advertMaterialDto.getGmtModified()));

        Optional<ResoureTagsDO> resourceOptional = resourceTagsService.getMaterialTagsDO(materialId);

        // 素材标签修改时间
        resourceOptional.ifPresent(resoureTagsDO -> resultMap.put("materialTagModifiedDate", DateUtils.getSecondStr(resoureTagsDO.getGmtModified())));
    }

    private void handleActivityWhiteResult(Map<String, String> resultMap, FilterResult filterResult) {
        ActivityAdvert4MonitorDto activityAdvert4MonitorDto = filterResult.getActivityAdvert4MonitorDto();
        if(activityAdvert4MonitorDto != null){

            // 投放模式
            ActivityDirectMode4MonitorDto directMode4MonitorDto = activityAdvert4MonitorDto.getDirectMode4MonitorDto();
            // 广告列表
            List<ActivityAdvertDto>  activityAdvertDtoList = activityAdvert4MonitorDto.getActivityAdvertDtoList();


            // 请求参数
            String activityAdvert4MonitorReq = filterResult.getActivityAdvert4MonitorReq();
            if(StringUtils.isNotBlank(activityAdvert4MonitorReq)){
                resultMap.put("activityAdvert4MonitorReq", activityAdvert4MonitorReq);
            }

            // 查询时间
            if(directMode4MonitorDto != null && directMode4MonitorDto.getGmtModified() != null){
               resultMap.put("directModeDate", DateUtils.getSecondStr(directMode4MonitorDto.getGmtModified()));
            }

            // 计划时间
            if(CollectionUtils.isNotEmpty(activityAdvertDtoList)){
                // 投出去的广告计划id
                Long successAdvertId = filterResult.getSuccessId();
                // 过滤出投出来计划的修改时间
                String advertModified = activityAdvertDtoList.stream().filter(e -> e.getAdvertId().equals(successAdvertId)).map(e -> DateUtils.getSecondStr(e.getGmtModified())).findFirst().orElse("");
                if(StringUtils.isNotBlank(advertModified)){
                    resultMap.put("activityAdvertDate", advertModified);
                }
            }
        }
    }

    /**
     * 处理媒体广告互选
     *
     * @param resultMap
     * @param filterResult
     */
    private void handleAppSelectResult(Map<String, String> resultMap, FilterResult filterResult) {

        Long appId = filterResult.getAppId();
        Long materialId = Long.valueOf(filterResult.getMaterialId());
        Long advertId = filterResult.getSuccessId();

        //
        AdvertSelectedDto advertSelectedDto = filterResult.getAdvertSelectedDto();
        Set<AuditedAdvertAndMaterial4AdDto> advertAndMaterial4AdDtoSet = filterResult.getAdvertAndMaterial4AdDtoSet();

        if(advertSelectedDto != null && advertSelectedDto.getAppId().equals(appId)){

            // 互选开启的时间
            resultMap.put(AdvertLaunchMonitorConstant.BannedType.APP_ADVERT_SELECT, DateUtils.getSecondStr(advertSelectedDto.getGmtModified()));
        }

        //
        if(CollectionUtils.isNotEmpty(advertAndMaterial4AdDtoSet)){

            AuditedAdvertAndMaterial4AdDto adDto = advertAndMaterial4AdDtoSet.stream().filter(e -> e.getAdvertId().equals(advertId)).findFirst().orElse(null);
            if(adDto != null){

                // 互选广告开启时间
                resultMap.put(AdvertLaunchMonitorConstant.BannedType.APP_ADVERT_SELECT_ADVERT, DateUtils.getSecondStr(adDto.getGmtModified()));

                Set<AuditedAdvertAndMaterial4AdDto.Material> materialSet = adDto.getMaterialIds();
                if(CollectionUtils.isNotEmpty(materialSet)){
                    Date materialDate = materialSet.stream().filter(e -> e.getId().equals(materialId)).map(AuditedAdvertAndMaterial4AdDto.Material::getGmtModified).findFirst().orElse(null);

                    // 互选素材开启的时间
                    resultMap.put(AdvertLaunchMonitorConstant.BannedType.APP_ADVERT_SELECT_MATERIAL, DateUtils.getSecondStr(materialDate));
                }
            }
        }
    }

    /**
     * 处理广告位维度
     *
     * @param resultMap
     * @param filterResult
     */
    private void handleSlotResult(Map<String, String> resultMap, FilterResult filterResult) {
        // 广告位标签查询时间
        SlotTagQueryTime slotTagQueryTime = filterResult.getSlotTagQueryTime();

        // 流量线广告位标签维度的时间
        String mediaTagTime = Optional.ofNullable(slotTagQueryTime)
                                    .map(SlotTagQueryTime::getMediaTagTime)
                                    .map(DateUtils::getSecondStr)
                                    .orElse(null);
        if(mediaTagTime != null){
            resultMap.put(AdvertLaunchMonitorConstant.BannedType.SLOT_MEDIA_TAG, mediaTagTime);
        }

        // 广告位标签修改时间
        String slotTagTime = Optional.ofNullable(slotTagQueryTime)
                                    .map(SlotTagQueryTime::getSlotTagTime)
                                    .map(DateUtils::getSecondStr)
                                    .orElse(null);
        if(slotTagTime != null){
            resultMap.put(AdvertLaunchMonitorConstant.BannedType.SLOT_TAG, slotTagTime);
        }

        // 广告位落地页标签（应用管理）
        String slotPromoteMgtTagTime = Optional.ofNullable(slotTagQueryTime)
                                    .map(SlotTagQueryTime::getSlotBannedUrlTagTime)
                                    .map(DateUtils::getSecondStr)
                                    .orElse(null);
        if(slotPromoteMgtTagTime != null){
            resultMap.put(AdvertLaunchMonitorConstant.BannedType.SLOT_PROMOTE_MGT_TAG, slotPromoteMgtTagTime);
        }

        // 广告位落地页标签（流量策略）
        String slotPromoteFlowTagTime = Optional.ofNullable(slotTagQueryTime)
                                                .map(SlotTagQueryTime::getSlotFlowBannedUrlTagTime)
                                                .map(DateUtils::getSecondStr)
                                                .orElse(null);
        if(slotPromoteFlowTagTime != null){
            resultMap.put(AdvertLaunchMonitorConstant.BannedType.SLOT_PROMOTE_FLOW_TAG, slotPromoteFlowTagTime);
        }

        // 广告位流量策略素材标签
        String slotMaterialTagTime = Optional.ofNullable(slotTagQueryTime)
                                                .map(SlotTagQueryTime::getSlotFlowBannedUrlTagTime)
                                                .map(DateUtils::getSecondStr)
                                                .orElse(null);
        if(slotMaterialTagTime != null){
            resultMap.put(AdvertLaunchMonitorConstant.BannedType.SLOT_MATERIAL_TAG, slotMaterialTagTime);
        }
    }

    /**
     * 处理媒体维度
     *
     * @param resultMap
     * @param filterResult
     */
    private void handleAppResult(Map<String, String> resultMap, FilterResult filterResult) {

        // 媒体标签查询时间
        AppTagQueryTime appTagQueryTime = filterResult.getAppTagQueryTime();

        // 应用标签修改时间
        String appTagTime = Optional.ofNullable(appTagQueryTime)
                                    .map(AppTagQueryTime::getAppTagTime)
                                    .map(DateUtils::getSecondStr)
                                    .orElse(null);
        if(appTagTime != null){
            resultMap.put(AdvertLaunchMonitorConstant.BannedType.APP_TAG, appTagTime);
        }

        // 应用落地页标签（应用管理）
        String appPromoteMgtTagTime = Optional.ofNullable(appTagQueryTime)
                                            .map(AppTagQueryTime::getAppBannedUrlTagTime)
                                            .map(DateUtils::getSecondStr)
                                            .orElse(null);
        if(appTagTime != null){
            resultMap.put(AdvertLaunchMonitorConstant.BannedType.APP_PROMOTE_MGT_TAG, appPromoteMgtTagTime);
        }

        //  应用落地页标签（流量策略）
        String appPromoteFlowTagTime = Optional.ofNullable(appTagQueryTime)
                                                .map(AppTagQueryTime::getAppFlowBannedUrlTagTime)
                                                .map(DateUtils::getSecondStr)
                                                .orElse(null);
        if(appTagTime != null){
            resultMap.put(AdvertLaunchMonitorConstant.BannedType.APP_PROMOTE_FLOW_TAG, appPromoteFlowTagTime);
        }

        // 素材标签
        String appMaterialTagTime = Optional.ofNullable(appTagQueryTime)
                                            .map(AppTagQueryTime::getAppShieldMaterialTagTime)
                                            .map(DateUtils::getSecondStr)
                                            .orElse(null);
        if(appTagTime != null){
            resultMap.put(AdvertLaunchMonitorConstant.BannedType.APP_MATERIAL_TAG, appMaterialTagTime);
        }
    }

    /**
     * 发送消息至rocketmq
     *
     * @param filterResult
     * @return
     */
    private boolean sendRocketMqMsg(FilterResult filterResult) {

        // 构建消息体
        String msgBody = buildMsgBody(filterResult);

        // 发送消息
       return advertLaunchMonitorProducer.sendMessage(msgBody);
    }

    /**
     * 构建消息体
     *
     * @param filterResult
     * @return
     */
    private String buildMsgBody(FilterResult filterResult) {
        Long appId = filterResult.getAppId();
        Long slotId = filterResult.getSlotId();
        String orderId = filterResult.getOrderId();

        Map<String, String> msgMap = Maps.newHashMap();
        msgMap.put("appId", appId + "");
        msgMap.put("slotId", slotId + "");
        msgMap.put("orderId", orderId);
        return JSON.toJSONString(msgMap);
    }
}
