package cn.com.duiba.tuia.service.impl;

import cn.com.duiba.tuia.cache.MediaCacheService;
import cn.com.duiba.tuia.constants.AdvertConstants;
import cn.com.duiba.tuia.constants.AdvertSystemConfigureConstants;
import cn.com.duiba.tuia.dao.material.AdvertMaterialDAO;
import cn.com.duiba.tuia.domain.dataobject.*;
import cn.com.duiba.tuia.domain.flow.MediaList;
import cn.com.duiba.tuia.domain.model.AdvQueryParam;
import cn.com.tuia.advert.enums.ResourceTagsTypeEnum;
import cn.com.duiba.tuia.service.AdvertMaterialRecommendService;
import cn.com.duiba.tuia.service.AdvertSystemConfigService.AdvertSystemConfigEnum;
import cn.com.duiba.tuia.service.ResourceTagsService;
import cn.com.tuia.advert.model.ObtainAdvertReq;
import cn.com.tuia.advert.enums.ActivityTypeEnum;
import cn.hutool.core.collection.ConcurrentHashSet;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import kotlin.collections.EmptySet;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.*;
import java.util.stream.Collectors;

/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2017年07月11日 09:43
 * @descript:
 * @version: 1.0
 */
@Service
public class AdvertMaterialRecommendServiceImpl implements AdvertMaterialRecommendService {

    private static Logger logger = LoggerFactory.getLogger(AdvertMaterialRecommendServiceImpl.class);

    /** 活动不支持弹层类型素材*/
    private final static Integer NO_SUPPORT_LAYER = 0;

    /** 活动支持弹层类型素材*/
    private final static Integer SUPPORT_LAYER = 1;

    /** 竖屏 */
    public static final Integer S_TYPE = 1;
    /** 横屏 */
    public static final Integer H_TYPE = 2;


    @Autowired
    private AdvertMaterialDAO  advertMaterialDAO;

    @Resource
    private ExecutorService executorService;

    @Autowired
    private ResourceTagsService resourceTagsService;

    @Autowired
    private MediaCacheService mediaCacheService;

    /**
     * key:广告ID value:有效素材列表
     */
    private final LoadingCache<Long, Set<Long>> ADVERT_MATERIALS = CacheBuilder.newBuilder().initialCapacity(1000).
            recordStats().refreshAfterWrite(15, TimeUnit.MINUTES).expireAfterWrite(1,TimeUnit.HOURS).build(new CacheLoader<Long, Set<Long>>() {
        @Override
        public Set<Long> load(Long advertId) throws Exception {
            return getMaterialsByAdvertId(advertId);
        }

        @Override
        public ListenableFuture<Set<Long>> reload(final Long key, Set<Long> oldValue) throws Exception {

            ListenableFutureTask<Set<Long>> task = ListenableFutureTask.create(new Callable<Set<Long>>() {
                public Set<Long> call() throws Exception {
                    return load(key);
                }
            });
            executorService.submit(task);
            return task;
        }
    });


    /**
     * key:广告ID value:有效的视频素材列表
     */
    private final LoadingCache<Long, Map<String,Set<Long>>> ADVERT_VIDEO_MATERIALS = CacheBuilder.newBuilder().initialCapacity(1000).
            recordStats().refreshAfterWrite(15, TimeUnit.MINUTES).expireAfterWrite(1, TimeUnit.HOURS).build(new CacheLoader<Long, Map<String,Set<Long>>>() {
        @Override
        public Map<String,Set<Long>> load(Long advertId) throws Exception {
            return getVideoMaterialsByAdvertId(advertId);
        }

        @Override
        public ListenableFuture<Map<String,Set<Long>>> reload(final Long key, Map<String,Set<Long>> oldValue) throws Exception {

            ListenableFutureTask<Map<String,Set<Long>>> task = ListenableFutureTask.create(new Callable<Map<String,Set<Long>>>() {
                public Map<String,Set<Long>> call() throws Exception {
                    return load(key);
                }
            });
            executorService.submit(task);
            return task;
        }
    });

    private Map<String,Set<Long>> getVideoMaterialsByAdvertId(Long advertId) {
        List<AdvertMaterialDto> advertMaterialList = advertMaterialDAO.selectVideoMaterialList(advertId);

        if (advertMaterialList == null || CollectionUtils.isEmpty(advertMaterialList)) {
            return Collections.emptyMap();
        }

        Map<String,Set<Long>> materialIdsMap = new ConcurrentHashMap<>();

        Set<Long> oneSet = new ConcurrentHashSet<>();
        Set<Long> twoSet = new ConcurrentHashSet<>();
        Set<Long> threeSet = new ConcurrentHashSet<>();

        for (AdvertMaterialDto advertMaterialDto : advertMaterialList) {

            if(advertMaterialDto.getIsActive() == 1 && advertMaterialDto.getCheckStatus() == 0){

                Long id = advertMaterialDto.getId();
                Integer videoFormat = advertMaterialDto.getVideoFormat();
                if(S_TYPE.equals(videoFormat)) {
                    oneSet.add(id);
                    threeSet.add(id);
                }

                if(H_TYPE.equals(videoFormat)){
                    twoSet.add(id);
                    threeSet.add(id);
                }
            }
        }
        materialIdsMap.put("1",oneSet);
        materialIdsMap.put("2",twoSet);
        materialIdsMap.put("3",threeSet);
        return materialIdsMap;
    }

    /**
     * key:广告ID value:有效素材列表
     */
    private final LoadingCache<Long, List<AdvertMaterialDto>> ADVERT_MATERIALS_LIST_CACHE = CacheBuilder.newBuilder().initialCapacity(1000).
            recordStats().refreshAfterWrite(15, TimeUnit.MINUTES).expireAfterWrite(2,TimeUnit.HOURS).build(new CacheLoader<Long, List<AdvertMaterialDto>>() {
        @Override
        public List<AdvertMaterialDto> load(Long advertId) throws Exception {
            List<AdvertMaterialDto> advertMaterialList = advertMaterialDAO.selectMaterialList(advertId);
            
            if (advertMaterialList == null || CollectionUtils.isEmpty(advertMaterialList)) {
                return Collections.emptyList();
            }
            List<AdvertMaterialDto> materialList = Lists.newArrayList();
            for (AdvertMaterialDto advertMaterialDto : advertMaterialList) {
                if(advertMaterialDto.getIsActive() == 1 && advertMaterialDto.getCheckStatus() == 0){
                    materialList.add(advertMaterialDto);
                }
            }
            return materialList;
        }

        @Override
        public ListenableFuture<List<AdvertMaterialDto>> reload(final Long key, List<AdvertMaterialDto> oldValue) throws Exception {

            ListenableFutureTask<List<AdvertMaterialDto>> task = ListenableFutureTask.create(new Callable<List<AdvertMaterialDto>>() {
                public List<AdvertMaterialDto> call() throws Exception {
                    return load(key);
                }
            });
            executorService.submit(task);
            return task;
        }
    });
    
    /**
     * key:广告ID value:默认素材列表
     */
    private final LoadingCache<Long, Long> ADVERT_DEFAULT_MATERIALS = CacheBuilder.newBuilder().initialCapacity(1000).
            recordStats().refreshAfterWrite(15, TimeUnit.MINUTES).expireAfterWrite(1,TimeUnit.HOURS).build(new CacheLoader<Long, Long>() {
        @Override
        public Long load(Long advertId) throws Exception {
            return getDefaultMaterialsByAdvertId(advertId);
        }

        @Override
        public ListenableFuture<Long> reload(final Long key, Long oldValue) throws Exception {

            ListenableFutureTask<Long> task = ListenableFutureTask.create(new Callable<Long>() {
                public Long call() throws Exception {
                    return load(key);
                }
            });
            executorService.submit(task);
            return task;
        }
    });


    @Override
    public void init(List<Long> validAdvertIds) {
        List<AdvertMaterialDto> advertMaterialDtoList = advertMaterialDAO.selectActivesByAdvertIds(validAdvertIds);

        if (advertMaterialDtoList == null || advertMaterialDtoList.isEmpty()) {
            logger.error("初始化素材列表为空");
            return;
        }
        for (AdvertMaterialDto advertMaterialDto : advertMaterialDtoList) {
            try {
                Set<Long> advertMaterialDtos = ADVERT_MATERIALS.get(advertMaterialDto.getAdvertId());
                if (advertMaterialDtos == null || CollectionUtils.isEmpty(advertMaterialDtos)) {
                    advertMaterialDtos = Sets.newConcurrentHashSet();
                }
                advertMaterialDtos.add(advertMaterialDto.getId());
                ADVERT_MATERIALS.put(advertMaterialDto.getAdvertId(), advertMaterialDtos);

                //此处get即可触发调用
                ADVERT_VIDEO_MATERIALS.getUnchecked(advertMaterialDto.getAdvertId());

                List<AdvertMaterialDto> advertMaterialList = ADVERT_MATERIALS_LIST_CACHE.get(advertMaterialDto.getAdvertId());
                if(advertMaterialList == null || CollectionUtils.isEmpty(advertMaterialList)){
                    advertMaterialList = Lists.newArrayList();
                }
                advertMaterialList.add(advertMaterialDto);
                ADVERT_MATERIALS_LIST_CACHE.put(advertMaterialDto.getAdvertId(), advertMaterialList);

                //保存默认素材
                if(advertMaterialDto.getIsDefault() == AdvertConstants.ADVERT_DEFAULT_MATERIAL){
                    ADVERT_DEFAULT_MATERIALS.put(advertMaterialDto.getAdvertId(),advertMaterialDto.getId());
                }
            } catch (ExecutionException e) {
                logger.error("获取素材缓存异常", e);
            }
        }

    }

    @Override
    public List<Long> getReceiveMaterial(List<ConsumerInteractiveRecordDO> consumerVOS) {

        //返回全局配置的次数
        int len = consumerVOS.size();
        if (len == 0)
            return Lists.newArrayList();

        if (AdvertSystemConfigEnum.materialFilterConsumerCount.getIntValue() <= len){
            consumerVOS = consumerVOS.subList(0, AdvertSystemConfigEnum.materialFilterConsumerCount.getIntValue());
        }
        return consumerVOS.stream().filter(dto -> dto.getMaterialId() != null).map(ConsumerInteractiveRecordDO::getMaterialId).collect(Collectors.toList());
    }


    @Override
    public Set<Long> getAdvertVideoMaterialSet(Long advertId) {
        Map<String,Set<Long>> advertVideoMarialIdsMap = ADVERT_VIDEO_MATERIALS.getUnchecked(advertId);
        Set<Long> longs = advertVideoMarialIdsMap.get("3");
        return null == longs ?  new HashSet<>() : longs;
    }

    @Override
    public Set<Long> filterByMaterialType(Set<Long> materialIds, Long advertId, AdvQueryParam advQueryParam, ObtainAdvertReq req) {
        //12类型失效 素材过滤
        List<Integer> activityTypeExt = req.getActivityTypeExt();
        if(null != activityTypeExt && activityTypeExt.contains(12)){
            return materialIds;
        }

        //视屏素材处理
        Map<String,Set<Long>> advertVideoMarialIdsMap = ADVERT_VIDEO_MATERIALS.getUnchecked(advertId);

        if(ActivityTypeEnum.NEW_VEDIO_ADVERT_ACTIVITY_TYPE.getCode().equals(advQueryParam.getActivitySceneType())) {
            Integer videoSpecification = advQueryParam.getVideoSpecification();

            //与崔浩 确认 没有传 则是 不限-3
            String format = (null == videoSpecification ? "3":String.valueOf(videoSpecification));
            Set<Long> advertVideoMarialIds = advertVideoMarialIdsMap.get(format);
            if(CollectionUtils.isEmpty(advertVideoMarialIds)){
                materialIds.clear();
                return materialIds;
            }
            materialIds.retainAll(advertVideoMarialIds);
        }else {
            Set<Long> allAdvertVideoMarialIds = advertVideoMarialIdsMap.get("3");
            if(CollectionUtils.isNotEmpty(allAdvertVideoMarialIds)) {
                materialIds.removeAll(allAdvertVideoMarialIds);
            }
        }
        return materialIds;
    }

    @Override
    public Set<Long> filterByAdvertId(Long advertId, Long accountId, AdvQueryParam advQueryParam) {
        List<Long> alreadyMaterials = advQueryParam.getReceiveMaterials();
        Set<String> shieldMaterialTags = advQueryParam.getShieldMaterialTags();
        Set<Long> advertMarialIds = ADVERT_MATERIALS.getUnchecked(advertId);
        Set<Long> copyAdvertMaterials = Sets.newHashSet(advertMarialIds);
        //去除已经领取过的素材ID
        copyAdvertMaterials.removeAll(alreadyMaterials);

        if (CollectionUtils.isEmpty(copyAdvertMaterials) || CollectionUtils.isEmpty(shieldMaterialTags)) {
            return copyAdvertMaterials;
        }

        MediaList mediaWhiteList = advQueryParam.getMediaWhiteList();

        //被屏蔽的素材
        List<Long> shieldMaterialIds = Lists.newArrayList();
        for (Long materialId : copyAdvertMaterials) {
            Set<String> materialTags = resourceTagsService.getResoureTagsDOById(materialId, ResourceTagsTypeEnum.MATERIAL);

            //将在白名单中的素材不用屏蔽
//            if(advQueryParam.getMediaSlotStatus() && mediaCacheService.isExistMediaSlotWhite(advQueryParam.getSlotId(),advertId)){
//                continue;
//            }

            //与前面几行代码理论上等价
            if(mediaWhiteList != null && mediaWhiteList.isAdvertInList(advertId,accountId)){
                continue;
            }

            if (CollectionUtils.isNotEmpty(materialTags)) {

                Set<String> copyMaterialTags = Sets.newHashSet(materialTags);
                copyMaterialTags.retainAll(shieldMaterialTags);
                if(copyMaterialTags.size() > 0) {
                    shieldMaterialIds.add(materialId);
                }
            }
        }

        if (CollectionUtils.isNotEmpty(shieldMaterialIds)) {
            copyAdvertMaterials.removeAll(shieldMaterialIds);
        }

        return copyAdvertMaterials;
    }

    @Override
    public Set<Long> filterByActivityType(Integer activityMaterialType,Long advertId, Set<Long> alreadyMaterials) {

        if(CollectionUtils.isEmpty(alreadyMaterials)){
            return alreadyMaterials;
        }

        Set<Long> resultMaterialIds = alreadyMaterials;
        if(NO_SUPPORT_LAYER.equals(activityMaterialType) || activityMaterialType == null){
            resultMaterialIds = getMaterialListByAdvertId(advertId).stream().filter(dto->(alreadyMaterials.contains(dto.getId()) && (dto.getNewMaterialType() == 0 || dto.getNewMaterialType() == 3))).map(AdvertMaterialDto::getId).collect(Collectors.toSet());
        }

        return resultMaterialIds;
    }
    
    @Override
    public Set<Long> getMaterialSetByAdvertId(Long advertId){
        try{
            Set<Long> advertMarialIds = ADVERT_MATERIALS.getUnchecked(advertId);
            return advertMarialIds;
        }catch(Exception e){
            return Collections.emptySet();
        }
    }
    
    @Override
    public List<AdvertMaterialDto> getMaterialListByAdvertId(Long advertId) {
        try{
            List<AdvertMaterialDto> advertMarialList = ADVERT_MATERIALS_LIST_CACHE.getUnchecked(advertId);
            return advertMarialList;
        }catch(Exception e){
            return Collections.emptyList();
        }
    }


    private Long getDefaultMaterialsByAdvertId(Long advertId){
        Long materialId = advertMaterialDAO.selectDefaultId(advertId);
        return materialId == null ? -1L : materialId;
    }

    @Override
    public Set<Long> getMaterialsByAdvertId(Long advertId) {
        List<AdvertMaterialDto> advertMaterialList = advertMaterialDAO.selectMaterialList(advertId);
       
        if (advertMaterialList == null || CollectionUtils.isEmpty(advertMaterialList)) {
            return Collections.emptySet();
        }
        Set<Long> materialIds = Sets.newConcurrentHashSet();
        for (AdvertMaterialDto advertMaterialDto : advertMaterialList) {

            if(advertMaterialDto.getIsActive() == 1 && advertMaterialDto.getCheckStatus() == 0){
                materialIds.add(advertMaterialDto.getId());
            }
        }
        return materialIds;
    }

    @Override
    public Long getDefaultByAdvertId(Long advertId) {
        try {
            return ADVERT_DEFAULT_MATERIALS.get(advertId);
        } catch (ExecutionException e) {
            logger.error("获取默认素材ID异常",e);
            return -1L;
        }
    }

    @Override
    public Long getWindId(Long advertId, Long materialId) {

        try {
            Long windId = null;
            List<AdvertMaterialDto> advertMarialList = ADVERT_MATERIALS_LIST_CACHE.getUnchecked(advertId);
            Optional<AdvertMaterialDto> advertMaterialDto = advertMarialList.stream().filter(dto -> dto.getId().equals(materialId)).findFirst();

            if (advertMaterialDto.isPresent()) {
                windId = advertMaterialDto.get().getWindId();
            }
            return windId;
        }catch (Exception e){
            logger.warn("getWindId exception",e);
            return null;
        }
    }

    @Override
    public Set<Long> filterBySlotId(AdvQueryParam advQueryParam, Set<Long> materialIds, Long advertId, ObtainAdvertReq req) {
        Set<Long> copyMaterials = Sets.newHashSet(materialIds);

        //美团adx 不需要过滤
        if (null != req.getAdxMediaType() && req.getAdxMediaType() == 1) {
            return copyMaterials;
        }

        AdvertFilterKeywordDO advertFilterKeywordDO = advQueryParam.getAdvertFilterKeywordDO();
        if (Objects.nonNull(advertFilterKeywordDO) && CollectionUtils.isNotEmpty(materialIds)) {
            GeneralFilterKeywordDO generalFilterKeywordDO = advertFilterKeywordDO.getGeneralFilterKeywordDO();
            if (Objects.nonNull(generalFilterKeywordDO) && MapUtils.isNotEmpty(generalFilterKeywordDO.getShieldMaterialList())) {
                //广告位下每个广告素材屏蔽列表
                Map<Long, Set<Long>> shieldMaterialList = generalFilterKeywordDO.getShieldMaterialList();
                if (shieldMaterialList.containsKey(advertId)) {
                    Set<Long> materialIdList = shieldMaterialList.get(advertId);
                    if (CollectionUtils.isNotEmpty(materialIdList) && CollectionUtils.isNotEmpty(copyMaterials)) {
                        copyMaterials.removeAll(materialIdList);
                    }
                }
            }
        }

        return copyMaterials;
    }

    @Override
    public Set<Long> filterNewTradeBySlotId(AdvQueryParam advQueryParam, Set<Long> materialIds, Long advertId, ObtainAdvertReq req) {
        Set<Long> copyMaterials = Sets.newHashSet(materialIds);

        //美团adx 不需要过滤
        if (null != req.getAdxMediaType() && req.getAdxMediaType() == 1) {
            return copyMaterials;
        }

        AdvertFilterKeywordDO advertFilterKeywordDO = advQueryParam.getAdvertFilterKeywordDO();
        if (Objects.nonNull(advertFilterKeywordDO) && CollectionUtils.isNotEmpty(materialIds)) {
            NewTradeFilterKeywordDO newTradeFilterKeywordDO = advertFilterKeywordDO.getNewTradeFilterKeywordDO();
            if (Objects.nonNull(newTradeFilterKeywordDO) && MapUtils.isNotEmpty(newTradeFilterKeywordDO.getShieldMaterialList())) {
                //广告位下每个广告素材屏蔽列表
                Map<Long, Set<Long>> shieldMaterialList = newTradeFilterKeywordDO.getShieldMaterialList();
                if (shieldMaterialList.containsKey(advertId)) {
                    Set<Long> materialIdList = shieldMaterialList.get(advertId);
                    if (CollectionUtils.isNotEmpty(materialIdList) && CollectionUtils.isNotEmpty(copyMaterials)) {
                        copyMaterials.removeAll(materialIdList);
                    }
                }
            }
        }

        return copyMaterials;
    }
}
