package cn.com.duiba.tuia.service.impl;

import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import cn.com.duiba.tuia.dao.engine.AdvertiserAuditDao;
import cn.com.duiba.tuia.domain.dataobject.*;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;

import cn.com.duiba.tuia.cache.AdvertMapCacheManager;
import cn.com.duiba.tuia.cache.AdvertPkgPutFlagCacheService;
import cn.com.duiba.tuia.cache.AdvertTargetAppCacheService;
import cn.com.duiba.tuia.cache.BeanCopierManager;
import cn.com.duiba.tuia.constants.AdvertConstants;
import cn.com.duiba.tuia.dao.advert.AdvertAppPackageDAO;
import cn.com.duiba.tuia.dao.advert_tag.AdvertTagDAO;
import cn.com.duiba.tuia.dao.apppackage.AppPackageDAO;
import cn.com.duiba.tuia.dao.engine.AdvertOrientationPackageDAO;
import cn.com.duiba.tuia.dao.engine.OrientationFocusAppConvertCostDAO;
import cn.com.duiba.tuia.dao.engine.TradeTagRuleDAO;
import cn.com.duiba.tuia.dao.promotetest.AdvertPromoteTestDAO;
import cn.com.duiba.tuia.dao.targetapp.AdvertTargetAppDAO;
import cn.com.duiba.tuia.domain.model.AdvertOrientationPackageDto;
import cn.com.duiba.tuia.domain.model.TagRuleScope;
import cn.com.duiba.tuia.domain.vo.AdvertOrientationPackageVO;
import cn.com.duiba.tuia.domain.vo.AdvertVO;
import cn.com.duiba.tuia.enums.AdvertTradeAcceptLevelEnum;
import cn.com.duiba.tuia.exception.TuiaException;
import cn.com.duiba.tuia.service.AdvertOrientationService;
import cn.com.duiba.tuia.service.AdvertPeriodService;
import cn.com.duiba.tuia.service.TradeTagRuleService;
import cn.com.duiba.tuia.tool.StringTool;
import cn.com.duiba.tuia.utils.TuiaStringUtils;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.tuia.advert.constants.CommonConstant;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2017年03月29日 11:20
 * @descript:
 * @version: 1.0
 */
@Service
public class AdvertOrientationServiceImpl implements AdvertOrientationService{

    private final int maxCapacity = 1000;

    @Autowired
    private AdvertPeriodService advertPeriodService;
    @Autowired
    private AdvertOrientationPackageDAO orientationPackageDAO;

    @Autowired
    private OrientationFocusAppConvertCostDAO orientationFocusAppConvertCostDAO;

    @Autowired
    private AdvertPkgPutFlagCacheService advertPkgPutFlagCacheService;

    @Resource
    private AdvertTargetAppDAO advertTargetAppDAO;

    @Resource
    private AppPackageDAO appPackageDAO;

    @Resource
    private AdvertAppPackageDAO advertAppPackageDAO;

    @Autowired
    private AdvertTargetAppCacheService advertTargetAppCacheService;
    @Autowired
    private AdvertMapCacheManager advertMapCacheManager;
    @Autowired
    private TradeTagRuleDAO tradeTagRuleDAO;
    @Autowired
    private TradeTagRuleService tradeTagRuleService;
    @Resource
    private AdvertTagDAO advertTagDAO;
    @Resource
    private ExecutorService executorService;

    @Autowired
    private AdvertPromoteTestDAO advertPromoteTestDAO;

    @Autowired
    private AdvertiserAuditDao advertiserAuditDao;


    private final AdvertOrientationPackageDto EMPTY = new AdvertOrientationPackageDto();

    private final List<AdvertOrientationPackageVO> EMPTY_LIST = Lists.newArrayListWithCapacity(0);

    /**定向配置投放缓存,根据配置包id查询 有消息同步，不用设置过期时间来保证避免读取旧值*/
    private final LoadingCache<Long, Optional<AdvertOrientationPackageDto>> ORIENTATION_CACHE =
            CacheBuilder.newBuilder().initialCapacity(maxCapacity).recordStats().refreshAfterWrite(30,
            TimeUnit.MINUTES).build(new CacheLoader<Long, Optional<AdvertOrientationPackageDto>>() {
                @Override
                public Optional<AdvertOrientationPackageDto> load(Long key) {
                    AdvertOrientationPackageDO orientationPackageDO = orientationPackageDAO.selectById(key);
                    if (orientationPackageDO == null || orientationPackageDO.getEnableStatus() == AdvertConstants.ENABLE_STATUS_NO) {
                        return Optional.ofNullable(null);
                    } 
                    return Optional.ofNullable(conversionOrientationDto(orientationPackageDO));
                }
                @Override
                public ListenableFuture<Optional<AdvertOrientationPackageDto>> reload(Long  key, Optional<AdvertOrientationPackageDto> oldValue) throws Exception {
                    ListenableFutureTask<Optional<AdvertOrientationPackageDto>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });

    /**默认定向配置缓存，根据广告id查询 有消息同步，不用设置过期时间来保证避免读取旧值*/
    private final LoadingCache<Long, Optional<AdvertOrientationPackageDto>> DEFAULT_ORIENTATION_CACHE =
            CacheBuilder.newBuilder().initialCapacity(maxCapacity).recordStats().refreshAfterWrite(30,
            TimeUnit.MINUTES).build(new CacheLoader<Long,  Optional<AdvertOrientationPackageDto>>() {
                @Override
                public Optional<AdvertOrientationPackageDto> load(Long key) {
                    Optional<AdvertOrientationPackageDO> defaultOrientPackage= orientationPackageDAO.selectByAdvertIdList(key).stream()
                            .filter(x->x.getIsDefault().equals(AdvertConstants.DEFAULT_ORIENTATION)).findFirst();
                    if(!defaultOrientPackage.isPresent()){
                        return Optional.ofNullable(null);
                    }
                    return Optional.ofNullable(conversionOrientationDto(defaultOrientPackage.get()));
                }
                @Override
                public ListenableFuture<Optional<AdvertOrientationPackageDto>> reload(Long  key, Optional<AdvertOrientationPackageDto> oldValue) throws Exception {
                    ListenableFutureTask<Optional<AdvertOrientationPackageDto>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });


    /**广告定向投放缓存,所有广告对应的定向配置缓存 有消息同步，不用设置过期时间来保证避免读取旧值*/
    private final LoadingCache<Long, List<AdvertOrientationPackageVO>> ADVERT_ORIENTATION_CACHE =
            CacheBuilder.newBuilder().initialCapacity(maxCapacity).
            recordStats().refreshAfterWrite(30, TimeUnit.MINUTES).build(new CacheLoader<Long,  List<AdvertOrientationPackageVO>>() {
        @Override
        public  List<AdvertOrientationPackageVO> load(Long key) {
            List<AdvertOrientationPackageVO> packageVOS = getOrientationPackagesByDB(key);
            return packageVOS.size() == 0 ? EMPTY_LIST : packageVOS;
        }
        @Override
        public ListenableFuture<List<AdvertOrientationPackageVO>> reload(Long  key, List<AdvertOrientationPackageVO> oldValue) throws Exception {
            ListenableFutureTask<List<AdvertOrientationPackageVO>> task = ListenableFutureTask.create(() -> load(key));
            executorService.submit(task);
            return task;
        }

    });

    @Override
    public List<AdvertOrientationPackageVO> getOrientationList(final Long advertId) {
       return ADVERT_ORIENTATION_CACHE.getUnchecked(advertId);
    }

    /**
     * 根据定向包id查询
     * @param id
     * @return
     */
    @Override
    public AdvertOrientationPackageDto getOrientation(final Long id) {
        return ORIENTATION_CACHE.getUnchecked(id).orElse(EMPTY);
    }

    /**
     * 根据广告id和定向包id组合查询，后者为0的话则查询广告的默认配置包
     * @param advertId
     * @param orientationPackageId
     * @return
     */
    @Override
    public AdvertOrientationPackageDto getOrientation(Long advertId, Long orientationPackageId) {
        if(orientationPackageId.equals(AdvertConstants.DEFAULT_ORIENTATION_ID)){
            return DEFAULT_ORIENTATION_CACHE.getUnchecked(advertId).orElse(EMPTY);
        }
        return ORIENTATION_CACHE.getUnchecked(orientationPackageId).orElse(EMPTY);

    }

    @Override
    public void updateOrientation(AdvertOrientationPackageDO packageDO) {
        //配置包缓存
        ORIENTATION_CACHE.refresh(packageDO.getId());
        Boolean isDefault=false;
        if (packageDO.getIsDefault() == AdvertConstants.DEFAULT_ORIENTATION) {
            //默认配置包缓存
            DEFAULT_ORIENTATION_CACHE.refresh(packageDO.getAdvertId());
            isDefault=true;
        }
        //广告对应的所有配置包缓存
        ADVERT_ORIENTATION_CACHE.refresh(packageDO.getAdvertId());
        //配置包日预算缓存
        advertPkgPutFlagCacheService.invalidPkgPutCache(packageDO.getAdvertId(),packageDO.getId());//失效配置包日预算投放缓存
        // 定向媒体缓存
        advertTargetAppCacheService.invalidAdvertTargetAppCache(packageDO.getAdvertId(),isDefault?0:packageDO.getId());
    }

    @Override
    public void updateDefaultOrientation(Long advertId) {
        //默认配置包缓存
        DEFAULT_ORIENTATION_CACHE.refresh(advertId);
        //广告对应的所有配置包缓存
        ADVERT_ORIENTATION_CACHE.refresh(advertId);
        //配置包日预算缓存
        advertPkgPutFlagCacheService.invalidPkgPutCache(advertId);
        // 定向媒体缓存
        advertTargetAppCacheService.invalidAdvertTargetAppCache(advertId, 0L);
    }


    @Override
    public void initAdvertOrirntationList(List<Long> advertIdList) {

        List<AdvertOrientationPackageDO> orientationPackageDOS = orientationPackageDAO.selectListByAdvertIds(advertIdList);
        if (CollectionUtils.isEmpty(orientationPackageDOS)) {
            return;
        }


        //根据广告id查询配置的所有多连接测试
        List<AdvertPromoteTestDO> advertPromoteTestDOS = advertPromoteTestDAO.selectOrientationMorePromoteUrlByAdvertList(advertIdList);

        Map<Long, List<AdvertOrientationPackageDO>> advertOrientationsMap = orientationPackageDOS.stream().collect(Collectors.groupingBy(AdvertOrientationPackageDO::getAdvertId));
        List<AdvertOrientationPackageVO> orientationPackageVOS = Lists.newArrayListWithCapacity(orientationPackageDOS.size());
        advertOrientationsMap.forEach((k, v) -> orientationPackageVOS.addAll(this.convertDoListToVo(v, k, advertPromoteTestDOS)));

        //构建定向列表
        Map<Long,List<AdvertOrientationPackageVO>> orientationMap = this.getOrientationListMap(orientationPackageVOS);

        //放入缓存
        for (Map.Entry<Long, List<AdvertOrientationPackageVO>> entry : orientationMap.entrySet()) {
            ADVERT_ORIENTATION_CACHE.put(entry.getKey(),entry.getValue());
            //TODO:看情况加DEFAULT_ORIENTATION_CACHE的初始化
        }

    }

    /**
     * 查询所有广告除开默认配置的广告配置
     * @see cn.com.duiba.tuia.service.AdvertOrientationService#getOrientationList(java.util.List)
     */
    @Override
    public List<AdvertOrientationPackageDO> getOrientationList(List<Long> advertIds) {
        return orientationPackageDAO.selectByAdvertIds(advertIds);
    }

    /**
     * 构建投放时段列表
     * @param orientationPackageVOS
     * @return
     */
    private Map<Long,List<AdvertOrientationPackageVO>> getOrientationListMap(List<AdvertOrientationPackageVO> orientationPackageVOS){

        Map<Long,List<AdvertOrientationPackageVO>> orientationMap = Maps.newConcurrentMap();
        for(AdvertOrientationPackageVO orientationPackageVO : orientationPackageVOS){

            if(orientationMap.containsKey(orientationPackageVO.getAdvertId())){
                orientationMap.get(orientationPackageVO.getAdvertId()).add(orientationPackageVO);
            }else {

                List<AdvertOrientationPackageVO> orientationPackageVOList = Lists.newArrayListWithCapacity(orientationPackageVOS.size());
                orientationPackageVOList.add(orientationPackageVO);
                orientationMap.put(orientationPackageVO.getAdvertId(),orientationPackageVOList);
            }
        }

        return orientationMap;
    }

    /**
     * 构建定向配置包
     * @param advertId
     * @return
     */
    @Override
    public List<AdvertOrientationPackageVO> getOrientationPackagesByDB(Long advertId){
        List<AdvertOrientationPackageDO> orientationPackages = orientationPackageDAO.selectByAdvertIdList(advertId);

        //根据广告id查询配置的所有多连接测试
        List<AdvertPromoteTestDO> advertPromoteTestDOS = advertPromoteTestDAO.selectOrientationMorePromoteUrl(advertId);
        return this.convertDoListToVo(orientationPackages, advertId, advertPromoteTestDOS);
    }

    /**
     * 将数据库模型转为VO
     * @param orientationPackageDOS
     * @param advertPromoteTestDOS
     * @return
     */
    private List<AdvertOrientationPackageVO> convertDoListToVo(List<AdvertOrientationPackageDO> orientationPackageDOS, Long advertId,
                                                               List<AdvertPromoteTestDO> advertPromoteTestDOS) {
        if(CollectionUtils.isEmpty(orientationPackageDOS)){
            return Lists.newArrayList();
        }
        // 查询广告
        AdvertVO advertVO  = advertMapCacheManager.getAdvertCache(advertId);
        String matchTagNums;
        if (advertVO == null || null == advertVO.getAdvertTagDO() || StringUtils.isEmpty(advertVO.getAdvertTagDO().getMatchTagNums())) {
            matchTagNums = getAdvertTag(advertId);
        }else{
            matchTagNums = advertVO.getAdvertTagDO().getMatchTagNums();
        }

        Map<Long, String> idMappingPromoteUrl = Maps.newHashMap();
        if(CollectionUtils.isNotEmpty(advertPromoteTestDOS)){
            idMappingPromoteUrl = advertPromoteTestDOS.stream().collect(toMap(AdvertPromoteTestDO::getOrientPackageId, AdvertPromoteTestDO::getPromoteUrl));
        }
        List<AdvertOrientationPackageVO> orientationPackageVOS = Lists.newArrayListWithCapacity(orientationPackageDOS.size());
        String tradeAcceptTagNum = getAdvertTradeAcceptTagNum(StringTool.getStringSetByStr(matchTagNums));
        for (AdvertOrientationPackageDO packageDO : orientationPackageDOS) {
            AdvertOrientationPackageVO packageVO = new AdvertOrientationPackageVO();
            //平台
            packageVO.setPlatform(TuiaStringUtils.getStringListByStr(packageDO.getPlatform()));
            //运营商
            packageVO.setOperators(TuiaStringUtils.getIntegerListByStr(packageDO.getOperators(), ","));
            //网络类型
            packageVO.setNetworkType(TuiaStringUtils.getIntegerListByStr(packageDO.getNetworkType(), ","));
            //年龄区间
            packageVO.setAgeRegion(TuiaStringUtils.getIntegerListByStr(packageDO.getAgeRegion(), "-"));
            //绑定素材
            packageVO.setMaterialIds(TuiaStringUtils.getLongSetByStr(packageDO.getMaterialIds(), ","));
            //屏蔽素材
            packageVO.setBannedAppFlowType(TuiaStringUtils.getStringListByStr(packageDO.getBannedAppFlowType()));
            
            BeanCopierManager.getAdvertPackageBeanCopier().copy(packageDO, packageVO, null);
            //是否是默认配置包
            packageVO.setDefaultOrientation(packageDO.getIsDefault().equals(CommonConstant.YES));
            //获取配置包id
            packageVO.setId(packageVO.isDefaultOrientation() ? 0L : packageDO.getId());
            //配置包地域
            packageVO.setRegionIds(packageDO.getRegionIds());
            //投放的活动类型
            packageVO.setActivityType(packageDO.getActivityType());
            //投放的包类型(普通生成，智能生成)
            packageVO.setPackageType(packageDO.getPackageType());
            //配置包预算
            packageVO.setBudgetPerDay(packageDO.getBudgetPerDay());
            //配置包真实id
            packageVO.setOrientationId(packageDO.getId());
            //手机品牌
            packageVO.setBrandName(packageDO.getBrandName());
            // 广告行业标签,构建行业-兴趣点
            //packageVO.setUserInterest(getAdvertTradeAccept(tradeAcceptTagNum, packageDO.getTradeAccept(), packageDO.getCrowdInterest(),packageDO.getCrowdTradePackageTag(),packageDO.getSystemRecommendTag()));
            // 配置投放类型
            packageVO.setPkgPutType(packageDO.getPkgPutType());
            packageVO.setHitStatus(packageDO.getHitStatus());
            packageVO.setHitValue(packageDO.getHitValue());
            packageVO.setSupportStatus(packageDO.getSupportStatus());
            packageVO.setTargetAppLimit(packageDO.getTargetAppLimit());
            packageVO.setAutoMatch(packageDO.getAutoMatch());
            //设置配置维度多连接测试落地页链接
            String orientationPromoteUrl = idMappingPromoteUrl.get(packageDO.getId());

            Integer enableAdvertUrl = packageDO.getEnableAdvertUrl();
            //改配置不使用广告推广链接
            packageVO.setPromoteTestUrl(enableAdvertUrl==1?null:orientationPromoteUrl);
            orientationPackageVOS.add(packageVO);
        }

        return orientationPackageVOS;
    }


    /**
     * 根据定向配置获取 配置的落地页链接
     *
     * @param orientId
     * @return
     */
    @Override
    public String getOrientPromoteUrl(Long orientId) {

        AdvertPromoteTestDO advertPromoteTestDO = advertPromoteTestDAO.selectByOrientationId(orientId);

        if (null == advertPromoteTestDO) {
            return null;
        }

        Long advertId = advertPromoteTestDO.getAdvertId();
        Long promoiteUrlId = advertPromoteTestDO.getId();
        AdvertiserCheckAdvertDO advertiserCheckAdvertDO = advertiserAuditDao.selectByUniqueKey(advertId, promoiteUrlId);

        if(null == advertiserCheckAdvertDO || Objects.equals(3,advertiserCheckAdvertDO.getCheckStatus())){
            return advertPromoteTestDO.getPromoteUrl();
        }

        return null;
    }

    /**
     * 
     * conversionOrientationDto:(AdvertOrientationPackageDO转AdvertOrientationPackageDto). <br/>
     *
     * @author chencheng
     * @param orientationPackageDO
     * @return
     * @since JDK 1.8
     */
    private AdvertOrientationPackageDto conversionOrientationDto(AdvertOrientationPackageDO orientationPackageDO) {
        
        AdvertOrientationPackageDto advertOrientationPackageDto = BeanUtils.copy(orientationPackageDO, AdvertOrientationPackageDto.class);
        //  构建多连接测试连接
        String orientationPromoteUrl = getOrientPromoteUrl(orientationPackageDO.getId());
        //不使用广告本身的推广链接
        if(advertOrientationPackageDto.getEnableAdvertUrl()!= 1){
            advertOrientationPackageDto.setPromoteTestUrl(orientationPromoteUrl);
        }
        // 查询广告
        /*AdvertVO advertVO  = advertMapCacheManager.getAdvertCache(orientationPackageDO.getAdvertId());
        String matchTagNums;
        if (advertVO == null || null == advertVO.getAdvertTagDO() || StringUtils.isEmpty(advertVO.getAdvertTagDO().getMatchTagNums())) {
            matchTagNums = getAdvertTag(orientationPackageDO.getAdvertId());
        }else{
            matchTagNums = advertVO.getAdvertTagDO().getMatchTagNums();
        }
        
        // 广告行业标签,构建行业-兴趣点
        String tradeAcceptTagNum = getAdvertTradeAcceptTagNum(StringTool.getStringSetByStr(matchTagNums));
        advertOrientationPackageDto.setUserInterest(getAdvertTradeAccept(tradeAcceptTagNum, advertOrientationPackageDto.getTradeAccept(), advertOrientationPackageDto.getCrowdInterest(),advertOrientationPackageDto.getCrowdTradePackageTag(),advertOrientationPackageDto.getSystemRecommendTag()));
*/        return advertOrientationPackageDto;
    }

    /**
     * 
     * getAdvertTag:(查询广告行业标签). <br/>
     *
     * @author chencheng
     * @param advertId
     * @return
     * @since JDK 1.8
     */
    private String getAdvertTag(Long advertId) {
        
        try {
            AdvertTagDO advertTagDO = advertTagDAO.selectByAdvertId(advertId);
            return advertTagDO.getMatchTagNums();
        } catch (TuiaException e) {
            return null;
        }
    }

    /**
    *
    * getAdvertTradeAcceptTagNum:(获取唯一的一个新行业标签). <br/>
    *
    * @author chencheng
    * @param advtags
    * @return
    * @since JDK 1.8
    */
   private String getAdvertTradeAcceptTagNum(Set<String> advtags) {

       if (CollectionUtils.isEmpty(advtags)) {
           return null;
       }
       List<String> newTaglist = advtags.stream().filter(tag -> tag.contains(".")).collect(toList());
       if (CollectionUtils.isEmpty(newTaglist)) {
           return null;
       }
       return newTaglist.get(0);
   }
   
   /**
   *
   * getAdvertTradeAccept:(将行业标签与行业接受度组装). <br/>
   *
   * @author chencheng
   * @param advtags
   * @param tradeAccept
   * @return
   * @since JDK 1.8
   */
   @Override
  public List<String> getAdvertTradeAccept(Set<String> advtags, List<String> tradeAccept) {

       // 获取唯一的一个新行业标签
       String tradeAcceptTagNum = getAdvertTradeAcceptTagNum(advtags);

      if (StringUtils.isBlank(tradeAcceptTagNum)) {
          return Lists.newArrayList();
      }
      List<String> tradeAcceptLevel = tradeAccept;
      // 不选或 行业兴趣全选表示不限
      /*if ((StringUtils.isBlank(tradeAccept) && StringUtils.isBlank(crowdInterest) && StringUtils.isBlank(crowdTradePackageTag) && StringUtils.isBlank(systemRecommendTag)) || tradeAcceptLevel.size() == AdvertTradeAcceptLevelEnum.values().length ) {
          return Lists.newArrayList();
      }*/
      List<String> advertTradeAccept = Lists.newArrayList();

      if (CollectionUtils.isEmpty(tradeAcceptLevel)) {
          return advertTradeAccept;
      }
      
      String nowTradeAcceptTagNum = getTradeTagRuleScopeTagNum(tradeAcceptTagNum);
      
      // 组装数据，tagNum-accept
      List<String> tradeAccepTagNum = tradeAcceptLevel.stream().map(accepType -> {
          return nowTradeAcceptTagNum+"-"+accepType;
      }).collect(toList());

      advertTradeAccept.addAll(tradeAccepTagNum);
      return advertTradeAccept;
  }

    private String getTradeTagRuleScopeTagNum(String tradeAcceptTagNum) {
        
        // 查询行业标签打分规则
        TagRuleScope tagRuleScope = tradeTagRuleService.getTagScopeAll();
        if (tagRuleScope == null || tagRuleScope.getTagRuleDOMap() == null) {
           return tradeAcceptTagNum;
        }
        Map<String, TradeTagRuleScopeDO> tagRuleDOMap = tagRuleScope.getTagRuleDOMap();
        if (tagRuleDOMap.get(tradeAcceptTagNum)!= null) {
           return tradeAcceptTagNum;
        }
        // 一级行业标签打分
        String levelTwoTagNum = tradeAcceptTagNum.substring(0, 5);
        if (tagRuleDOMap.get(levelTwoTagNum)!= null) {
            return levelTwoTagNum;
        }
        return tradeAcceptTagNum;
    }

@Override
public Boolean checkUserInterest(Set<String> userInterest, List<String> queryUserInterest) {
    
    // 请求为空或者配置中不限为true
    if (CollectionUtils.isEmpty(queryUserInterest) || CollectionUtils.isEmpty(userInterest) || userInterest.contains("-1")) {
        return true;
    }
    // 定向人群取交集
    return CollectionUtils.isNotEmpty(CollectionUtils.intersection(userInterest, queryUserInterest));
}

/*
@Override
public Set<String> getAdvertTradeAccept(Set<String> advtags, String tradeAccept, String crowdInterest,
                                         String crowdTradePackageTag ,String systemRecommendTag) {
    // 获取唯一的一个新行业标签
    String tradeAcceptTagNum = getAdvertTradeAcceptTagNum(advtags);
    // 将行业标签与行业接受度组装
    return new HashSet<>(getAdvertTradeAccept(tradeAcceptTagNum, tradeAccept, crowdInterest, crowdTradePackageTag, systemRecommendTag));
}
*/


}
