package cn.com.duiba.tuia.service.impl;

import static cn.com.duiba.tuia.core.api.enums.advert.AdvertPkgPeriodTypeEnum.PERIOD_TYPE_COUNT_BUDGET;
import static cn.com.duiba.tuia.core.api.enums.advert.AdvertPkgPeriodTypeEnum.PERIOD_TYPE_COUNT_COUPON;
import static cn.com.duiba.tuia.core.api.enums.advert.AdvertPkgPeriodTypeEnum.PERIOD_TYPE_HOUR_BUDGET;
import static cn.com.duiba.tuia.core.api.enums.advert.AdvertPkgPeriodTypeEnum.PERIOD_TYPE_HOUR_COUPON;
import static cn.com.duiba.tuia.core.api.enums.advert.AdvertPkgPeriodTypeEnum.getByCode;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import cn.com.duiba.tuia.message.rocketmq.RefreshCacheMqProducer;
import cn.com.tuia.advert.constants.ConfigRocketMqTags;
import com.alibaba.fastjson.JSON;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import cn.com.duiba.tuia.constants.AdvertConstants;
import cn.com.duiba.tuia.core.api.enums.advert.AdvertPkgPeriodTypeEnum;
import cn.com.duiba.tuia.dao.advert.AdvertPlanPeriodDAO;
import cn.com.duiba.tuia.dao.advert.PeriodIssueCountDAO;
import cn.com.duiba.tuia.dao.advert_tag.AdvertTagDAO;
import cn.com.duiba.tuia.domain.dataobject.AdvertPlanPeriodDO;
import cn.com.duiba.tuia.domain.dataobject.AdvertTagDO;
import cn.com.duiba.tuia.domain.dataobject.PeriodIssueCountDO;
import cn.com.duiba.tuia.domain.vo.GlobalConfigFlowVO;
import cn.com.duiba.tuia.exception.TuiaException;
import cn.com.duiba.tuia.service.AdvertPeriodService;
import cn.com.duiba.tuia.service.GlobalConfigFlowService;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.tuia.advert.message.RedisMessageChannel;
import cn.hutool.db.Entity;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Splitter;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;

/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2017年04月06日 16:18
 * @descript:
 * @version: 1.0
 */
@Service
public class AdvertPeriodServiceImpl implements AdvertPeriodService, InitializingBean {
    public   final Logger logger = LoggerFactory.getLogger(AdvertPeriodServiceImpl.class);

    public static final Integer IS_PERIOD=3;

    public static final String WEI_XIN_JIA_FEN="04.01.0001";


    private static final String START_HOUR="startHour";
    private static final String END_HOUR="endHour";
    
    private Map<String, Boolean> advertPeriodPut;

    @Autowired
    private PeriodIssueCountDAO periodIssueCountDAO;
    @Resource
    private RedisTemplate<String, String> redisTemplate;
    @Autowired
    private AdvertPlanPeriodDAO advertPlanPeriodDAO;
    @Resource
    private ExecutorService executorService;

    @Autowired
    private AdvertTagDAO advertTagDao;

    @Autowired
    private GlobalConfigFlowService globalConfigFlowService;

    @Autowired
    private RefreshCacheMqProducer refreshCacheMqProducer;

    /**
     * 启动为当前时间，每次取缓存都会判断时间间隔是否大于6个小时，如果大于6个小时则执行一次“清除昨天缓存
     */
    private  long perTime = System.currentTimeMillis();

    //有消息同步，不用设置过期时间来保证避免读取旧值
    private final LoadingCache<Long, Optional<AdvertPlanPeriodDO>> ADVERT_PERIOD_CACHE = CacheBuilder.newBuilder().initialCapacity(1000).
            recordStats().refreshAfterWrite(5, TimeUnit.MINUTES).build(new CacheLoader<Long, Optional<AdvertPlanPeriodDO>>() {
        @Override
        public Optional<AdvertPlanPeriodDO> load(Long period) throws Exception {
            AdvertPlanPeriodDO periodDO = advertPlanPeriodDAO.selectPeriodById(period);
            return Optional.ofNullable(periodDO);
        }
        @Override
        public ListenableFuture<Optional<AdvertPlanPeriodDO>> reload(Long period, Optional<AdvertPlanPeriodDO> oldValue) throws Exception {
            ListenableFutureTask<Optional<AdvertPlanPeriodDO>> task = ListenableFutureTask.create(() -> load(period));
            executorService.submit(task);
            return task;
        }
    });
    private final LoadingCache<String, List<AdvertPlanPeriodDO>> ADVERT_PERIOD_ORIENT_CACHE1 = CacheBuilder.newBuilder().initialCapacity(1000)
            .refreshAfterWrite(1, TimeUnit.HOURS).build(new CacheLoader<String, List<AdvertPlanPeriodDO>>() {
        @Override
        public List<AdvertPlanPeriodDO> load(String key) throws Exception {
            List<String> strs= Splitter.on("|").splitToList(key);//advertId|orientId,0的话要转换为null数据库存储的是null
            Long advertId=Long.parseLong(strs.get(0));
            Long orientId=Long.parseLong(strs.get(1));

            List<AdvertPlanPeriodDO> advertPlanPeriodDOS=advertPlanPeriodDAO.selectByAdvertId(advertId,orientId.equals(AdvertConstants.DEFAULT_ORIENTATION_ID)?null:orientId);

            if(CollectionUtils.isEmpty(advertPlanPeriodDOS)){
                return advertPlanPeriodDOS;
            }

            AdvertTagDO advertTagDO = advertTagDao.selectByAdvertId(advertId);

            //全局流量配置只考虑微信加粉
            List<String> promoteUrlTags=advertTagDO.getPromoteUrlTags();
            if(!CollectionUtils.isEmpty(promoteUrlTags) && promoteUrlTags.contains(WEI_XIN_JIA_FEN)){
                List<GlobalConfigFlowVO> globalConfigFlowVOS=globalConfigFlowService.queryAll();

                //获取微信加粉的投放时段配置
                List<GlobalConfigFlowVO> periodConfig=globalConfigFlowVOS.stream().filter(vo-> IS_PERIOD.equals(vo.getConfigItem()) && vo.getConfigConditionValue().containsValue(WEI_XIN_JIA_FEN)).collect(Collectors.toList());

                if(!CollectionUtils.isEmpty(periodConfig)){
                    JSONArray periodArr =periodConfig.get(0).getConfigItemValue();

                    advertPlanPeriodDOS= buildPeriodList(periodArr,advertId,orientId);
                }
            }
            return advertPlanPeriodDOS;
        }
        @Override
        public ListenableFuture<List<AdvertPlanPeriodDO>> reload(String  key, List<AdvertPlanPeriodDO> oldValue) throws Exception {
            ListenableFutureTask<List<AdvertPlanPeriodDO>> task = ListenableFutureTask.create(() -> load(key));
            executorService.submit(task);
            return task;
        }
    });

    //组装投放时段
    public List<AdvertPlanPeriodDO> buildPeriodList(JSONArray pushDate,Long advertId,Long orientationId){
        List<AdvertPlanPeriodDO> periodList = Lists.newArrayList();
        for(Object item:pushDate){
            JSONObject pushDateJson=(JSONObject) item;
            AdvertPlanPeriodDO advertPlanPeriodDto=new AdvertPlanPeriodDO(null,
                    pushDateJson.getString(START_HOUR),
                    pushDateJson.getString(END_HOUR),
                    null,
                    advertId,
                    orientationId,
                    "countCoupon",
                    null
            );
            periodList.add(advertPlanPeriodDto);
        }
        return periodList;
    }
    /**
     * setAdvertTargetAppIds : set广告的是否可投放状态. <br/>
     *
     * @author cdm
     * @since JDK 1.7
     */
    @Override
    public void setAdvertPeriodPut(String key, Boolean isPut) {
        clearYesterdayCache();
        if (isPut != null) {
            advertPeriodPut.put(key, isPut);
        }
    }

    /**
     * getAdvertCache:获取广告缓存对象. <br/>
     *
     * @return AdvertVO 广告对象
     * @author cdm
     * @since JDK 1.7
     */
    @Override
    public Boolean getAdvertPeriodPut(Long periodId,String periodType) {
        String key = getPeriodPutKey(periodId,getByCode(periodType));
        if (!advertPeriodPut.isEmpty() && advertPeriodPut.containsKey(key)) {
            return advertPeriodPut.get(key);
        }
        // 如果没有，说明是新的，默认可投放
        return true;
    }
    
    @Override
    public String getPeriodPutKey(Long periodId, AdvertPkgPeriodTypeEnum typeEnum) {
        //默认 总发券
//        if(typeEnum == null){
//            typeEnum = PERIOD_TYPE_COUNT_COUPON;
//            logger.warn("period type is null  periodId="+periodId);
//        }
        //如果类型为（总预算/总发券量）key = periodId.periodType.2017-07-20
        if(PERIOD_TYPE_COUNT_BUDGET.equals(typeEnum) || PERIOD_TYPE_COUNT_COUPON.equals(typeEnum)){
            return periodId + "." + typeEnum.getCode() + "." + DateUtils.getDayStr(new Date());
        }
        //如果类型为（每小时预算/每小时发券量）key = periodId.periodType.2017-07-20 07。 时间维度"小时"
        return periodId + "." + typeEnum.getCode() + "." + getMinuteTime();
    }

    @Override
    public void addPeriodCountAndEditCache(Long periodId,Long periodValue,String periodType) throws TuiaException {
        //时间
        String curPeriodTime = handleDate(periodType);
        //插入 投放时段发券量 ，得到消耗(发券量/消耗预算)
        Long num = handlePeriodIssueCount(periodId,periodType,curPeriodTime,1L);
        //更新缓存
        Map<String,Object> map=  doCheck(periodId,periodValue,periodType,num,curPeriodTime);
        Boolean isPut = (Boolean) map.get("value");
        boolean oldIsPut = getAdvertPeriodPut(periodId,periodType);
        //如果缓存中已经有当前value  且  缓存中的值 和 要set的值相等
        if(!oldIsPut && isPut || oldIsPut && !isPut){
            //发送消息，更新其他主机缓存
            refreshCacheMqProducer.sendMsgWithId(ConfigRocketMqTags.UPDATE_PERIOD_ISPUT_NEW_MSG, JSON.toJSONString(map));
        }
    }

    @Override
    public String getCurPeriodTime(String periodType) {
        return handleDate(periodType);
    }

    @Override
    public AdvertPlanPeriodDO getPeriodCacheById(Long periodId) {
        if(null != periodId){
            try {
                Optional<AdvertPlanPeriodDO> optional = ADVERT_PERIOD_CACHE.getUnchecked(periodId);
                if(optional.isPresent()){
                    return optional.get();
                }
            } catch (Exception e) {
               logger.error("get period cache error",e);
            }
        }
        return null;
    }

    @Override
    public void invalidPeriodCache(List<Long> periodIds) {
        if(CollectionUtils.isEmpty(periodIds)){
            return;
        }
        // 更新时段是否可投放信息:没有从缓存获取是因为缓存是异步加载，获取的可能是旧值
        for (Long periodId : periodIds) {
            try {
                //刷新广告时段获取cache
                ADVERT_PERIOD_CACHE.refresh(periodId);
                //查询最新时段信息
                AdvertPlanPeriodDO advertPlanPeriodDO = advertPlanPeriodDAO.selectPeriodById(periodId);
                //时段被删除
                if(advertPlanPeriodDO == null)
                    continue;

                String curPeriodTime = handleDate(advertPlanPeriodDO.getPeriodType());
                Map<String,Object> map = this.doCheck(advertPlanPeriodDO.getId(), advertPlanPeriodDO.getPeriodValue(),
                             advertPlanPeriodDO.getPeriodType(), null, curPeriodTime);
                String key = (String) map.get("key");
                Boolean isPut = (Boolean) map.get("value");
                setAdvertPeriodPut(key,isPut);
            } catch (Exception e) {
                logger.error("收到消息重新计算时段可投状态异常", e);
            }
        }
    }


    private Map<String,Object> doCheck(Long periodId,Long periodValue,String periodType,Long issueCount,String curPeriodTime) throws TuiaException {
        boolean isPut = false;
        if(periodValue == null){
            isPut = true;
        }else{
            if(issueCount == null){
                issueCount = doGetIssueCount(periodId,periodType,curPeriodTime);
            }
            //如果 投放时段限制为空  ||  已消耗 < 投放时段限制，则当前时段是可投放的
            if (issueCount < periodValue) {
                isPut = true;
            }
        }
        String key = getPeriodPutKey(periodId,getByCode(periodType));
        Map<String,Object> map = Maps.newHashMapWithExpectedSize(2);
        map.put("key",key);
        map.put("value",isPut);
        return map;
    }

    private Long handlePeriodIssueCount(Long periodId,String periodType,String curPeriodTime,Long expendBudget) throws TuiaException {
        PeriodIssueCountDO pc = new PeriodIssueCountDO();
        pc.setIssueCount(expendBudget);
        pc.setPlanPeriodId(periodId);
        pc.setPeriodType(periodType);
        pc.setCurPeriodTime(curPeriodTime);

        //获取消耗消耗(发券量/消耗预算)
        Long num = doGetIssueCount(periodId,periodType,curPeriodTime);
        if (num == 0L) {
            periodIssueCountDAO.insertPeriodIssueCount(pc);
        }else {
            periodIssueCountDAO.updatePeriodIssueCount(pc);
        }
        return num + pc.getIssueCount();
    }

    /**
     * 获取消耗(发券量/消耗预算)
     * @param periodId
     * @param periodType
     * @param curPeriodTime
     * @return
     * @throws TuiaException
     */
    private Long doGetIssueCount(Long periodId,String periodType,String curPeriodTime) throws TuiaException {
        return periodIssueCountDAO.selectPeriodIssueCount(periodId,periodType,curPeriodTime);
    }

    //返回格式 2017-07-21 10
    public String getMinuteTime(){
        String date = DateUtils.getMinuteStr(new Date());
        return date.substring(0,date.length() - 3);
    }

    private String handleDate(String periodType){
        //类型为(每小时发券量/每小时预算),日期格式 2017-07-21 10
        if(PERIOD_TYPE_HOUR_COUPON.getCode().equals(periodType) || PERIOD_TYPE_HOUR_BUDGET.getCode().equals(periodType)){
            return getMinuteTime();
        }
        //格式为 2017-07-21
        return DateUtils.getDayStr(new Date());
    }


    /**
     * 清除昨日缓存，每6小时执行一次清除,
     */
    private void clearYesterdayCache()  {
        //小时单位
        int hour = 60 * 60 * 1000;
        long curTime = System.currentTimeMillis();
        //（当前时间 - 上一次时间 ）
        int time = (int) ((curTime - perTime) / hour);
        // 如果大于 6个小时，则执行清除昨天的缓存
        if(time >= 6){
            //更新静态变量（上一次更新时间）为当前时间
            perTime = curTime;

            advertPeriodPut.forEach((key,val) -> {
                //如果key 包含昨日的日期如(2017-07-26)，则删除key
                if(key.indexOf(DateUtils.getYesterday()) > -1){
                    advertPeriodPut.remove(key);
                }
            });
        }
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        // 广告是否可投放
        advertPeriodPut = new ConcurrentHashMap<>(10000);
    }

    @Override
    public void initPeriodCache(List<Long> advertIds) {
        // 获取有效广告下配置里的时段信息
        List<AdvertPlanPeriodDO> advertPeriodList = advertPlanPeriodDAO.selectByAdvertIds(advertIds);
        if (CollectionUtils.isEmpty(advertPeriodList)) {
            return;
        }
        // 广告是否可投放
        advertPeriodPut = new ConcurrentHashMap<>(10000);
        // 根据投放时段类型分组,同时对时段投放缓存赋值
        Map<String, List<AdvertPlanPeriodDO>>  periodTypeMap = advertPeriodList.stream().map(dto -> {
            ADVERT_PERIOD_CACHE.put(dto.getId(), Optional.ofNullable(dto));
            return dto;
        }).collect(Collectors.groupingBy(AdvertPlanPeriodDO::getPeriodType));
        // 根据类型，配置id,查询发券/消耗数据
        Map<Long, Long> periodIssueCountMap = periodTypeMap.entrySet().stream().map(map -> {
            // 每种类型查询的时间不同
            String curPeriodTime = handleDate(map.getKey());
            List<Long> periodIds = map.getValue().stream().map(AdvertPlanPeriodDO::getId).collect(Collectors.toList());
            return periodIssueCountDAO.selectPeriodIssueCountList(periodIds, map.getKey(), curPeriodTime);
        }).flatMap(List::stream).filter(dto -> null != dto && null != dto.getIssueCount())
                .collect(Collectors.toMap(PeriodIssueCountDO::getPlanPeriodId, PeriodIssueCountDO::getIssueCount, (v1, v2) -> v2));

        // 判断配置下时段是否能投放
        advertPeriodList.forEach(dto -> {
            boolean isPut = false;
            Long issueCount = periodIssueCountMap.get(dto.getId());
            //  已消耗 < 投放时段限制，则当前时段是可投放的
            if (dto.getPeriodValue() == null ||issueCount == null || issueCount < dto.getPeriodValue().longValue()) {
                isPut = true;
            }
            String key = getPeriodPutKey(dto.getId(),getByCode(dto.getPeriodType()));
            advertPeriodPut.put(key, isPut);
        });
    }

    @Override
    public Map<String, Boolean> getTheAdvertPeriodPut() {

        return advertPeriodPut;
    }

}
