package cn.com.duiba.tuia.service.impl;

import cn.com.duiba.tuia.cache.AdvertMapCacheManager;
import cn.com.duiba.tuia.dao.privilege.AdvertPrivilegeDAO;
import cn.com.duiba.tuia.domain.vo.AdvertVO;
import cn.com.duiba.tuia.service.AdvertPrivilegeService;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2017年05月31日 10:14
 * @descript:
 * @version: 1.0
 */
@Service
public class AdvertPrivilegeServiceImpl implements AdvertPrivilegeService, InitializingBean {

    @Autowired
    private ExecutorService executorService;

    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    /**
     * 特权库缓存
     */
    private final LoadingCache<String, Set<Long>> ADVERT_PRIVILEGE_CACHE = CacheBuilder.newBuilder().initialCapacity(1000).
            recordStats().expireAfterWrite(3, TimeUnit.HOURS).refreshAfterWrite(1,TimeUnit.HOURS).build(new CacheLoader<String, Set<Long>>() {
        @Override
        public Set<Long> load(String key) throws Exception {
            return Sets.newConcurrentHashSet(advertPrivilegeDAO.selectAllPrivileges());
        }

        @Override
        public ListenableFuture<Set<Long>> reload(String key, Set<Long> oldValue) throws Exception {

            ListenableFutureTask<Set<Long>> task = ListenableFutureTask.create(() -> {
                return load(key);
            });

            executorService.submit(task);
            return task;
        }
    });

    private final String CACHE_KEY = "advert_privilege_cache";


    @Autowired
    private AdvertPrivilegeDAO advertPrivilegeDAO;

    @Autowired
    private AdvertMapCacheManager advertMapCacheManager;

    @Override
    public Set<Long> getAdvertPrivileges() {

        try {

            return ADVERT_PRIVILEGE_CACHE.get(CACHE_KEY);
//            return ADVERT_PRIVILEGE_CACHE.get(CACHE_KEY, new Callable<Set<Long>>() {
//                @Override
//                public Set<Long> call() throws Exception {
//                    return Sets.newConcurrentHashSet(advertPrivilegeDAO.selectAllPrivileges());
//                }
//            });
        } catch (ExecutionException e) {
            logger.error("获取特权广告库列表异常", e);
        }

        return Collections.emptySet();
    }

    @Override
    public void updateAdvertPrivileges() {
        ADVERT_PRIVILEGE_CACHE.refresh(CACHE_KEY);
//        this.getAdvertPrivileges();
    }

    @Override
    public Set<Long> getValidPrivileges() {

        List<Long> validAdvertIds = advertMapCacheManager.getValidAdvertIds();
        if(CollectionUtils.isEmpty(validAdvertIds))
            return Collections.emptySet();

        Set<Long> advertPrivileges = this.getAdvertPrivileges();
        Set<Long> validAdverts = Sets.newLinkedHashSet(validAdvertIds);
        Sets.SetView<Long> differences = Sets.intersection(validAdverts,advertPrivileges);
        return differences;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        //在类加载的时候 触发加载缓存
        ADVERT_PRIVILEGE_CACHE.get(CACHE_KEY);
//        this.getAdvertPrivileges();
    }
}
