package cn.com.duiba.tuia.service.impl;

import cn.com.duiba.tuia.cache.BaseCacheService;
import cn.com.duiba.tuia.cache.RefreshCacheMqService;
import cn.com.duiba.tuia.cache.ServiceManager;
import cn.com.duiba.tuia.domain.model.NewAppTest;
import cn.com.duiba.tuia.domain.model.UpdateNewAppTestCacheMsg;
import cn.com.duiba.tuia.exception.TuiaException;
import cn.com.duiba.tuia.service.AdvertRealDataService;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.tuia.advert.cache.CacheKeyTool;
import cn.com.tuia.advert.cache.RedisCommonKeys;
import cn.com.tuia.advert.cache.RedisPreKey;
import cn.com.tuia.advert.constants.AdvertSupportConstant;
import cn.com.tuia.advert.constants.CommonConstant;
import cn.com.tuia.advert.constants.SystemConfigKeyConstant;
import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.HashOperations;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 *  * ClassName:AdvertRealDataServiceImpl <br/>
 *  * Date:     2017年6月21日 下午7:31:51 <br/>
 *  * @author   zp
 *  * @version 
 *  * @since    JDK 1.6
 *  * @see     
 *  
 */
@Service
public class AdvertRealDataServiceImpl  extends BaseCacheService implements AdvertRealDataService, InitializingBean {
    private static Logger log = LoggerFactory.getLogger(AdvertRealDataServiceImpl.class);

    private HashOperations<String, String, Object> hashOperations;
    @Resource
    private StringRedisTemplate stringRedisTemplate;
    @Resource
    private ExecutorService executorService;
    
    @Autowired
    private ServiceManager serviceManager;

    @Autowired
    private RefreshCacheMqService refreshCacheMqService;

    // 广告新媒体试投的单广告发券上限
    public static final long NEW_APP_ADVERT_LUNCH_LMIT = 3000L; 
    // 广告新媒体试投的单广告点击上限
    public static final long NEW_APP_ADVERT_CLICK_LMIT = 1000L;

    /**
     * 增加曝光日志
     * @param agentId   代理商id
     * @param accountId 帐号id
     * @param advertId  广告id
     * @param advertPlanId
     */
    @Override
    public void incrExposureAdvert(Long agentId, Long accountId, Long advertId, Long advertPlanId, Integer effectiveMainType) {
        try {
            String agentKey = getAgentKey(agentId);
            String agentMainTypeKey = getAgentMainTypeKey(agentId, effectiveMainType);
            String hashKey = getAccountHashKey(accountId, advertId,  advertPlanId,RedisPreKey.KEY_EXPOSURE);
            Long result = hashOperations.increment(agentKey, hashKey, 1L);
            hashOperations.increment(agentMainTypeKey, hashKey, 1L);
            if (result == 1L) {
                this.expire(agentKey);
                this.expire(agentMainTypeKey);
            }
        } catch (Exception e) {
            log.error("incrExposureAdvert error", e);
        }
    }

    /**
     * 增加点击日志
     * @param agentId   代理商id
     * @param accountId 帐号id
     * @param advertId  广告id
     */
    @Override
    public void incrClickAdvert(Long agentId, Long accountId, Long advertId, Long advertPlanId ,Integer effectiveMainType, NewAppTest newAppTest) {
        try {
            String agentKey = getAgentKey(agentId);
            String agentMainTypeKey = getAgentMainTypeKey(agentId, effectiveMainType);
            String hashKey = getAccountHashKey(accountId, advertId, advertPlanId, RedisPreKey.KEY_CLICK);
            hashOperations.increment(agentKey, hashKey, 1L);
            hashOperations.increment(agentMainTypeKey, hashKey, 1L);
        } catch (Exception e) {
            log.error("incrClickAdvert error", e);
        }
    }

    /**
     * 
     * addNewAppNotTestAdvert:(新媒体不能测试的广告). <br/>
     *
     * @author chencheng
     * @param appId
     * @param advertId
     * @since JDK 1.8
     */
    private void addNewAppNotTestAdvert(Long appId, Long advertId) {
        try {
            String appKey = CacheKeyTool.getCacheKey(RedisPreKey.KEY_NEW_APP_CANNOT_TEST_ADVERT, appId);
            stringRedisTemplate.opsForSet().add(appKey, String.valueOf(advertId));
            stringRedisTemplate.expire(appKey, getTodaySeconds(), TimeUnit.SECONDS);
        } catch (Exception e) {
            log.error("addNewAppNotTestAdvert error, appId=[{}], advertId=[{}]", appId, advertId);
        }
    }
    
    private long getTodaySeconds() {
        return (long)DateUtils.getToTomorrowSeconds() + (long)new Random().nextInt(1800);
    }


    /**
     * 增加发券
     * @param agentId   代理商id
     * @param accountId 帐号id
     * @param advertId  广告id
     */
    @Override
    public void incrLaunchAdvert(Long agentId, Long accountId, Long advertId,Long advertPlanId, Integer effectiveMainType,NewAppTest newAppTest, Long appId, Integer supportStatus) {
        try {
            executorService.submit(new Runnable() {
                @Override
                public void run() {
                    String agentKey = getAgentKey(agentId);
                    String agentMainTypeKey = getAgentMainTypeKey(agentId, effectiveMainType);
                    String hashKey = getAccountHashKey(accountId, advertId, advertPlanId, RedisPreKey.KEY_LAUNCH);
                    hashOperations.increment(agentKey, hashKey, 1L);
                    hashOperations.increment(agentMainTypeKey, hashKey, 1L);

                    //按天存储的媒体的发券数
                    String appDailyLaunchKey = getAppDailyLaunchKey(appId);
                    hashOperations.increment(appDailyLaunchKey, DateUtils.getDayStr(new Date()), 1L);
                    if (Optional.ofNullable(supportStatus).orElse(0) == CommonConstant.YES) {
                        //按天存储的媒体扶持并且被加权发券数
                        String appSupportDailyLaunchKey = getAppSupportDailyLaunchKey(advertId, appId);
                        hashOperations.increment(appSupportDailyLaunchKey, DateUtils.getDayStr(new Date()), 1L);
                        //广告主总的扶持并且被加权的发券数(给管理后台离线统计使用)
                        String accountSupportLaunchKey = getAccountSupportLaunchKey(accountId);
                        hashOperations.increment(accountSupportLaunchKey, AdvertSupportConstant.ACCOUNT_SUPPORT_LAUNCH_HASH_KEY, 1L);
                    }

                    if (newAppTest != null && newAppTest.getNewAppType()==1) {
                        Long newAppLunchLmit = getNewAppLunchLmit();

                        // 当日单广告发券
                        Long newAppDayLunch= incrNewAppDayLaunch(newAppTest, advertId);
                        // 当日不可发券的广告
                        if (newAppDayLunch.longValue() >= newAppLunchLmit) {
                            addNewAppNotTestAdvert(newAppTest.getAppId(), advertId);
                            refreshCacheMqService.updateNewAppTestCacheMsg(newAppTest.getAppId(), newAppTest.getAdvertId(), UpdateNewAppTestCacheMsg.TYPE_NEW_APP_NOT_TEST_ADVERT_CACHE);
                        }

                    }
                }


            });
        } catch (Exception e) {
            log.error("incrLaunchAdvert", e);
        }
    }

    private String getAccountSupportLaunchKey(Long accountId) {
        return CacheKeyTool.getCacheKey(RedisCommonKeys.KC135, AdvertSupportConstant.ACCOUNT_SUPPORT_WEIGHT_LAUNCH, accountId);
    }

    private String getAppSupportDailyLaunchKey(Long advertId, Long appId) {
        return CacheKeyTool.getCacheKey(RedisCommonKeys.KC135, AdvertSupportConstant.APP_SUPPORT_WEIGHT_LAUNCH, advertId, appId);
    }

    private String getAppDailyLaunchKey(Long appId) {
        return CacheKeyTool.getCacheKey(RedisCommonKeys.KC135, AdvertSupportConstant.APP_LAUNCH_COUNT, appId);
    }

    @Override
    public boolean appSupportLaunchCheck(Long advertId, Long appId) {
        try {
            String appDailyLaunchKey = this.getAppDailyLaunchKey(appId);
            String appSupportDailyLaunchKey = this.getAppSupportDailyLaunchKey(advertId, appId);
            String hashKey = DateUtils.getDayStr(new Date());
            //媒体每日总的发券数
            BigDecimal appDailyLaunchCount = parseObjectToBigDecimal(this.getAppDailyLaunchCount(appDailyLaunchKey, hashKey));
            //媒体上每日的扶持发券数
            BigDecimal appSupportDailyLaunchCount = parseObjectToBigDecimal(this.getAppSupportDailyLaunchCount(appSupportDailyLaunchKey, hashKey));
            if (appDailyLaunchCount.equals(BigDecimal.ZERO) || appSupportDailyLaunchCount.equals(BigDecimal.ZERO)) {
                return true;
            }
            //如果在该媒体上扶持的发券数超过了媒体上总的发券数的5%,则在该媒体上不予加权了
            return appSupportDailyLaunchCount.divide(appDailyLaunchCount, 2, RoundingMode.HALF_UP).compareTo(AdvertSupportConstant.APP_LAUNCH_LIMIT_RATIO) < 0;
        } catch (Exception e) {
            logger.error("appSupportLaunchCheck exception, appId:{}", appId, e);
            return false;
        }
    }

    private BigDecimal parseObjectToBigDecimal(Object object) {
        if (object == null) {
            return BigDecimal.ZERO;
        }
        return new BigDecimal(object.toString());
    }

    /**
     * 
     * getNewAppLunchLmit:(广告在新媒体发券上限). <br/>
     *
     * @author chencheng
     * @return
     * @since JDK 1.8
     */
    private Long getNewAppLunchLmit() {
        
        try {
            String newAppLunchLmitStr = serviceManager.getStrValue(SystemConfigKeyConstant.TUIA_NEW_APP_LAUNCH_MAX);
            return Optional.ofNullable(newAppLunchLmitStr).map(a -> Long.parseLong(a)).orElse(NEW_APP_ADVERT_LUNCH_LMIT);
        } catch (TuiaException e) {
           return NEW_APP_ADVERT_LUNCH_LMIT;
        }
        
    }
    /**
     * 
     * incrNewApp10DayLaunch:(新媒体当日点击). <br/>
     *
     * @author chencheng
     * @param newAppTest
     * @param advertId 
     * @return
     * @since JDK 1.8
     */
    private Long incrNewAppDayClick(NewAppTest newAppTest, Long advertId) {
        try {
            String appKey = CacheKeyTool.getCacheKey(RedisPreKey.KEY_NEW_APP_DAY_CLICK, newAppTest.getAppId(), DateUtils.getDayStr(new Date()));
            return incrementResult(appKey, String.valueOf(advertId), 1L);
        } catch (Exception e) {
            log.error("incrNewAppDayClick error", e);
            return 0L;
        }
    }

    
    /**
     * 
     * incrNewApp10DayLaunch:(新媒体当日发券). <br/>
     *
     * @author chencheng
     * @param newAppTest
     * @param advertId 
     * @return
     * @since JDK 1.8
     */
    private Long incrNewAppDayLaunch(NewAppTest newAppTest, Long advertId) {
        try {
            String appKey = CacheKeyTool.getCacheKey(RedisPreKey.KEY_NEW_APP_DAY_LAUNCH, newAppTest.getAppId(), DateUtils.getDayStr(new Date()));
            return incrementResult(appKey, String.valueOf(advertId), 1L);
        } catch (Exception e) {
            log.error("incrNewAppDayLaunch error", e);
            return 0L;
        }
    }


    /**
     * 累加每个广告扣费
     * incrTuiaAdvertConsumeFee:(这里用一句话描述这个方法的作用). <br/>
     * @author zp
     * @param advertId
     * @param fee
     * @return
     * @since JDK 1.6
     */
    @Override
    public Long incrTuiaAdvertConsumeFee(Long advertId,Long fee){
        try{
           String key = getKey(RedisPreKey.KEY_ADVERT_CONSUME_FEE);
           Long total = hashOperations.increment(key,String.valueOf(advertId), fee);
           if(total.equals(fee)){
               this.expire(key);
           }
           return total;
        }catch(Exception e){
           log.error("incrTuiaAdvertConsumeFee", e);
           return 0L;
        }
    }

    @Override
    public Long incrTuiaAdvertPlanConsumeFee(Long advertPlanId, Long fee) {
        try{
            String key = getKey(RedisPreKey.KEY_ADVERT_PLAN_CONSUME_FEE);
            Long total = hashOperations.increment(key,String.valueOf(advertPlanId), fee);
            if(total.equals(fee)){
                this.expire(key);
            }
            return total;
        }catch(Exception e){
            log.error("incrTuiaAdvertPlanConsumeFee", e);
            return 0L;
        }
    }

    /**
     * 累加每个广告主扣费
     * incrTuiaAdvertConsumeFee:(这里用一句话描述这个方法的作用). <br/>
     * @author zp
     * @param accountId
     * @param fee
     * @return
     * @since JDK 1.6
     */
    @Override
    public Long incrTuiaAccountConsumeFee(Long accountId,Long fee){
        try{
           String key = getKey(RedisPreKey.KEY_ACCOUNT_CONSUME_FEE);
           Long total = hashOperations.increment(key,String.valueOf(accountId), fee);
           if(total.equals(fee)){
               this.expire(key);
           }
           return total;
        }catch(Exception e){
           log.error("incrTuiaAccountConsumeFee", e);
           return 0L;
        }
    }

    @Override
    public Long incrTuiaAccountBalance(Long accountId, Long fee) {
        try{
            String key = getKey(RedisPreKey.KEY_ACCOUNT_BALANCE);
            Long total = hashOperations.increment(key,String.valueOf(accountId), fee);
            if(total.equals(fee)){
                this.expire(key);
            }
            return total;
        }catch(Exception e){
            log.error("incrTuiaAccountBalance", e);
            return 0L;
        }
    }

    private void expire(String advertKey) {
        //今晚过期
        stringRedisTemplate.expire(advertKey,
                (long)DateUtils.getToTomorrowSeconds() + (long)new Random().nextInt(1800), TimeUnit.SECONDS);
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        hashOperations = stringRedisTemplate.opsForHash();
    }
    
    private String getAgentKey(Long agentId) {
        return CacheKeyTool.getCacheKey(agentId, DateUtils.getDayStr(new Date()));
    }

    private String getAgentMainTypeKey(Long agentId, Integer effectiveMainType) {
        return CacheKeyTool.getCacheKey(agentId, effectiveMainType, DateUtils.getDayStr(new Date()));
    }

    private String getKey(String keyId) {
        return CacheKeyTool.getCacheKey(keyId, DateUtils.getDayStr(new Date()));
    }

    private String getAccountHashKey(Long accountId, Long advertId, Long advertPlanId, String key) {
        return accountId + "_" + advertId + "_" + advertPlanId + "_" +key;
    }

    @Override
    public Long getAdvertLaunch(Long agentId, Long accountId, Long advertId,Long advertPlanId) {
        
        // 获取发券次数
        return getRedisHashValue(getAgentKey(agentId), getAccountHashKey(accountId, advertId, advertPlanId, RedisPreKey.KEY_LAUNCH));
    }


    /**
     * 
     * get1024Key:(redis对key分区). <br/>
     *
     * @author chencheng
     * @return
     * @since JDK 1.8
     */
    private String get1024KeyNotDate(Long key, String preKey) {
        Long code = key % 1024;
        return CacheKeyTool.getCacheKey(preKey, code);
    }

    
    /**
     * 
     * get1024Key:(redis对key分区). <br/>
     *
     * @author chencheng
     * @return
     * @since JDK 1.8
     */
    private String get1024Key(Long key, String preKey) {
        Long code = key % 1024;
        return CacheKeyTool.getCacheKey(preKey, code, DateUtils.getDayStr(new Date()));
    }

    /**
     * 
     * incrementResult:(存储hash值). <br/>
     *
     * @author chencheng
     * @param key
     * @param hashKey
     * @param value
     * @return
     * @since JDK 1.8
     */
    private Long incrementResult(String key, String hashKey, Long value) {
        Long result = hashOperations.increment(key, hashKey, value);
        if (result.equals(value)) {
            this.expire(key);
        }
        return result;
    }

    /**
     * 
     * getRedisHashValue:(根据hash可key获取值). <br/>
     *
     * @author chencheng
     * @param key
     * @param hashKey
     * @return
     * @since JDK 1.8
     */
    private Long getRedisHashValue(String key, String hashKey) {
        try {
            Object value = hashOperations.get(key, hashKey);
            if (null == value) {
                return 0L;
            }
            return Long.parseLong(value.toString());
        } catch (Exception e) {
            log.error("hashOperations get hash error, key=[{}], hashKey=[{}],", key, hashKey, e);
            return 0L;
        }
    }

    @Override
    public List<Long> getNewAppDayLaunch(Long appId) {
        try {
            // 获取媒体已发券广告
            String appKey = CacheKeyTool.getCacheKey(RedisPreKey.KEY_NEW_APP_DAY_LAUNCH, appId, DateUtils.getDayStr(new Date()));
            Set<String> setKyes = hashOperations.keys(appKey);
            if (setKyes.isEmpty()) {
                return Lists.newArrayList();
            }
            return setKyes.stream().map(key -> Long.parseLong(key)).collect(Collectors.toList());
        } catch (Exception e) {
            log.error("getNewAppDayLaunch error, appId=[{}]", appId, e);
            return Lists.newArrayList();
        }
    }

    @Override
    public Object getAppDailyLaunchCount(String appDailyLaunchKey, String hashKey) {
        try {
            return hashOperations.get(appDailyLaunchKey, hashKey);
        } catch (Exception e) {
            log.error("getAppDailyLaunchCount get exception", e);
            return null;
        }
    }

    @Override
    public Object getAppSupportDailyLaunchCount(String appSupportDailyLaunchKey, String hashKey) {
        try {
            return hashOperations.get(appSupportDailyLaunchKey, hashKey);
        } catch (Exception e) {
            log.error("getAppSupportDailyLaunchCount get exception", e);
            return null;
        }
    }
}