package cn.com.duiba.tuia.service.impl;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import cn.com.duiba.bigdata.dmp.service.api.remoteservice.dto.AqyAttributeDto;
import cn.com.duiba.bigdata.dmp.service.api.remoteservice.remoteservice.RemoteDMPService;
import cn.com.duiba.boot.profiler.DBTimeProfiler;
import cn.com.duiba.tuia.constants.ABTestConstant;
import cn.com.duiba.tuia.enums.RandomServiceEnum;
import cn.com.duiba.tuia.service.AdvertSystemConfigService.AdvertSystemConfigEnum;
import cn.com.duiba.tuia.service.RandomService;
import com.netflix.hystrix.contrib.javanica.annotation.HystrixCommand;
import com.netflix.hystrix.contrib.javanica.annotation.HystrixProperty;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.dubbo.config.annotation.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.HashOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import com.alibaba.fastjson.JSON;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;

import cn.com.duiba.tuia.cache.AdvertMapCacheManager;
import cn.com.duiba.tuia.cache.ServiceManager;
import cn.com.duiba.tuia.domain.dataobject.ConsumerInteractiveRecordDO;
import cn.com.duiba.tuia.domain.model.FilterResult;
import cn.com.duiba.tuia.domain.vo.AdvertVO;
import cn.com.duiba.tuia.domain.vo.ConsumerRecordJsonVO;
import cn.com.duiba.tuia.domain.vo.ConsumerRecordVO;
import cn.com.duiba.tuia.exception.TuiaException;
import cn.com.duiba.tuia.service.ConsumerRecordSerivce;
import cn.com.duiba.tuia.service.ConsumerService;
import cn.com.duiba.tuia.utils.DateUtil;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.tuia.advert.cache.RedisCommonKeys;
import cn.com.tuia.advert.constants.CommonConstant;
import cn.com.tuia.advert.constants.SystemConfigKeyConstant;

/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2017年05月02日 19:17
 * @descript:
 * @version: 1.0
 */
@Service
public class ConsumerServiceImpl implements ConsumerService,InitializingBean {
    
    private static Logger logger = LoggerFactory.getLogger(ConsumerServiceImpl.class);

    @Autowired
    private ConsumerRecordSerivce consumerRecordSerivce;

    @Resource
    protected RedisTemplate<String,Integer> redisTemplate;
    @Autowired
    private AdvertMapCacheManager                 advertMapCacheManager;

    @Resource
    protected StringRedisTemplate stringRedisTemplate;
    
    @Resource
    private ExecutorService executorService;
    
    /**
     * key:consumerId HK:advertId value:isClicked
     */
    private HashOperations<String,Long,Integer> hashOperations;

    @Autowired
    private ServiceManager serviceManager;

    @Autowired
    private RandomService randomService;
    @Reference
    private RemoteDMPService remoteDMPService;

    @Override
    public void flagAdvertClicked(Long consumerId, Long advertId) {
        try{
            String key = this.formatConsumerKey(consumerId);
            hashOperations.put(key,advertId, CommonConstant.YES);
            long expireTime = (long) DateUtils.getToTomorrowSeconds() + (long) new Random().nextInt(7200);
            redisTemplate.expire(key, expireTime, TimeUnit.SECONDS);
        }catch(Exception e){
            logger.error("flagAdvertClicked error",e);
        }

    }

    @Override
    public Map<Long, Integer> getAllClickStatus(Long consumerId) {
        //hard work:当filed越多时性能会越差
        return hashOperations.entries(this.formatConsumerKey(consumerId));
    }


    @Override
    public void afterPropertiesSet() throws Exception {
        hashOperations = redisTemplate.opsForHash();
    }

    /**
     * 获取用户key:K39_${consumerId}_${date}
     * @param consumerId
     * @return
     */
    @Override
    public String formatConsumerKey(Long consumerId){

        StringBuffer fieldBuffer = new StringBuffer(RedisCommonKeys.KC119.toString());
        fieldBuffer.append("_").append(consumerId).append("_").append(DateUtils.getDayStr(new Date()));
        return fieldBuffer.toString();
    }

    @Override
    public Map<Long,Integer> getConsumerLimit(List<ConsumerInteractiveRecordDO> consumerVOS){

        Map<Long,Integer> limitAdvertMap = Maps.newConcurrentMap();
        for (ConsumerInteractiveRecordDO consumerVO : consumerVOS){

            Integer limit = limitAdvertMap.get(consumerVO.getAdvertId());
            limitAdvertMap.put(consumerVO.getAdvertId(),limit == null ? 1 : ++limit);
        }

        return limitAdvertMap;
    }


    @Override
    public List<ConsumerInteractiveRecordDO> getTodayConsumerList(List<ConsumerInteractiveRecordDO> consumerVOS) {
        if (CollectionUtils.isEmpty(consumerVOS)) {
            return Lists.newArrayList();
        }    
        return consumerVOS.stream().filter(dto -> DateUtils.getDayDate(new Date()).equals(dto.getCurDate())).collect(Collectors.toList());
    }

    @Override
    public List<Long> getLimitAccountIds(List<ConsumerInteractiveRecordDO> todayConsumeVoList) throws TuiaException {
        if (CollectionUtils.isEmpty(todayConsumeVoList)) {
            return Lists.newArrayList();
        }
        // 查询广告对应的广告主id
        List<ConsumerInteractiveRecordDO> todayConsumeList = todayConsumeVoList.stream().map(consumer -> {
            if(null == consumer.getAccountId()){
                AdvertVO advertVO = advertMapCacheManager.getAdvertCache(consumer.getAdvertId());
                Optional.ofNullable(advertVO).ifPresent(advert -> 
                Optional.ofNullable(advert.getAdvertPlan()).ifPresent(advertPlan -> consumer.setAccountId(advertPlan.getAccountId())));
            }
            return consumer;
        }).filter(consumer -> consumer.getAccountId() != null).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(todayConsumeList)) {
            return Lists.newArrayList();
        }
        // 广告主对应的发券数量
        Map<Long, Long> accountCountingMap = todayConsumeList.stream().collect(Collectors.groupingBy(ConsumerInteractiveRecordDO::getAccountId, Collectors.counting()));
        List<Long> limitAccountIds = Lists.newArrayList();
        // 达到单用户广告主发券上限设置
        accountCountingMap.forEach((k,v) -> {
            if (v.intValue() >= AdvertSystemConfigEnum.advertiserLaunchLimit.getIntValue()) {
                limitAccountIds.add(k);
            }
        });
        // 没有超过发券间隔。（领券记录倒序后，前N个广告对应的广告主不能投放）
        limitAccountIds.addAll(todayConsumeList.stream().limit(AdvertSystemConfigEnum.advertiserLaunchInterval.getIntValue()).map(ConsumerInteractiveRecordDO::getAccountId).collect(Collectors.toList()));
        return limitAccountIds.stream().distinct().collect(Collectors.toList());
    }
    

    /**
     * 广告券发券限制用户比例缓存，后期会删除
     */
    private final LoadingCache<String, Integer> percentTestCache = CacheBuilder.newBuilder().initialCapacity(1).
            recordStats().refreshAfterWrite(30, TimeUnit.MINUTES).expireAfterWrite(2, TimeUnit.HOURS).build(new CacheLoader<String, Integer>() {
        @Override
        public Integer load(String key) throws Exception {
            return getPercentTestFromRedis();
        }

        @Override
        public ListenableFuture<Integer> reload(String key, Integer oldValue) throws Exception {
            ListenableFutureTask<Integer> task = ListenableFutureTask.create(() -> load(key));
            executorService.submit(task);
            return task;
        }

    });
    

    private Integer getPercentTestFromRedis() {
        try {
            String value = stringRedisTemplate.opsForValue().get("TUIA_LIMIT_ACCOUNT_PERCENT");
            if (StringUtils.isEmpty(value)) {
                return 20;
            }
            return Integer.parseInt(value);
        } catch (Exception e) {
            logger.info("get TUIA_LIMIT_ACCOUNT_PERCENT error");
           return 20;
        }
    }

    
    @Override
    public int getPercentTest() {
        try {
            return percentTestCache.get("test");
        } catch (ExecutionException e) {
            logger.info("get TUIA_LIMIT_ACCOUNT_PERCENT error", e);
            return 20;
        }
    }

    private List<ConsumerInteractiveRecordDO> getHoursConsumerList(List<ConsumerInteractiveRecordDO> consumerVOList) throws TuiaException {
        if (CollectionUtils.isEmpty(consumerVOList)) {
            return Lists.newArrayList();
        } 
        // 获取时间
        int beforeHours = serviceManager.getIntValue(SystemConfigKeyConstant.TUIA_ADVERT_SINGLE_REPEAT_HOURS);
        Date hourBeforeDate = DateUtil.changeByHour(new Date(), -beforeHours);
        // 过滤小时内的领券记录
        return consumerVOList.stream().filter(dto -> dto.getGmtCreate().after(hourBeforeDate)).collect(Collectors.toList());
    }

    /**
     * [重复发券测试需求]过滤指定小时内的领券记录
     *
     * @param interval 重复发券测试组的小时间隔
     * @param consumerVOList 领券记录
     * @return 过滤后的领券记录
     */
    private List<ConsumerInteractiveRecordDO> getRepeatTestConsumerList(Integer interval, List<ConsumerInteractiveRecordDO> consumerVOList) {
        if (CollectionUtils.isEmpty(consumerVOList)) {
            return Lists.newArrayList();
        }
        // 获取时间
        Date hourBeforeDate = DateUtil.changeByHour(new Date(), -interval);
        // 过滤小时内的领券记录
        return consumerVOList.stream().filter(dto -> dto.getGmtCreate().after(hourBeforeDate)).collect(Collectors.toList());
    }

    @Override
    public void setFilterResultConsumeList(FilterResult filterResult, List<ConsumerInteractiveRecordDO> consumerVOList) throws TuiaException {
        // 今日用户领券记录
        List<ConsumerInteractiveRecordDO> todayConsumeList = getTodayConsumerList(consumerVOList);
        filterResult.setTodayConsumeList(todayConsumeList);

        // N小时内用户领券记录
        List<ConsumerInteractiveRecordDO> hoursConsumeList = getHoursConsumerList(consumerVOList);
        List<ConsumerRecordVO> hoursConsumeVOList = hoursConsumeList.stream().map(record -> {
            ConsumerRecordVO recordVO = BeanUtils.copy(record, ConsumerRecordVO.class);
            if (StringUtils.isNotBlank(record.getJson())) {
                recordVO.setConsumerRecordJsonVO(JSON.parseObject(record.getJson(), ConsumerRecordJsonVO.class));
            }
            return recordVO;
        }).collect(Collectors.toList());
        filterResult.setHoursConsumeList(hoursConsumeVOList);

        // [重复发券测试需求]N小时内用户领券记录
        if (null != filterResult.getRepeatTestGroupInterval()) {
            List<ConsumerInteractiveRecordDO> repeatTestConsumeList =
                    getRepeatTestConsumerList(filterResult.getRepeatTestGroupInterval(), consumerVOList);
            filterResult.setRepeatTestConsumeList(repeatTestConsumeList);
        }

        if (CollectionUtils.isEmpty(todayConsumeList)) {
            Long randomServiceTag = randomService.makeRandomServiceTag();
            filterResult.setRandomServiceTag(randomServiceTag);
            Boolean randomServce = randomService.decodeRandomTag(randomServiceTag,RandomServiceEnum.REPEAT_EXPOSURE_RANDOM_SERVICE);
            filterResult.setRepeatExposureLuckyConsumer(randomServce);
            return;
        }
        // 今日最后一条领券记录。N小时的最后一条不一定是今日的最后一条
        ConsumerInteractiveRecordDO lastRecord = todayConsumeList.get(0);
        ConsumerRecordVO lastRecordVO = BeanUtils.copy(lastRecord, ConsumerRecordVO.class);
        filterResult.setLastOfTodayConsumeRecord(lastRecordVO);
        if (StringUtils.isBlank(lastRecord.getJson())) {
            return;
        }

        ConsumerRecordJsonVO consumerRecordJsonVO = JSON.parseObject(lastRecord.getJson(), ConsumerRecordJsonVO.class);

        //随机服务标记 此处用了 long 后续可扩展(此处为兼容)
        Long randomServiceTag = consumerRecordJsonVO.getRandomServiceTag();
        if(null == randomServiceTag){
            randomServiceTag = randomService.makeRandomServiceTag();
        }
        filterResult.setRandomServiceTag(randomServiceTag);
        //解析 切量
        Boolean randomServce = randomService.decodeRandomTag(randomServiceTag,RandomServiceEnum.REPEAT_EXPOSURE_RANDOM_SERVICE);
        filterResult.setRepeatExposureLuckyConsumer(randomServce);

        lastRecordVO.setConsumerRecordJsonVO(consumerRecordJsonVO);
    }

    @Override
    public AqyAttributeDto getAqyAttribute(String deviceId) {
        try {
            if (deviceId == null || ABTestConstant.DEVICEID_NULL.equals(deviceId)) {
                return null;
            }
            return remoteDMPService.getAqyAttribute(deviceId);
        } catch (Exception e) {
            logger.error("remoteDMPService.getAqyAttribute is error", e);
        }
        return null;
    }

    @Override
    @DBTimeProfiler
    public String getAqySexFeature(String deviceId) {
        try {
            if (deviceId == null || ABTestConstant.DEVICEID_NULL.equals(deviceId)) {
                return null;
            }
            return remoteDMPService.getAqySex(deviceId);
        } catch (Exception e) {
            logger.error("remoteDMPService.getAqySex is error", e);
        }
        return null;
    }

}
