package cn.com.duiba.tuia.service.impl;

import cn.com.duiba.tuia.cache.AdvertABTestCacheService;
import cn.com.duiba.tuia.constants.AdvertReqLogExtKeyConstant;
import cn.com.duiba.tuia.dao.abtest.MaterialPromoteBindABTestGroupDAO;
import cn.com.duiba.tuia.dao.promotetest.AdvertPromoteTestDAO;
import cn.com.duiba.tuia.domain.dataobject.*;
import cn.com.duiba.tuia.domain.enums.ABTestSkipEnum;
import cn.com.duiba.tuia.domain.model.FilterResult;
import cn.com.duiba.tuia.service.DomainTestService;
import cn.com.duiba.tuia.service.IQiyiService;
import cn.com.tuia.advert.model.ObtainAdvertReq;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 域测试服务接口实现
 *
 * @author zhangbaiqiang
 * @date 2021/3/29
 */
@Service
public class DomainTestServiceImpl implements DomainTestService {

    private static final Logger logger = LoggerFactory.getLogger(DomainTestServiceImpl.class);

    @Autowired
    private MaterialPromoteBindABTestGroupDAO materialPromoteBindABTestGroupDAO;

    @Autowired
    private AdvertPromoteTestDAO advertPromoteTestDAO;

    @Autowired
    private AdvertABTestCacheService advertABTestCacheService;

    @Autowired
    private IQiyiService iQiyiService;

    @Resource
    private ExecutorService executorService;


    private final LoadingCache<Long, List<String>> PROMOTE_URL_CACHE = CacheBuilder.newBuilder()
            .maximumSize(100)
            .refreshAfterWrite(10, TimeUnit.SECONDS)
            .expireAfterWrite(10, TimeUnit.MINUTES)
            .build(new CacheLoader<Long, List<String>>() {

                @Override
                public List<String> load(Long key)  {
                    return selectGroupPromoteUrl(key);
                }

                @Override
                public ListenableFuture<List<String>> reload(final Long key, List<String> oldValue) {
                    ListenableFutureTask<List<String>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });

    @Override
    public List<String> selectGroupPromoteUrlCache(Long planId) {
        if (null == planId) {
            return Collections.emptyList();
        }

        try {
            return PROMOTE_URL_CACHE.get(planId);
        } catch (ExecutionException e) {
            logger.warn("selectGroupPromoteUrlCache error, planId={}", planId);
        }
        return Collections.emptyList();
    }

    @Override
    public boolean isPlanValid(Long planId, Long advertId, ObtainAdvertReq req, FilterResult filterResult) {
        List<String> promoteUrls = selectGroupPromoteUrlCache(planId);
        return isPromoteUrlsValid(promoteUrls, advertId, req, filterResult);
    }

    /**
     * 校验所有的落地页链接是否有效
     *
     * @param promoteUrls 落地页列表
     * @param advertId 广告Id
     * @param req 广告请求参数
     * @param filterResult 过滤结果
     * @return 是否全部有效
     */
    private boolean isPromoteUrlsValid(List<String> promoteUrls, Long advertId, ObtainAdvertReq req,
                                       FilterResult filterResult) {
        if (CollectionUtils.isEmpty(promoteUrls)) {
            return false;
        }

        //媒体关键词过滤
        if (filterMaterialLandingPage(promoteUrls, filterResult, advertId, req)) {
            abtestSkipReasonLog(req, ABTestSkipEnum.AB3004);
            return false;
        }

        //爱奇艺特殊判断
        if (iQiyiService.isIQiyiFlow(req)) {
            Map<String, Boolean> filterMap = advertABTestCacheService.getLandPageRepoInfoCache(advertId);
            for (String url : promoteUrls) {
                if (!filterMap.getOrDefault(url, false)) {
                    abtestSkipReasonLog(req, ABTestSkipEnum.AB3005);
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * 过滤测试落地页
     * @param urls
     * @param filterResult
     * @return true 屏蔽 false 不屏蔽
     */
    private boolean filterMaterialLandingPage(List<String> urls, FilterResult filterResult, Long advertId, ObtainAdvertReq req) {
        Set<String> copyNewUrls = Sets.newHashSet(urls);

        //美团adx 不需要过滤
        if (null != req.getAdxMediaType() && req.getAdxMediaType() == 1) {
            return false;
        }

        AdvertFilterKeywordDO advertFilterKeywordDO = filterResult.getAdvertFilterKeywordDO();
        if (Objects.nonNull(advertFilterKeywordDO)) {

            NewTradeFilterKeywordDO newTradeFilterKeywordDO = advertFilterKeywordDO.getNewTradeFilterKeywordDO();
            if (Objects.nonNull(newTradeFilterKeywordDO) && MapUtils.isNotEmpty(newTradeFilterKeywordDO.getShieldMaterialLandingPageList())) {
                Map<Long, Set<String>> advertShieldList = newTradeFilterKeywordDO.getShieldMaterialLandingPageList();
                if (advertShieldList.containsKey(advertId)) {
                    Set<String> urlShieldList = advertShieldList.get(advertId);
                    if (org.apache.commons.collections.CollectionUtils.isNotEmpty(urlShieldList)) {
                        for (String url : copyNewUrls) {
                            if (urlShieldList.contains(url)) {
                                return true;
                            }
                        }
                    }
                }
            }

            GeneralFilterKeywordDO generalFilterKeywordDO = advertFilterKeywordDO.getGeneralFilterKeywordDO();
            if (Objects.nonNull(generalFilterKeywordDO) && MapUtils.isNotEmpty(generalFilterKeywordDO.getShieldMaterialLandingPageList())) {
                Map<Long, Set<String>> advertShieldList = generalFilterKeywordDO.getShieldMaterialLandingPageList();
                if (advertShieldList.containsKey(advertId)) {
                    Set<String> urlShieldList = advertShieldList.get(advertId);
                    if (org.apache.commons.collections.CollectionUtils.isNotEmpty(urlShieldList)) {
                        for (String url : copyNewUrls) {
                            if (urlShieldList.contains(url)) {
                                return true;
                            }
                        }
                    }
                }
            }
        }

        return false;
    }

    /**
     * 查询域测试的实验组的落地页
     *
     * @param planId 测试计划Id
     * @return 落地页列表
     */
    private List<String> selectGroupPromoteUrl(Long planId) {
        List<MaterialPromoteBindABTestGroupDO> groups = materialPromoteBindABTestGroupDAO.selectByPlanId(planId);
        if (CollectionUtils.isEmpty(groups)) {
            return Collections.emptyList();
        }

        List<Long> promoteIds = groups.stream().map(MaterialPromoteBindABTestGroupDO::getPromoteId).collect(Collectors.toList());
        return advertPromoteTestDAO.selectPromoteUrlPrimaryKey(promoteIds);
    }

    /**
     * 不进行ABTest的原因记录logExtMap日志
     */
    private void abtestSkipReasonLog(ObtainAdvertReq req, ABTestSkipEnum skipReason) {
        if (null == skipReason) {
            return;
        }

        String abtestSkip = req.getLogExtMap().get(AdvertReqLogExtKeyConstant.ABTEST_SKIP);
        JSONObject jo = Optional.ofNullable(abtestSkip).map(s -> JSON.parseObject(s, JSONObject.class)).orElse(new JSONObject());
        jo.put(skipReason.getCategory(), skipReason.getCode());
        req.getLogExtMap().put(AdvertReqLogExtKeyConstant.ABTEST_SKIP, jo.toJSONString());
    }
}
