package cn.com.duiba.tuia.service.impl;

import cn.com.duiba.tuia.dao.adx.AdxAdvertMaterialDao;
import cn.com.duiba.tuia.dao.engine.AdvertOrderDAO;
import cn.com.duiba.tuia.domain.dataobject.AdvertOrderDO;
import cn.com.duiba.tuia.domain.dataobject.AdvertOrderExtDO;
import cn.com.duiba.tuia.domain.dataobject.AdxAdvertMaterialDO;
import cn.com.duiba.tuia.domain.vo.OrderJsonVO;
import cn.com.duiba.tuia.log.InnerExtTwoLog;
import cn.com.duiba.tuia.service.DspComparePriceService;
import cn.com.duiba.tuia.tool.CatUtil;
import cn.com.duiba.tuia.tool.StringTool;
import cn.com.duiba.tuia.utils.TuiaStringUtils;
import cn.com.tuia.advert.model.SpmlogReq;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpStatus;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import static cn.com.duiba.tuia.enums.DspMappingEnum.DSP_2;

@RefreshScope
@Service
public class DspComparePriceServiceImpl implements DspComparePriceService {

    private static Logger logger = LoggerFactory.getLogger(DspComparePriceServiceImpl.class);

    @Resource
    private ExecutorService executorService;
    @Resource
    private AdxAdvertMaterialDao adxAdvertMaterialDao;
    @Resource
    private CloseableHttpClient httpClient;
    @Autowired
    private AdvertOrderDAO advertOrderDAO;

    private RequestConfig requestConfig1 = RequestConfig.custom()
            .setSocketTimeout(50)
            .setConnectTimeout(50)
            .setConnectionRequestTimeout(50).build();

    /**
     * 组装adx有效广告缓存key
     *
     * @param dspId
     * @param advertId
     * @return
     */
    private String buildKey(Long dspId, Long advertId) {
        return String.valueOf(dspId) + ";" + String.valueOf(advertId);
    }

    /**
     * key:dspId+";"+advertId value:adx广告 有消息同步，不用设置过期时间来保证避免读取旧值
     */
    private LoadingCache<String, Optional<AdxAdvertMaterialDO>> ADX_ADVERTS_CACHE = CacheBuilder.newBuilder().initialCapacity(500).maximumSize(1000)
            .refreshAfterWrite(5, TimeUnit.MINUTES)
            .build(new CacheLoader<String, Optional<AdxAdvertMaterialDO>>() {
                @Override
                public Optional<AdxAdvertMaterialDO> load(String key) {
                    String[] val = key.split(";");
                    return Optional.ofNullable(getAdxAdvertByAdxAdvertId(Long.valueOf(val[0]), Long.valueOf(val[1])));
                }

                @Override
                public ListenableFutureTask<Optional<AdxAdvertMaterialDO>> reload(final String key, Optional<AdxAdvertMaterialDO> oldValue) {
                    ListenableFutureTask<Optional<AdxAdvertMaterialDO>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });

    @Override
    public void init() {
        //查询所有的adx互动广告数据
        List<AdxAdvertMaterialDO> adxAdverts = adxAdvertMaterialDao.getAdxAdverts();
        logger.info("dsp adx :{}", JSON.toJSONString(adxAdverts));
        if (CollectionUtils.isEmpty(adxAdverts)) {
            return;
        }
        //组装dspId"+advertId  维度adx广告数据
        Map<String, Optional<AdxAdvertMaterialDO>> adxAdvertMap = adxAdverts.stream().map(dto -> {
            buildAdxAdvertMaterial(dto);
            return Optional.ofNullable(dto);
        }).collect(Collectors.toMap(dto -> buildKey(dto.get().getDspId(), dto.get().getAdvertId()), Function.identity(), (p, q) -> p));
        ADX_ADVERTS_CACHE.putAll(adxAdvertMap);

    }

    private void buildAdxAdvertMaterial(AdxAdvertMaterialDO dto) {
        dto.setBannedTags(TuiaStringUtils.getStringListByStr(dto.getBannedTag()));
        dto.setIndustryTags(TuiaStringUtils.getStringListByStr(dto.getIndustryTag()));
        dto.setMaterialTags(TuiaStringUtils.getStringListByStr(dto.getMaterialTag()));
        dto.setPromoteTags(TuiaStringUtils.getStringListByStr(dto.getPromoteTag()));
        dto.setBannedTag(null);
        dto.setIndustryTag(null);
        dto.setMaterialTag(null);
        dto.setPromoteTag(null);
    }

    @Override
    public void updateAdxAdvertCache(String key) {
        try {
            if (StringUtils.isNotBlank(key)) {
                ADX_ADVERTS_CACHE.refresh(key);
            }
        } catch (Exception e) {
            logger.error("update AdxAdvertMaterialDO error", e);
        }
    }

    @Override
    public AdxAdvertMaterialDO getAdxAdvertCache(Long dspId, Long advertId) {
        //京准通 没有广告
        if(DSP_2.equals(dspId)){
            return null;
        }
        String key = buildKey(dspId, advertId);
        Optional<AdxAdvertMaterialDO> result = ADX_ADVERTS_CACHE.getUnchecked(key);
        if (!result.isPresent()) {
            return null;
        }
        return result.get();
    }

    @Override
    public AdxAdvertMaterialDO getValidAdxAdvert(Long dspId, Long advertId) {
        //京准通 没有广告
        if(DSP_2.equals(dspId)){
            return null;
        }
        String key = buildKey(dspId, advertId);
        Optional<AdxAdvertMaterialDO> result = ADX_ADVERTS_CACHE.getUnchecked(key);
        if (!result.isPresent()) {
            return null;
        }
        AdxAdvertMaterialDO adxAdvert = result.get();
        return adxAdvert.getCheckStatus() == 1 && adxAdvert.getIsDeleted() == 0 ? adxAdvert : null;
    }

    @Override
    public AdxAdvertMaterialDO getAdxAdvertByAdxAdvertId(Long dspId, Long advertId) {
        AdxAdvertMaterialDO dto = adxAdvertMaterialDao.getAdxAdvertByAdxAdvertId(dspId, advertId);
        if (dto == null) {
            return null;
        }
        buildAdxAdvertMaterial(dto);
        return dto;
    }

    @Override
    public void adxAdvertShowLog(SpmlogReq req, AdvertOrderDO advertOrderDO, OrderJsonVO vo) {
        //1.广告曝光日志 组装打印
        JSONObject log = new JSONObject();
        buildCommonLog(req, advertOrderDO, vo, log);
        InnerExtTwoLog.exposureLog(log);

        //2.发送曝光监控链接
        if (StringUtils.isBlank(vo.getsUrl())) {
            return;
        }
        List<String> urls = StringTool.getStringListByStr(vo.getsUrl());
        if(urls.size()>2){
            CatUtil.catLog("dsp 曝光回调链接个数超过2个");
        }
        executorService.submit(() -> {
            for (String url : urls) {
                sendUrl(url);
            }
        });

    }

    @Override
    public void adxAdvertConsumeLog(SpmlogReq req, AdvertOrderDO advertOrderDO, OrderJsonVO vo) {
        //1.广告计费日志 组装打印
        JSONObject log = new JSONObject();
        buildCommonLog(req, advertOrderDO, vo, log);
        InnerExtTwoLog.consumeLog(log);
    }

    /**
     * dsp 曝光日志 公共部分
     *
     * @param req
     * @param advertOrderDO
     * @param log
     */
    private void buildCommonLog(SpmlogReq req, AdvertOrderDO advertOrderDO, OrderJsonVO vo, JSONObject log) {
        log.put("consumer_id", req.getConsumerId());
        //每次请求 的唯一标识，推啊活动id
        log.put("activity_order_id", req.getOrderId());
        log.put("activity_id", req.getActivityId());
        //0-兑吧普通活动,1-兑吧商业活动,2-推啊活动，3-友推
        log.put("activity_use_type", vo.getuT());
        log.put("order_id", advertOrderDO.getId());
        //dsp_id
        log.put("dsp_id", vo.getDspId());
        log.put("advert_id", advertOrderDO.getAdvertId());
        //adx 广告单价
        log.put("adx_fee", vo.getAdxFe());
        //展示广告通知链接
        log.put("show_url", vo.getsUrl());
        //点击广告通知链接
        log.put("click_url", vo.getcUrl());
        //广告展示设备信息
        log.put("device_id", req.getDeviceId());

        //本地应用信息
        log.put("app_id", req.getAppId());
        //广告位id
        log.put("slot_id", req.getSlotId());

        //广告展现位次（互动广告中发券次序）
        log.put("put_index", vo.getApi());
        //可接受的广告类型
        log.put("ad_type", 1);
        //千次展示出价底价，单位为分;注意:价格为动态门槛(暂定默认2000)
        log.put("bid_floor", vo.getAdxFe());

        //城市
        log.put("city", vo.getCi());

        //地区
        log.put("region", vo.getProv());
        //浏览器的 user agent string
        log.put("ua", advertOrderDO.getUa());

        //ip 地址
        log.put("ip", req.getIp());

        //MD5加密后的oaid设备标示号，32位小写
        log.put("oaid_md5", vo.getOaid5());

        //MD5加密后的imei设备标示号，32位小写
        log.put("imei_md5", vo.getIme5());

        //MD5加密后的idfa设备标示号，32位小写
        log.put("idfa_md5", vo.getIdfa5());

        //操作系统(如iOS or Android)
        log.put("os", req.getOs());

        //操作系统版本(如7.1)
        log.put("osv", vo.getOsv());
        log.put("material_id", null);
        //流量来源:1-推啊(默认)2-兑吧
        log.put("flow_source", req.getFlowSource());
        //活动主体类型:1-杭州推啊，2-霍尔果斯推啊	无
        log.put("effective_main_type", 1);
        //投放类型:1-福袋2-直投3-浮标
        log.put("delivery_type", req.getDeliveryType());
        //活动测试字段,活动大类
        log.put("activity_skin_type", vo.getAskt());

        Map<String, String> extExtMap = Optional.ofNullable(vo.getLgEp()).orElse(new HashMap<>());
        log.put("'act_main_title_td'", extExtMap.get("actMainTitleId"));
        log.put("'act_sub_title_td'", extExtMap.get("actSubTitleId"));
        log.put("'act_page_main_title_id'", extExtMap.get("actPageMainTitleId"));
        log.put("'act_page_sub_title_id'", extExtMap.get("actPageSubTitleId"));

        log.put("time", req.getTime());
    }

    @Override
    public void adxAdvertClickLog(SpmlogReq req, AdvertOrderDO advertOrderDO, OrderJsonVO vo) {

        //如果是dsp 查询广告扩展表  获取监控链接数据
        AdvertOrderExtDO orderExt = advertOrderDAO.getAdvertOrderExt(req.getConsumerId(), req.getOrderId());
        if (orderExt != null) {
            vo.setcUrl(orderExt.getClickUrl());
            vo.setsUrl(orderExt.getShowUrl());
        }

        //1.广告曝光日志 组装打印
        JSONObject log = new JSONObject();
        buildCommonLog(req, advertOrderDO, vo, log);
        InnerExtTwoLog.clickLog(log);
        //发送点击监控链接 ,京准通点击不需要回调
        if (StringUtils.isBlank(vo.getcUrl()) || DSP_2.getDspId().equals(vo.getDspId())) {
            return;
        }
        executorService.submit(() -> sendUrl(vo.getcUrl()));
    }

    /**
     * 发送曝光监控链接
     *
     * @return
     */
    private void sendUrl(String sUrl) {

        try (CloseableHttpResponse httpResponse = httpClient.execute(buildRequestGet(sUrl))) {
            if (httpResponse.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
            }
        } catch (Exception e) {
            // 报错打日志，不做其他处理
            logger.warn("sendUrl:{} error", sUrl, e);
        }
    }

    /**
     * 组装http get调用参数
     *
     * @param url
     * @return
     * @throws URISyntaxException
     */
    private HttpGet buildRequestGet(String url) throws URISyntaxException {
        HttpGet request = new HttpGet(url);
        request.setConfig(requestConfig1);
        return request;
    }

}
