package cn.com.duiba.tuia.service.impl;

import cn.com.duiba.tuia.cache.MediaCacheService;
import cn.com.duiba.tuia.core.api.enums.EnableStatusEnum;
import cn.com.duiba.tuia.domain.dataobject.AppDO;
import cn.com.duiba.tuia.domain.dataobject.SlotDO;
import cn.com.duiba.tuia.domain.model.AppDetail;
import cn.com.duiba.tuia.domain.model.AdvBannedTag;
import cn.com.duiba.tuia.domain.model.CheckStrategyRet;
import cn.com.duiba.tuia.domain.model.FlowStrategyRet;
import cn.com.duiba.tuia.domain.model.RspEngineAppFlowStrategyDto;
import cn.com.duiba.tuia.exception.TuiaException;
import cn.com.duiba.tuia.service.ShieldingStrategyService;
import cn.com.duiba.tuia.service.SlotFlowStrategyService;
import cn.com.duiba.tuia.utils.ShieldingStrategyUtil;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.tuia.advert.enums.StrategyTypeEnum;
import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 媒体配置策略
 * Created by zhengjy on 2017/4/14
 */
@Service
public class FlowStrategyServiceImpl implements ShieldingStrategyService {

    private static final Logger logger = LoggerFactory.getLogger(FlowStrategyServiceImpl.class);

    /** 新媒体流量策略广告白名单排序位数*/
    public static final int FLOW_STRATEGY_WHITE_SORT = 1;

    @Autowired
    private MediaCacheService mediaCacheService;
    @Autowired
    private SlotFlowStrategyService slotFlowStrategyService;

    @Override
    public List<String> getTags(AppDO appDO, SlotDO slotDO, Boolean isHandledSlot) {

        if (!isHandledSlot || slotDO == null) {
            return Lists.newArrayListWithCapacity(0);
        }

        return ShieldingStrategyUtil.getTags(slotDO.getFlowBannedTags(), slotDO.getFlowBannedAdvertTags(), slotDO.getFlowPromoteUrlTags(), slotDO.getFlowBannendResourceTags());

    }

    @Override
    public void handleBannedTag(AppDO appDO, SlotDO slotDO, boolean isHandledSlot, AdvBannedTag advBannedTag) {
        if (!isHandledSlot || slotDO == null) {
            return;
        }
        if (StringUtils.isNotEmpty(slotDO.getFlowBannedTags())) {
            advBannedTag.getAttributeTags().addAll(Arrays.asList(slotDO.getFlowBannedTags().split(",")));
        }
        if (StringUtils.isNotEmpty(slotDO.getFlowBannedAdvertTags())) {
            advBannedTag.getIndustryTags().addAll(Arrays.asList(slotDO.getFlowBannedAdvertTags().split(",")));
        }
        if (StringUtils.isNotEmpty(slotDO.getFlowPromoteUrlTags())) {
            advBannedTag.getPromoteUrlTags().addAll(Arrays.asList(slotDO.getFlowPromoteUrlTags().split(",")));
        }
        if (StringUtils.isNotEmpty(slotDO.getFlowBannendResourceTags())) {
            advBannedTag.getResourceTags().addAll(Arrays.asList(slotDO.getFlowBannendResourceTags().split(",")));
        }
    }

    @Override
    public List<String> getShieldMaterialTags(AppDO appDO, SlotDO slotDO) {
        if (appDO == null || slotDO == null || StringUtils.isBlank(slotDO.getFlowShieldMaterialTag())) {
            return Lists.newArrayListWithCapacity(0);
        }
        return Arrays.asList(slotDO.getFlowShieldMaterialTag().split(","));
    }

    private List<String> getShieldUrls(String shieldUrl){
        List<String> shieldUrls = Lists.newArrayList();
        if(StringUtils.isNotBlank(shieldUrl)){
            shieldUrls = Arrays.asList(shieldUrl.split("\n"));
        }
        return shieldUrls;
    }

    @Override
    public CheckStrategyRet checkStrategy(AppDO appDO, SlotDO slotDO, Boolean isProxy, String cityId, Long slotId) throws TuiaException {

        //1.是否已转换为广告位的标示,默认值 没转换
        Boolean isHandledSlot = false;
        RspEngineAppFlowStrategyDto engineAppFlowStrategyDto = null;

        //2.查询广告位的策略配置
        if (null != slotId) {
            engineAppFlowStrategyDto = slotFlowStrategyService.findSlotFlowStrategyBySlotId(slotId);
            isHandledSlot = true;
        }
        //3.如果广告位策略为空，或者未开启，则直接返回
        if(engineAppFlowStrategyDto == null || EnableStatusEnum.ENABLE_STATUS_FALSE.getCode() == engineAppFlowStrategyDto.getSwitchStrategy()){
            return new CheckStrategyRet(Boolean.TRUE,null,null, isHandledSlot, 2);
        }

        //7.开启验证代理
        if(EnableStatusEnum.ENABLE_STATUS_TRUE.getCode() == engineAppFlowStrategyDto.getSwitchProxy()){
            //7.1当前是代理/vpn流量
            if(isProxy){
                return new CheckStrategyRet(Boolean.TRUE,null,null, isHandledSlot, 2);
            }
        }
        Date date = new Date();
        //8.失效时间，当前时间小于开始日期 或 当前时间大于结束日期
        if (!checkInValidTime(engineAppFlowStrategyDto, date)) {
            return new CheckStrategyRet(Boolean.TRUE, null, null, isHandledSlot, 2);
        }
        //9.验证当前周期段
        if(!checkMinute(engineAppFlowStrategyDto.getValidPeriodList(),date)){
            return new CheckStrategyRet(Boolean.TRUE,null,null, isHandledSlot, 2);
        }
        //10.地域是否匹配
        if(StringUtils.isNotBlank(engineAppFlowStrategyDto.getValidRegions())){
            List<String> regions = Arrays.asList(engineAppFlowStrategyDto.getValidRegions().split(","));
            //10.1当前传入的地址不匹配
            if(!regions.contains(cityId)){
                return new CheckStrategyRet(Boolean.TRUE,null,null, isHandledSlot, 2);
            }
        }
        //11.是否开启福袋标识
        boolean isSendLuckybag = EnableStatusEnum.ENABLE_STATUS_TRUE.getCode() == engineAppFlowStrategyDto.getSwitchLuckybag() ? Boolean.TRUE : Boolean.FALSE;
        //12.媒体策略屏蔽url
        List<String> shieldUrls = getShieldUrls(engineAppFlowStrategyDto.getShieldUrl());
        FlowStrategyRet fs = new FlowStrategyRet(isSendLuckybag,engineAppFlowStrategyDto.getId());
        return new CheckStrategyRet(Boolean.FALSE, fs, shieldUrls, isHandledSlot, 2);
    }

    /**
     * 检查开始失效时间的有效性
     * @param engineAppFlowStrategyDto
     * @param date
     * @return
     */
    private boolean checkInValidTime(RspEngineAppFlowStrategyDto engineAppFlowStrategyDto, Date date) {
        if(engineAppFlowStrategyDto.getInvalidStartTime() != null && engineAppFlowStrategyDto.getInvalidEndTime() != null &&
                date.after(engineAppFlowStrategyDto.getInvalidStartTime())){
            if(date.before(engineAppFlowStrategyDto.getInvalidEndTime())){
                return false;
            }
        }
        return true;
    }

    @Override
    public String getShieldingStrategyType() {
        return StrategyTypeEnum.STRATEGY_TYEP_FLOW.getCode();
    }

    @Override
    public boolean isSendLuckybag(Object obj) {
        return ((FlowStrategyRet) obj).isSendLuckybag();
    }

    /**
     * 获取广告位白名单，新策略的
     * @param appDetail
     * @return
     * @throws TuiaException
     */
    @Override
    public List<Long> getSlotWhiteList(AppDetail appDetail) throws TuiaException {
        try {
            DBTimeProfile.enter("getSlotWhiteList");
            FlowStrategyRet sr = (FlowStrategyRet) appDetail.getObjParam();
            return mediaCacheService.getSlotWhiteList(StrategyTypeEnum.STRATEGY_TYEP_FLOW, sr.getAppFlowStrategyId(), appDetail.getSlotDO().getSlotId());
        }catch (Exception e){
            logger.error("getSlotWhiteList exception, appDetail={}", JSON.toJSONString(appDetail), e);
        }finally {
            DBTimeProfile.release();
        }
        return Collections.emptyList();
    }

    private static int getWeekOfDate(Date dt) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(dt);
        int w = cal.get(Calendar.DAY_OF_WEEK) - 1;
        if (w < 0){
            w = 0;
        }
        if (w == 0){
            w = 7;
        }
        return w;
    }
    /**
     *
     * @param validPeriodDtos
     * @param date
     * @return true：在可发放时间段 ，false：不在发放时间段
     */
    @SuppressWarnings("squid:S3776")
    private boolean checkMinute(List<RspEngineAppFlowStrategyDto.ValidPeriodDto> validPeriodDtos,Date date){
        if(CollectionUtils.isEmpty(validPeriodDtos)){
            return Boolean.TRUE;
        }
        //当前周期时间
        int day = getWeekOfDate(date);
        //当前时间段
        int curHour = Integer.valueOf(DateUtils.getMinuteOnlyStr(date).substring(0,2));

        //当前时间段(时分)
        SimpleDateFormat hhmm = new SimpleDateFormat("HHmm");
        int curMinute = Integer.valueOf(hhmm.format(date));

        for(RspEngineAppFlowStrategyDto.ValidPeriodDto vd : validPeriodDtos){
            //找到当天
            if(vd.getWeek() == day){
                if (vd.getHourBegin().length() == 2) {//兼容处理,老数据存储的是02、04这种小时数据
                    //开始时间段
                    int begin = Integer.valueOf(vd.getHourBegin());
                    //结束时间段
                    int end = Integer.valueOf(vd.getHourEnd());
                    //开启时间段 <= 当前时间段 || 结束时间段 >= 当前时间段，在可发放券的时间段
                    if (begin <= curHour && end >= curHour) {
                        return Boolean.TRUE;
                    }
                } else {//新数据存储的是0000、0030、1030这种到分钟级别的数据
                    //开始时间段
                    int begin = Integer.valueOf(vd.getHourBegin());
                    //结束时间段
                    int end = Integer.valueOf(vd.getHourEnd());
                    //开启时间段 <= 当前时间段 || 结束时间段 >= 当前时间段，在可发放券的时间段
                    if (begin <= curMinute && end >= curMinute) {
                        return Boolean.TRUE;
                    }
                }
            }
        }
        return Boolean.FALSE;
    }

    /**
     * 获取策略code用于计算白名单竞价排序开关
     */
    @Override
    public Integer getStrategyCode() {
        return FLOW_STRATEGY_WHITE_SORT;
    }
}
