package cn.com.duiba.tuia.service.impl;

import cn.com.duiba.tuia.domain.model.*;
import cn.com.duiba.tuia.enums.CatGroupEnum;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Maps;

import cn.com.duiba.boot.utils.WarningUtils;
import cn.com.duiba.tuia.cache.AdvertMapCacheManager;
import cn.com.duiba.tuia.cache.MediaCacheService;
import cn.com.duiba.tuia.constants.AdvertReqLogExtKeyConstant;
import cn.com.duiba.tuia.constants.CommonConstants;
import cn.com.duiba.tuia.constants.ErrorCode;
import cn.com.duiba.tuia.constants.TrusteeshipConstants;
import cn.com.duiba.tuia.domain.vo.AdvertPriceVO;
import cn.com.duiba.tuia.domain.vo.AdvertVO;
import cn.com.tuia.advert.enums.AdvertFilterTypeEnum;
import cn.com.duiba.tuia.exception.TuiaException;
import cn.com.duiba.tuia.log.StatRequestJsonLog;
import cn.com.duiba.tuia.service.AdvertExposeService;
import cn.com.duiba.tuia.service.AdvertMaterialRecommendService;
import cn.com.duiba.tuia.service.CommonService;
import cn.com.duiba.tuia.service.ImitateAdvertService;
import cn.com.duiba.tuia.tool.CatUtil;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.tuia.advert.constants.CommonConstant;
import cn.com.tuia.advert.enums.ChargeTypeEnum;
import cn.com.tuia.advert.enums.PutTargetTypeEnum;
import cn.com.tuia.advert.model.ObtainAdvertReq;
import cn.com.tuia.advert.model.ObtainAdvertRsp;

/**
 *
 * @author peanut.huang
 * @date 2019/9/20
 * @since JDK 1.8
 */
@Service
@Slf4j
public class ImitateAdvertServiceImpl implements ImitateAdvertService {

    @Resource
    private CommonService                   commonService;
    @Resource
    private AdvertMapCacheManager           advertMapCacheManager;
    @Resource
    private MediaCacheService               mediaCacheService;
    @Resource
    private AdvertExposeService             advertExposeService;
    @Resource
    private AdvertMaterialRecommendService  advertMaterialRecommendService;


    /**
     * 模拟请求广告
     *
     * @param req
     * @return
     */
    @Override
    public ObtainAdvertRsp imitateObtainAdvert(ObtainAdvertReq req) {
        DBTimeProfile.enter("ImitateAdvertService.imitateObtainAdvert");
        ObtainAdvertRsp rsp = new ObtainAdvertRsp();
        rsp.setSlotId(req.getSlotId());
        rsp.setResult(false);

        try{

            // 参数检查
            checkReq(req);

            // 1、打印请求日志
            FilterResult filterResult = printReqLog(req);

            // 2、查询有效的广告列表
            List<AdvOrientationItem> originalList = advertMapCacheManager.getValidPkgFilterCache();
            if (CollectionUtils.isEmpty(originalList)) {
                CatUtil.log(CatGroupEnum.CAT_106005.getCode());
                return rsp;
            }

            // 复制配置荐
            List<AdvOrientationItem> orientationItemList = copyOrientationItem(req.getActivityMaterialType(),originalList);


            // 2、查询应用管理广告位白名单
            List<Long> advertWhiteList = fetchWhiteList(req.getSlotId());
            if(CollectionUtils.isEmpty(advertWhiteList)){
                log.warn("imitateObtainAdvert white list is empty, orderId={}", req.getOrderId());
                return rsp;
            }

            // 3、广告列表与白名单取交，并随便取出一个配置进行发券
            AdvOrientationItem orientationItem =  retainAndPickOrientation(advertWhiteList, orientationItemList);
            if(orientationItem == null){
                log.warn("imitateObtainAdvert retainAndPickOrientation is null, orderId={}, whiteList={}", req.getOrderId(), advertWhiteList);
                return rsp;
            }

            AdvertFilter advertFilter = new AdvertFilter();
            advertFilter.setAppId(req.getAppId());
            advertFilter.setSlotId(req.getSlotId());
            advertFilter.setOrderId(req.getOrderId());
            advertFilter.setActivityId(req.getActivityId());

            // 4、出券
            finishObtain(req, rsp, filterResult, orientationItem, advertFilter);
        } catch (Throwable ex) {
            log.error("imitateObtainAdvert error", ex);
        } finally {
            DBTimeProfile.release();
        }
        return rsp;
    }

    private List<AdvOrientationItem> copyOrientationItem(Integer activityMaterialType,List<AdvOrientationItem> originalList) {
        List<AdvOrientationItem> result = new ArrayList<>(originalList.size());

        originalList.stream().filter(e -> {
            int chargeType = Optional.ofNullable(e.getChargeType()).orElse(0);
            return ChargeTypeEnum.TYPE_CPC.getCode() == chargeType;
        }).forEach(item ->{
            // 此处copy一份item，指向新的内存对象，避免线程间的错乱
            AdvOrientationItem advOrientationItem = BeanUtils.copy(item, AdvOrientationItem.class);

            Set<Long> materialIds = advertMaterialRecommendService.getMaterialSetByAdvertId(item.getAdvertId());

            //TODO 虚拟广告加入活动类型过滤(等待产品决定)
            materialIds = advertMaterialRecommendService.filterByActivityType(activityMaterialType,item.getAdvertId(),materialIds);

            advOrientationItem.setLeftMaterial(materialIds);

            result.add(advOrientationItem);
        });
        return result;
    }

    private void checkReq(ObtainAdvertReq req) throws Exception{
        // 1.虚拟广告位模拟请求发券标识
        String imitateReq = Optional.ofNullable(req.getLogExtMap()).orElse(new HashMap<>()).get(AdvertReqLogExtKeyConstant.IMITATE_REQ);
        if(!"1".equals(imitateReq)){
            log.warn("非模拟发券请求， LogExtMap=[{}]", req.getLogExtMap());
            throw new TuiaException(ErrorCode.E0100002);
        }

        // 2.用户id,媒体id,活动id,活动订单id为空验证
        if (req.getConsumerId() == null || req.getAppId() == null || req.getSlotId() == null
            || req.getActivityId() == null || StringUtils.isEmpty(req.getOrderId())) {
            log.warn("obtainAdvert error, req = [{}], please check the", req);
            throw new TuiaException(ErrorCode.E0100002);
        }
        // 3.请求ip为空验证
        if (StringUtils.isEmpty(req.getIp())) {
            log.warn("obtainAdvert error, req = [{}], please check the", req);
            throw new TuiaException(ErrorCode.E0100002);
        }

        // 4.UA校验和监控
        if (StringUtils.isBlank(req.getUa()) || CommonConstants.UNKNOWN.equals(req.getUa())) {
            CatUtil.log(CatGroupEnum.CAT_107004.getCode());
            req.setUa(CommonConstants.UNKNOW);
        }
    }

    /**
     * 出券
     *
     * @param req
     * @param rsp
     * @param filterResult
     *@param orientationItem  @throws TuiaException
     */
    private void finishObtain(ObtainAdvertReq req, ObtainAdvertRsp rsp, FilterResult filterResult, AdvOrientationItem orientationItem, AdvertFilter advertFilter) throws TuiaException {

        // 计划id
        Long advertId = orientationItem.getAdvertId();

        // 广告属性
        AdvertVO advertVO = advertMapCacheManager.getAdvertCache(advertId);
        if(advertVO == null){
            log.warn("advertMapCacheManager getAdvertCache is null, orderId = {}", req.getOrderId());
            return;
        }

        // 配置对象转化为PriceVO
        AdvertPriceVO advertPriceVO = orientation2PriceVO(orientationItem);

        Long fee = advertPriceVO.getFee();

        // 出券
        advertExposeService.finishBiz(req, rsp, advertId, advertVO, filterResult, advertPriceVO, fee, advertFilter);
    }

    /**
     *
     * @param advOrientationItem
     * @return
     */
    private AdvertPriceVO orientation2PriceVO(AdvOrientationItem advOrientationItem) {
        // 计划id
        Long advertId = advOrientationItem.getAdvertId();

        AdvertPriceVO priceVO = new AdvertPriceVO(advOrientationItem.getPeriodList(), advOrientationItem.getCpcPrice(), advOrientationItem.getChargeType());

        priceVO.setAdvertId(advertId);
        priceVO.setAdvertOrientationPackageId(advOrientationItem.getOrientationId());
        priceVO.setOriginalOrientationId(advOrientationItem.getInitialOrientationId());

        priceVO.setSupportStatus(advOrientationItem.getSupportStatus());
        priceVO.setNewTradeTagNum(advOrientationItem.getNewTradeTagNum());
        priceVO.setNewTradeTagId(advOrientationItem.getNewTradeTagId());

        //
        priceVO.setTrusteeship(advOrientationItem.getTargetAppLimit() == TrusteeshipConstants.AUTO_MODE ? CommonConstant.YES : CommonConstant.NO);
        priceVO.setTargetAppLimit(Optional.ofNullable(advOrientationItem.getTargetAppLimit()).orElse(1));
        priceVO.setStrongTarget(advOrientationItem.getStrongTarget());

        priceVO.setPackageType(advOrientationItem.getPackageType());
        priceVO.setActivityType(advOrientationItem.getActivityType());
        priceVO.setBudgetPerDay(advOrientationItem.getBudgetPerDay());
        //如果开启了自动托管则设置自动托管转换出价，用于打日志和传给哪吒
        priceVO.setConvertCost(advOrientationItem.getCpaPrice());
        priceVO.setAppTargetPackage(advOrientationItem.getAppTargetPackage());
        priceVO.setSlotTargetPackage(advOrientationItem.getSlotTargetPackage());
        priceVO.setTargetRecommendType(advOrientationItem.getTargetRecommendType());

        //设置签收因子,如果是ocpc并且改配置选择了 调整因子为签收
        priceVO.setSubtype(advOrientationItem.getSubtype());
        priceVO.setDepthSubtype(advOrientationItem.getDepthSubtype());
        priceVO.setDepthTargetPrice(advOrientationItem.getDepthTargetPrice());
        priceVO.setPutTargetType(AdvOrientationItem.configPutTargetType(advOrientationItem.getPutTargetType(), advOrientationItem.getStrongTarget(), advOrientationItem.getTargetAppLimit()));

        // 分媒体出价（重点媒体转化出价，潜力广告分媒体出价，普通的分媒体出价）
        // disAppFeeChange(priceVO, advOrientationItem, req.getAppId());

        priceVO.setResourceTag(advOrientationItem.getResourceTag());

        // 消耗模式
        priceVO.setBudgetSmooth(advOrientationItem.getBudgetSmooth());

        //多连接测试链接设置
        priceVO.setPromoteTestUrl(advOrientationItem.getPromoteTestUrl());

        //设置 是否在 托管底价白名单 中
        priceVO.setIsObctTag(advOrientationItem.getIsObctTag());

        priceVO.setMaterialsBind(advOrientationItem.getLeftMaterial());

        return priceVO;
    }

    /**
     * 有效广告与白名单广告取交集
     *
     * @param advertWhiteList
     * @param validAdvertIdList
     * @return
     */
    private AdvOrientationItem retainAndPickOrientation(List<Long> advertWhiteList, List<AdvOrientationItem> validAdvertIdList) {

        // 过滤出在白名单中的配置集
        List<AdvOrientationItem> restPkgList = validAdvertIdList.stream()
                                                                .filter(e -> advertWhiteList.contains(e.getAdvertId()))
                                                                .collect(Collectors.toList());
        if(CollectionUtils.isEmpty(restPkgList)){
            return null;
        }

        // 所有广告配置中随机出一个配置
        int index = ThreadLocalRandom.current().nextInt(1, restPkgList.size() + 1);
        return restPkgList.get(index - 1);
    }

    /**
     * 获取应用管理广告位白名单
     *
     * @param slotId
     * @return
     */
    private List<Long> fetchWhiteList(Long slotId) {
        return mediaCacheService.getSlotWhiteList(slotId);
    }

    /**
     * 打印请求日志
     *
     * @param req
     */
    private FilterResult printReqLog(ObtainAdvertReq req) {

        FilterResult filterResult = new FilterResult(req,null);

        // 1.获取ip对应的城市id
        AdvQueryParam advQueryParam = commonService.ipGeoAnalysis(req.getIp(), req.getIpAreaDto(), req.getLogExtMap(), req.getDeviceId(), false, null, filterResult);
        String cityId = advQueryParam.getRegionId();
        filterResult.setCityId(cityId);

        // 2.设置城市
        if (req.getLogExtMap() != null) {
            req.getLogExtMap().put("cityId", cityId);
            req.getLogExtMap().put(AdvertReqLogExtKeyConstant.MAIN_TYPE, String.valueOf(AdvertReqLogExtKeyConstant.INTERACT));

        } else {
            Map<String, String> logExtMap = Maps.newHashMap();
            logExtMap.put("cityId", cityId);
            logExtMap.put(AdvertReqLogExtKeyConstant.MAIN_TYPE, String.valueOf(AdvertReqLogExtKeyConstant.INTERACT));
            req.setLogExtMap(logExtMap);
        }

        // 3.打印请求日志
        req.setType(SpmType.SPM_LOG_REQUEST);
        StatRequestJsonLog.imitateLog(req);

        return filterResult;
    }
}
