package cn.com.duiba.tuia.service.impl;

import cn.com.duiba.tuia.cache.ServiceManager;
import cn.com.duiba.tuia.constants.AdvertConstants;
import cn.com.duiba.tuia.constants.AdvertReqLogExtKeyConstant;
import cn.com.duiba.tuia.dao.apppackage.AppPackageDAO;
import cn.com.duiba.tuia.dao.engine.AdvertOrientationPackageDAO;
import cn.com.duiba.tuia.dao.layered.LayeredStrategyAdvertDAO;
import cn.com.duiba.tuia.dao.layered.LayeredStrategyAppPkgDAO;
import cn.com.duiba.tuia.dao.layered.LayeredStrategyDAO;
import cn.com.duiba.tuia.domain.dataobject.*;
import cn.com.duiba.tuia.domain.model.*;
import cn.com.duiba.tuia.domain.vo.AdvertPriceVO;
import cn.com.duiba.tuia.pangea.center.api.localservice.apollopangu.ApolloPanGuService;
import cn.com.duiba.tuia.service.BaseService;
import cn.com.duiba.tuia.service.LayeredStrategyService;
import cn.com.duiba.tuia.tool.StringTool;
import cn.com.duiba.tuia.utils.WeightRandomUtil;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.tuia.advert.cache.CacheKeyTool;
import cn.com.tuia.advert.cache.RedisCommonKeys;
import cn.com.tuia.advert.constants.LayeredStrategyConstant;
import cn.com.tuia.advert.enums.LayeredStrategyConditionEnum;
import cn.com.tuia.advert.enums.LayeredStrategyStatusEnum;
import cn.com.tuia.advert.model.ObtainAdvertReq;
import cn.com.tuia.advert.utils.DomainUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;
import com.google.common.base.Splitter;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * [广告分媒体调控]服务实现
 *
 * @author zhangbaiqiang
 * @date 2020/7/23
 */
@Service
public class LayeredStrategyServiceImpl extends BaseService implements LayeredStrategyService {

    @Autowired
    private AppPackageDAO appPackageDAO;

    @Autowired
    private LayeredStrategyAdvertDAO layeredStrategyAdvertDAO;

    @Autowired
    private LayeredStrategyDAO layeredStrategyDAO;

    @Autowired
    private LayeredStrategyAppPkgDAO layeredStrategyAppPkgDAO;

    @Resource
    protected StringRedisTemplate stringRedisTemplate;

    @Resource
    private ExecutorService executorService;

    @Autowired
    private ServiceManager serviceManager;

    @Autowired
    private AdvertOrientationPackageDAO advertOrientationPackageDAO;

    @Autowired
    private ApolloPanGuService apolloPanGuService;

    /**
     * 广告默认配置缓存
     * key: 广告id
     * value: 默认配置id
     */
    private final LoadingCache<Long, Optional<Long>> defaultOrientCache = CacheBuilder.newBuilder()
            .maximumSize(100)
            .refreshAfterWrite(5, TimeUnit.MINUTES)
            .expireAfterWrite(30,TimeUnit.MINUTES)
            .build(new CacheLoader<Long, Optional<Long>>() {

                @Override
                public Optional<Long> load(Long key) throws Exception {
                    AdvertOrientationPackageDO orientDO = advertOrientationPackageDAO.selectDefaultByAdvertId(key);
                    if (null == orientDO) {
                        return Optional.empty();
                    }
                    return Optional.ofNullable(orientDO.getId());
                }

                @Override
                public ListenableFuture<Optional<Long>> reload(Long key, Optional<Long> oldValue) throws Exception {
                    ListenableFutureTask<Optional<Long>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });

    /**
     * 广告-策略缓存
     * key: 媒体id
     * value: 广告配置id-策略列表映射
     */
    private final LoadingCache<Long, Map<Long, List<LayeredStrategyWeightDto>>> strategyCache = CacheBuilder.newBuilder()
            .maximumSize(500)
            .refreshAfterWrite(5, TimeUnit.MINUTES)
            .expireAfterWrite(1,TimeUnit.HOURS)
            .build(new CacheLoader<Long, Map<Long, List<LayeredStrategyWeightDto>>>() {

                @Override
                public Map<Long, List<LayeredStrategyWeightDto>> load(Long key) throws Exception {
                    return queryStrategyWeightMap(key);
                }

                @Override
                public ListenableFuture<Map<Long, List<LayeredStrategyWeightDto>>> reload(Long key, Map<Long, List<LayeredStrategyWeightDto>> oldValue) throws Exception {
                    ListenableFutureTask<Map<Long, List<LayeredStrategyWeightDto>>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });

    @Override
    public Double calculateAppWeight(AdvertPriceVO advertPriceVO, Double weight, Map<Long, List<LayeredStrategyWeightDto>> strategyMap) {
        if (null == advertPriceVO || null == weight || null == strategyMap) {
            return weight;
        }

        Long orientId = advertPriceVO.getAdvertOrientationPackageId();

        // 参数校验
        if (null == orientId || !(strategyMap.containsKey(orientId) || strategyMap.containsKey(LayeredStrategyConstant.TOTAL_ORIENT))) {
            return weight;
        }

        try {
            Date now = new Date();
            BigDecimal appWeight = BigDecimal.ONE;
            List<Long> strategyIds = new ArrayList<>();

            // 查询配置对应的策略列表，并计算权重
            List<LayeredStrategyWeightDto> strategyWeightDtos = new ArrayList<>();
            strategyWeightDtos.addAll(strategyMap.getOrDefault(orientId, Collections.emptyList()));
            strategyWeightDtos.addAll(strategyMap.getOrDefault(LayeredStrategyConstant.TOTAL_ORIENT, Collections.emptyList()));
            for (LayeredStrategyWeightDto strategyWeightDto : strategyWeightDtos) {
                // 校验广告id
                if (!Objects.equals(advertPriceVO.getAdvertId(), strategyWeightDto.getAdvertId())) {
                    continue;
                }

                // 校验生效时间
                if (now.before(strategyWeightDto.getEffectStart()) || now.after(strategyWeightDto.getEffectEnd())) {
                    continue;
                }

                // 计算权重
                appWeight = appWeight.multiply(strategyWeightDto.getWeight());
                strategyIds.add(strategyWeightDto.getStrategyId());
            }

            // 未匹配到策略，不计算直接返回
            if (CollectionUtils.isEmpty(strategyIds)) {
                return weight;
            }

            advertPriceVO.setLayeredStrategyIds(strategyIds);
            return appWeight.multiply(new BigDecimal(weight)).doubleValue();
        } catch (Exception e) {
            logger.warn("[广告分媒体调控]计算媒体权重异常, advertId={}, orientId={}", advertPriceVO.getAdvertId(), orientId, e);
        }
        return weight;
    }

    @Override
    public Map<Long, List<LayeredStrategyWeightDto>> queryStrategyWeightMapCache(Long appId) {
        try {
            return strategyCache.get(appId);
        } catch (ExecutionException e) {
            logger.warn("[广告分媒体调控]查询广告配置-策略权重映射缓存异常，appId={}", appId, e);
        }
        return Collections.emptyMap();
    }

    /**
     * 查询广告配置-策略权重映射
     *
     * @param appId 媒体id
     * @return 广告配置-策略权重映射
     */
    @SuppressWarnings("squid:S3776")
    private Map<Long, List<LayeredStrategyWeightDto>> queryStrategyWeightMap(Long appId) {
        if (null == appId) {
            return Collections.emptyMap();
        }

        Map<Long, List<LayeredStrategyWeightDto>> strategyWeightMap = new HashMap<>();
        Date now = new Date();

        try {
            // 查询策略广告配置信息
            List<LayeredStrategyAdvertDto> strategyAdvertDtos = queryStrategyAdvertList();
            if (CollectionUtils.isEmpty(strategyAdvertDtos)) {
                return Collections.emptyMap();
            }

            for (LayeredStrategyAdvertDto strategyAdvertDto : strategyAdvertDtos) {
                Long strategyId = strategyAdvertDto.getStrategyId();

                // 校验开启状态
                if (!Objects.equals(strategyAdvertDto.getStrategyStatus(), LayeredStrategyStatusEnum.OPEN.getStatus())) {
                    continue;
                }

                // 检查落地页生效条件
                if (!Objects.equals(strategyAdvertDto.getConditionValid(), true)) {
                    continue;
                }

                LayeredStrategyWeightDto strategyWeightDto = new LayeredStrategyWeightDto();
                strategyWeightDto.setStrategyId(strategyAdvertDto.getStrategyId());
                strategyWeightDto.setAdvertId(strategyAdvertDto.getAdvertId());
                strategyWeightDto.setWeight(strategyAdvertDto.getWeight());
                parseEffectPeriod(strategyWeightDto, strategyAdvertDto.getEffectPeriod());

                // 校验生效时间（只校验是否失效）
                if (now.after(strategyWeightDto.getEffectEnd())) {
                    continue;
                }

                // 查询策略生效的媒体ID集合, 并校验媒体ID
                Set<Long> appIds = queryAppIdsByStrategyId(strategyId);
                if (!appIds.contains(appId)) {
                    continue;
                }

                // 转成配置-策略权重映射
                for (Long orientId : strategyAdvertDto.getOrientIds()) {
                    List<LayeredStrategyWeightDto> strategyWeightDtos = strategyWeightMap.getOrDefault(orientId, new ArrayList<>());
                    strategyWeightDtos.add(strategyWeightDto);
                    strategyWeightMap.put(orientId, strategyWeightDtos);
                }
            }
        } catch (Exception e) {
            logger.warn("[广告分媒体调控]查询广告配置-策略权重映射异常, appId={}", appId, e);
        }

        return strategyWeightMap;
    }

    @Override
    public List<LayeredStrategyAdvertDto> queryStrategyAdvertList() {
        try {
            // 优先查询 Redis 缓存
            String redisKey = CacheKeyTool.getCacheKey(RedisCommonKeys.KC150);
            String redisValue = stringRedisTemplate.opsForValue().get(redisKey);
            if (StringUtils.isNotBlank(redisValue)) {
                return JSON.parseArray(redisValue, LayeredStrategyAdvertDto.class);
            }

            List<LayeredStrategyAdvertDto> strategyAdvertDtos = new ArrayList<>();

            // 查询 DB
            List<LayeredStrategyAdvertDO> strategyAdvertDOS = layeredStrategyAdvertDAO.selectAll();
            if (CollectionUtils.isNotEmpty(strategyAdvertDOS)) {
                // 查询策略映射
                Map<Long, LayeredStrategyDO> strategyMap = queryStrategyMap(strategyAdvertDOS);

                for (LayeredStrategyAdvertDO strategyAdvertDO : strategyAdvertDOS) {
                    // 广告Id
                    Long advertId = strategyAdvertDO.getAdvertId();

                    // 查询策略信息
                    LayeredStrategyDO strategyDO = strategyMap.get(strategyAdvertDO.getStrategyId());
                    if (null == strategyDO) {
                        continue;
                    }

                    // 解析配置Id
                    Set<Long> orientIdSet = Splitter.on(",").trimResults().omitEmptyStrings()
                            .splitToList(strategyAdvertDO.getOrientIds()).stream().map(Long::valueOf).collect(Collectors.toSet());

                    // 查询广告的默认配置
                    Long defaultOrientId = queryDefaultOrientId(advertId);
                    if (null != defaultOrientId && orientIdSet.contains(defaultOrientId)) {
                        orientIdSet.add(AdvertConstants.DEFAULT_ORIENTATION_ID);
                    }

                    // 组装策略广告配置信息
                    LayeredStrategyAdvertDto strategyAdvertDto = new LayeredStrategyAdvertDto();
                    strategyAdvertDto.setStrategyId(strategyAdvertDO.getStrategyId());
                    strategyAdvertDto.setEffectCondition(strategyDO.getEffectCondition());
                    strategyAdvertDto.setStrategyStatus(strategyDO.getStrategyStatus());
                    strategyAdvertDto.setEffectPeriod(strategyDO.getEffectPeriod());
                    strategyAdvertDto.setAdvertId(advertId);
                    strategyAdvertDto.setOrientIds(orientIdSet);
                    strategyAdvertDto.setWeight(strategyAdvertDO.getWeight());

                    // 落地页链接及是否满足生效条件
                    String promoteUrl = queryPromoteUrlByAdvertId(advertId);
                    strategyAdvertDto.setConditionValid(checkPromoteUrl(strategyAdvertDto.getEffectCondition(), promoteUrl));

                    strategyAdvertDtos.add(strategyAdvertDto);
                }
            }

            // 缓存到 Redis
            stringRedisTemplate.opsForValue().set(redisKey, JSON.toJSONString(strategyAdvertDtos), 10, TimeUnit.MINUTES);
            return strategyAdvertDtos;
        } catch (Exception e) {
            logger.warn("[广告分媒体调控]查询策略广告列表", e);
        }

        return Collections.emptyList();
    }

    @Override
    public Set<Long> queryAppIdsByStrategyId(Long strategyId) {
        if (null == strategyId) {
            return Collections.emptySet();
        }

        // 优先查询 Redis 缓存
        String redisKey = CacheKeyTool.getCacheKey(RedisCommonKeys.KC151, strategyId);
        String redisValue = stringRedisTemplate.opsForValue().get(redisKey);
        if (StringUtils.isNotBlank(redisValue)) {
            return JSON.parseObject(redisValue,  new TypeReference<Set<Long>>() {});
        }

        Set<Long> appIdSet = new HashSet<>();

        // 查询策略配置的流量包
        List<LayeredStrategyAppPkgDO> appPkgDOS = layeredStrategyAppPkgDAO.selectByStrategyId(strategyId);
        if (CollectionUtils.isNotEmpty(appPkgDOS)) {
            // 查询流量包配置的媒体ID
            List<Long> pkgIds = appPkgDOS.stream().map(LayeredStrategyAppPkgDO::getPkgId).collect(Collectors.toList());
            appIdSet.addAll(queryAppIdsByAppPkgId(pkgIds));
        }

        // 缓存到 Redis
        stringRedisTemplate.opsForValue().set(redisKey, JSON.toJSONString(appIdSet), 10, TimeUnit.MINUTES);
        return appIdSet;
    }

    @Override
    public Map<Long, Double> querySpecifyWeightMap(Long appId, ObtainAdvertReq req) {
        if (null == appId || null == req) {
            return Collections.emptyMap();
        }

        try {
            // 获取盘古配置
            String whiteListStr = apolloPanGuService.getIdMapStrByKeyStr("whitelist.specified.multi.weight");
            if (StringUtils.isBlank(whiteListStr)) {
                return Collections.emptyMap();
            }

            // 解析配置
            List<SpecifyWeightConfigDto> configs = JSON.parseArray(whiteListStr, SpecifyWeightConfigDto.class);
            if (CollectionUtils.isEmpty(configs)) {
                return Collections.emptyMap();
            }

            // 遍历配置的媒体
            for (SpecifyWeightConfigDto config : configs) {
                // 命中媒体进行分流，命中返回广告-权重映射，未命中返回空映射
                if (null != config.getAppIds() && config.getAppIds().contains(appId)) {
                    if (null != config.getRatio() && CollectionUtils.isNotEmpty(config.getMulti())) {
                        // 是否命中分流
                        boolean isHit =  WeightRandomUtil.weightToSuccess(config.getRatio());

                        // 记录命中分流日志
                        Map<String, String> logExtExpMap = req.getLogExtExpMap();
                        if (null == logExtExpMap) {
                            logExtExpMap = new HashMap<>();
                            req.setLogExtExpMap(logExtExpMap);
                        }
                        logExtExpMap.put(AdvertReqLogExtKeyConstant.HIT_SPECIFY, isHit ? "1" : "0");

                        // 命中分流构造广告权重映射
                        if (isHit) {
                            Map<Long, Double> specifyWeightMap = new HashMap<>();
                            config.getMulti().forEach(advert -> advert.getAdvertIds().forEach(advertId -> specifyWeightMap.put(advertId, advert.getWeight())));
                            return specifyWeightMap;
                        }
                    }
                    return Collections.emptyMap();
                }
            }
        } catch (Exception e) {
            logger.error("[广告指定媒体提权]querySpecifyWeightMap error, appId={}", appId, e);
        }
        return Collections.emptyMap();
    }

    @Override
    public Double calculateSpecifyWeight(AdvertPriceVO advertPriceVO, Double weight, Map<Long, Double> specifyWeightMap) {
        if (null == advertPriceVO || null == weight || MapUtils.isEmpty(specifyWeightMap)) {
            return weight;
        }

        try {
            Double specifyWeight = specifyWeightMap.get(advertPriceVO.getAdvertId());
            if (null == specifyWeight) {
                return weight;
            }

            advertPriceVO.setSpecifyWeight(specifyWeight);
            return new BigDecimal(weight).multiply(new BigDecimal(specifyWeight)).doubleValue();
        } catch (Exception e) {
            logger.warn("[广告指定媒体提权]计算权重异常, advertId={}", advertPriceVO.getAdvertId(), e);
        }
        return weight;
    }

    /**
     * 查询流量包的媒体ID集合
     *
     * @param appPkgIds 流量包id列表
     * @return 媒体ID集合
     */
    private Set<Long> queryAppIdsByAppPkgId(List<Long> appPkgIds) {
        if (CollectionUtils.isEmpty(appPkgIds)) {
            return Collections.emptySet();
        }

        List<AppPackageDO> appPackageDOS = appPackageDAO.selectByIds(appPkgIds);
        if (CollectionUtils.isEmpty(appPackageDOS)) {
            return Collections.emptySet();
        }

        return appPackageDOS.stream()
                .map(appPackageDO -> StringTool.getLongListByStr(appPackageDO.getAppIds()))
                .flatMap(List::stream)
                .collect(Collectors.toSet());
    }

    /**
     * 查询策略映射
     *
     * @param strategyAdvertDOS 策略广告配置列表
     * @return 策略id-策略映射
     */
    private Map<Long, LayeredStrategyDO> queryStrategyMap(List<LayeredStrategyAdvertDO> strategyAdvertDOS) {
        if (CollectionUtils.isEmpty(strategyAdvertDOS)) {
            return Collections.emptyMap();
        }

        if (strategyAdvertDOS.size() == 1) {
            LayeredStrategyDO strategyDO = layeredStrategyDAO.selectByPrimaryKey(strategyAdvertDOS.get(0).getStrategyId());
            if (null == strategyDO) {
                return Collections.emptyMap();
            }
            Map<Long, LayeredStrategyDO> map = new HashMap<>(1);
            map.put(strategyDO.getId(), strategyDO);
            return map;
        }

        List<Long> strategyIds = strategyAdvertDOS.stream().map(LayeredStrategyAdvertDO::getStrategyId).collect(Collectors.toList());
        List<LayeredStrategyDO> strategyDOS = layeredStrategyDAO.selectByIds(strategyIds);
        if (CollectionUtils.isEmpty(strategyDOS)) {
            return Collections.emptyMap();
        }
        return strategyDOS.stream().collect(Collectors.toMap(LayeredStrategyDO::getId, Function.identity(), (o, n) -> n));
    }

    /**
     * 解析策略的生效时间范围
     *
     * @param strategyDto 策略
     * @param effectPeriod 生效时间范围
     */
    private void parseEffectPeriod(LayeredStrategyWeightDto strategyDto, String effectPeriod) {
        Date now = new Date();

        try {
            // 不限的时间范围设置为昨天开始的一个月(远大于缓存过期时间)
            if (StringUtils.isBlank(effectPeriod)) {
                strategyDto.setEffectStart(DateUtils.daysAddOrSub(now, -1));
                strategyDto.setEffectEnd(DateUtils.daysAddOrSub(now, 30));
            }
            // 解析时间范围（yyyy.MM.dd-yyyy.MM.dd）
            else {
                String[] times = effectPeriod.split("-");
                strategyDto.setEffectStart(DateUtils.getDayStartTime(times[0].replace(".", "-")));
                strategyDto.setEffectEnd(DateUtils.getDayEndTime(times[1].replace(".", "-")));
            }
        } catch (Exception e) {
            logger.warn("[广告分媒体调控]解析生效时间异常, strategyId={}, effectPeriod={}", strategyDto.getStrategyId(), effectPeriod, e);
            // 降级设置为不生效的时间段
            strategyDto.setEffectStart(DateUtils.daysAddOrSub(now, -1));
            strategyDto.setEffectEnd(DateUtils.daysAddOrSub(now, -1));
        }
    }

    /**
     * 查询广告的落地页
     *
     * @param advertId 广告id
     * @return 落地页地址
     */
    private String queryPromoteUrlByAdvertId(Long advertId) {
        try {
            AdvertCoupon advertCoupon = serviceManager.getAdvertCouponByLocal(advertId);
            if (null != advertCoupon) {
                return advertCoupon.getPromoteURL();
            }
        } catch (Exception e) {
            logger.warn("[广告分媒体调控]查询落地页异常, advertId={}", advertId, e);
        }

        return null;
    }

    /**
     * 查询广告的默认配置id
     *
     * @param advertId 广告id
     * @return 默认配置id
     */
    private Long queryDefaultOrientId(Long advertId) {
        if (null == advertId) {
            return AdvertConstants.DEFAULT_ORIENTATION_ID;
        }

        try {
            return defaultOrientCache.get(advertId).orElse(AdvertConstants.DEFAULT_ORIENTATION_ID);
        } catch (Exception e) {
            logger.warn("[广告分媒体调控]查询广告默认配置缓存异常, advertId={}", advertId, e);
        }
        return AdvertConstants.DEFAULT_ORIENTATION_ID;
    }

    /**
     * 判断是否满足落地页条件
     *
     * @param effectCondition 生效条件
     * @param promoteUrl 落地页链接
     * @return 是否满足
     */
    private boolean checkPromoteUrl(Integer effectCondition, String promoteUrl) {
        if (null == effectCondition || StringUtils.isBlank(promoteUrl)) {
            return false;
        }

        if (Objects.equals(effectCondition, LayeredStrategyConditionEnum.ADVERTISER_LANDPAGE.getType())) {
            return isAdvertiserLandPage(promoteUrl);
        }

        return Objects.equals(effectCondition, LayeredStrategyConditionEnum.BAIQI_LANDPAGE.getType())
                && DomainUtil.isBaiqiDomain(promoteUrl);
    }

    /**
     * 判断是否是广告主的落地页
     *
     * @param promoteUrl 落地页链接
     * @return 是否是广告主落地页
     */
    private boolean isAdvertiserLandPage(String promoteUrl) {
        return StringUtils.isNotBlank(promoteUrl)
                && !DomainUtil.isJimuDomain(promoteUrl)
                && !DomainUtil.isBaiqiDomain(promoteUrl);
    }
}
