package cn.com.duiba.tuia.service.impl;

import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import cn.com.duiba.tuia.enums.CatGroupEnum;
import cn.com.duiba.wolf.redis.RedisAtomicClient;
import cn.com.duiba.wolf.redis.RedisLock;
import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import cn.com.tuia.advert.cache.CacheKeyTool;
import cn.com.tuia.advert.cache.RedisCommonKeys;
import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSONObject;
import com.github.benmanes.caffeine.cache.Caffeine;
import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import com.google.common.base.Splitter;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;

import cn.com.duiba.tuia.constants.AdvertConstants;
import cn.com.duiba.tuia.dao.advert.AdvertConsumeCurrentDAO;
import cn.com.duiba.tuia.dao.limiting.AdvertLimitingAppPackageDao;
import cn.com.duiba.tuia.dao.limiting.AdvertLimitingDAO;
import cn.com.duiba.tuia.dao.slot.AppPackageSlotDao;
import cn.com.duiba.tuia.dao.slot.OrientationAppSlotDAO;
import cn.com.duiba.tuia.domain.dataobject.AdvertLimitingDO;
import cn.com.duiba.tuia.domain.dataobject.AdvertOrientationPackageDO;
import cn.com.duiba.tuia.exception.TuiaException;
import cn.com.duiba.tuia.service.AdvertOrientationService;
import cn.com.duiba.tuia.service.LimitingMaximunService;
import cn.com.duiba.tuia.tool.CatUtil;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.tuia.advert.constants.CommonConstant;


/**
 * @author: panaihua
 * @date: 2017年03月03日 11:23
 * @descript: 限流已经达到峰值的应用缓存
 * <p>不需要初始化，如果获取是空的，说明今天还没有达到峰值的APP</p>
 * @version: 1.0
 */
@Service
public class LimitingMaximunServiceImpl implements LimitingMaximunService {

    private final Logger logger = LoggerFactory.getLogger(LimitingMaximunServiceImpl.class);


    @Resource
    private StringRedisTemplate stringRedisTemplate;

    @Autowired
    private AdvertLimitingDAO limitingDAO;

    @Autowired
    private AdvertConsumeCurrentDAO consumeCurrentDAO;

    @Autowired
    private OrientationAppSlotDAO orientationAppSlotDAO;

    @Autowired
    private AdvertOrientationService advertOrientationService;

    private final int maxCapacity = 5000;

    @Autowired
    private AdvertLimitingAppPackageDao advertLimitingAppPackageDao;

    @Autowired
    private AppPackageSlotDao appPackageSlotDao;

    @Resource
    private ExecutorService executorService;

    @Resource(name = "redisTemplate")
    private RedisAtomicClient redisAtomicClient;

    /**
     * 限流达到峰值的app只保存一天 有消息同步，不用设置过期时间来保证避免读取旧值
     */
    // true 表示预算被消耗，要被限流。
    private LoadingCache<String, Map<Long ,Boolean>> ADVERT_LIMITING_BUDGET_CACHE = CacheBuilder.newBuilder().initialCapacity(maxCapacity).
            recordStats().refreshAfterWrite(10, TimeUnit.MINUTES)
            .build(
                    new CacheLoader<String, Map<Long,Boolean>>() {
                        public Map<Long,Boolean> load(String key) {
                            List<String> strs= Splitter.on("-").splitToList(key);
                            Long advertId=Long.parseLong(strs.get(0));
                            Long orientId=Long.parseLong(strs.get(1));
                            return getAppLimitBudget(advertId, orientId);

                        }

                        @Override
                        public ListenableFuture<Map<Long,Boolean>> reload(String key, Map<Long, Boolean> oldValue) throws Exception {
                            ListenableFutureTask<Map<Long,Boolean>> task = ListenableFutureTask.create(() -> load(key));
                            executorService.submit(task);
                            return task;
                        }
                    });

    /**
     * 配置下对应的媒体预算限制
     * @param advertId
     * @param orientId
     * @return
     */
    private Map<Long ,Boolean> getAppLimitBudget(Long advertId, Long orientId) {
        List<AdvertLimitingDO> limits = limitingDAO.selectLisstByAdvertIdAndPlanId(advertId, orientId);
        if (CollectionUtils.isEmpty(limits)) {
            return Maps.newHashMap();
        }
        return limits.stream().collect(Collectors.groupingBy(AdvertLimitingDO::getAppId,
                Collectors.collectingAndThen(Collectors.toList(), limits1 -> limits1.stream().anyMatch(limit -> checkLimitingMaximunApps(limit)))));

    }

    //有消息同步，不用设置过期时间来保证避免读取旧值
    private com.github.benmanes.caffeine.cache.LoadingCache<String, Map<Long, List<AdvertLimitingDO>>> ADVERT_PLAN_CACHE = Caffeine.newBuilder().initialCapacity(maxCapacity).
            recordStats().refreshAfterWrite(15, TimeUnit.MINUTES)
            .build(
                    new com.github.benmanes.caffeine.cache.CacheLoader<String, Map<Long, List<AdvertLimitingDO>>>() {
                        public Map<Long, List<AdvertLimitingDO>> load(String key) {
                            List<String> strs= Splitter.on("-").splitToList(key);
                            Long advertId=Long.parseLong(strs.get(0));
                            Long orientId=Long.parseLong(strs.get(1));

                            // 先从缓存捞一遍
                            String redisKey = CacheKeyTool.getCacheKey(RedisCommonKeys.KC146, advertId, orientId);

                            List<AdvertLimitingDO> list1 = new ArrayList<>();
                            try {
                                String s = stringRedisTemplate.opsForValue().get(redisKey);
                                list1 = JSONObject.parseArray(s, AdvertLimitingDO.class);
                            } catch (Exception e) {
                                logger.info("ADVERT_PLAN_CACHE.getRedisValue is error, redisKey:{}", redisKey);
                            }

                            //不为空则用缓存的
                            if(org.apache.commons.collections.CollectionUtils.isNotEmpty(list1)){

                                return getLongListMap(list1);

                            }

                            //redis为空，则从数据库里捞一下
                            List<AdvertLimitingDO> list = limitingDAO.selectLisstByAdvertIdAndPlanId(advertId, orientId);

                            //加1秒锁，重新再放到redis,下次用
                            try(RedisLock lock = redisAtomicClient.getLock(redisKey, 1)) {
                                if(lock != null){
                                    stringRedisTemplate.opsForValue().set(redisKey, JSONObject.toJSONString(list), 10, TimeUnit.MINUTES);
                                }
                            } catch (Exception e) {
                                logger.error("ADVERT_PLAN_CACHE.settRedisValue is error, redisKey:{}", redisKey);
                            }


                            return getLongListMap(list);
                        }
                    });

    @NotNull
    private Map<Long, List<AdvertLimitingDO>> getLongListMap(List<AdvertLimitingDO> list) {
        Map<Long, List<AdvertLimitingDO>> listMap = new HashMap<>();
        if(!CollectionUtils.isEmpty(list)){
            list.forEach(dto->{
                // 1.遍历循环list，获取一个appId为Key的Map
                // 缓存结果对list进行了分组，就相当于把计算放在了缓存之前
                Long appId = dto.getAppId();

                // 2.从map中获取，保存了的list
                List<AdvertLimitingDO> doList = listMap.get(appId);
                if (CollectionUtils.isEmpty(doList)) {
                    doList = new ArrayList<>();
                }

                // 3.计算，doList.get(0)第一个元素的getIsSlotAll==1的（或者都为0）
                // list只要用1个有效的就可以了
                if (doList.size() == 0) {
                    doList.add(dto);
                }else {
                    // 保证list.get(0).getIsSlotAll是为1的
                    Integer isSlotAll = dto.getIsSlotAll();
                    if (null != isSlotAll && isSlotAll.equals(CommonConstant.YES)) {
                        doList.set(0, dto);
                    }
                }

                listMap.put(appId, doList);
            });
        }
        return listMap;
    }


    /**
     * 限流广告位缓存 有消息同步，不用设置过期时间来保证避免读取旧值
     */
    private LoadingCache<String, Map<Long, Boolean>> ADVERT_SLOT_LIMIT_CACHE = CacheBuilder.newBuilder().initialCapacity(maxCapacity).
            recordStats().refreshAfterWrite(15, TimeUnit.MINUTES)
            .build(
                    new CacheLoader<String, Map<Long, Boolean>>() {
                        public Map<Long, Boolean> load(String key) {
                            List<String> strs= Splitter.on("-").splitToList(key);
                            Long advertId=Long.parseLong(strs.get(0));
                            Long initialOrientationId=Long.parseLong(strs.get(1));

                            //先从redis缓存取下
                            Map<Long, Boolean> slotLimitMap = getSlotLimitMap(advertId, initialOrientationId);

                            if(slotLimitMap != null){
                                return slotLimitMap;
                            }

                            //如果缓存没有，则再找下数据库
                            Map<Long, Boolean> slotLimit = getSlotLimit(advertId, initialOrientationId);

                            //放入到缓存，给下次用
                            String redisKey = CacheKeyTool.getCacheKey(RedisCommonKeys.KC147, advertId, initialOrientationId);
                            //加1秒锁，重新再放到redis,下次用
                            try(RedisLock lock = redisAtomicClient.getLock(redisKey, 1)) {
                                if(lock != null){
                                    stringRedisTemplate.opsForValue().set(redisKey, JSONObject.toJSONString(slotLimit), 10, TimeUnit.MINUTES);
                                }
                            } catch (Exception e) {
                                logger.error("ADVERT_SLOT_LIMIT_CACHE.settRedisValue is error, redisKey:{}", redisKey);
                            }

                            return slotLimit;

                        }

                        @Override
                        public ListenableFuture<Map<Long, Boolean>> reload(String key, Map<Long, Boolean> oldValue) throws Exception {
                            ListenableFutureTask<Map<Long, Boolean>> task = ListenableFutureTask.create(() -> load(key));
                            executorService.submit(task);
                            return task;
                        }
                    });


    /**
     * 从redis缓存中获取需要限流的广告位
     * @param advertId
     * @param initialOrientationId
     * @return
     */
    private Map<Long, Boolean> getSlotLimitMap(Long advertId, Long initialOrientationId) {

        try {

            // 先从缓存捞一遍
            String redisKey = CacheKeyTool.getCacheKey(RedisCommonKeys.KC147, advertId, initialOrientationId);
            String s = stringRedisTemplate.opsForValue().get(redisKey);

            JSONObject jsonObject = JSONObject.parseObject(s);


            if (jsonObject == null) {
                return null;
            }

            //转成map
            Map<Long, Boolean> LimitMap = new HashMap<>();
            for (Object key : jsonObject.keySet()) {

                Boolean o = (Boolean) jsonObject.get(key);

                LimitMap.put(Long.valueOf(key.toString()), o);
            }

            return LimitMap;
        } catch (Exception e) {
            logger.error("ADVERT_SLOT_LIMIT_CACHE.getSlotLimitMap is error, advertId:{}, pkgId:{}", advertId, initialOrientationId, e);

            return null;
        }
    }

    /**
     * 配置下对应的广告位限流
     * @param advertId
     * @param initialOrientationId
     * @return
     */
    private Map<Long, Boolean> getSlotLimit(Long advertId, Long initialOrientationId) {
        // 单独限流的广告位
        List<Long> limitSlotList = orientationAppSlotDAO.selectByPlanIdAndAppId(advertId, initialOrientationId, null);
        Map<Long, Boolean> slotLimit = Maps.newHashMap();
        limitSlotList.forEach(slotId -> {
            slotLimit.put(slotId, true);
        });
        //获取配置绑定的限流包
        List<Long> advertLimitPackageIds = advertLimitingAppPackageDao.selectByOrientPkgId(initialOrientationId);
        advertLimitPackageIds = Optional.ofNullable(advertLimitPackageIds).orElse(Collections.emptyList());

        //获取限流包绑定的广告位
        List<Long> packageSlotIds = appPackageSlotDao.selectSlotsByPackageIds(advertLimitPackageIds);
        packageSlotIds.forEach(slotId -> {
            slotLimit.put(slotId, true);
        });

        return slotLimit;
    }

    private <K,V> void putHashMapCache(LoadingCache<String,Map<K,V>> cache,String firstkey, K key,V val) throws Exception {
        Map<K, V> secondKV = cache.getIfPresent(firstkey);
        if (secondKV == null) {
            secondKV = Maps.newConcurrentMap();
            secondKV.put(key, val);
            cache.put(firstkey, secondKV);
        }
        secondKV.put(key,val);
    }

    /**
     * planId针对默认配置包级别，用的是原始值
     * getAdvertLimitSlotCache:(这里用一句话描述这个方法的作用). <br/>
     * @author zp
     * @param advertId
     * @param initialOrientationId
     * @param slotId
     * @return
     * @since JDK 1.6
     */
    public Boolean getAdvertLimitSlotCache(Long advertId, Long initialOrientationId, Long slotId){

        String key = formatAdvertOrientationCacheKey(advertId,initialOrientationId);
        try {
            Map<Long, Boolean> slotLimtMap = ADVERT_SLOT_LIMIT_CACHE.getIfPresent(key);
            if (slotLimtMap != null) {
                return slotLimtMap.get(slotId) == null ? false: slotLimtMap.get(slotId);
            }
//            slotLimtMap = getSlotLimit(advertId, initialOrientationId);
            slotLimtMap = CatUtils.executeInCatTransaction(()->getSlotLimit(advertId, initialOrientationId),"memoryFilter","getSlotLimit");

            ADVERT_SLOT_LIMIT_CACHE.put(key, slotLimtMap);
            return slotLimtMap.get(slotId) == null ? false: slotLimtMap.get(slotId);

        } catch (Throwable e) {
            logger.error("getAdvertLimitSlotCache error",e);
            // 如果发生异常，这次先限流。下次再正常判断，否则会出现无法限流的问题
            return true;
        }
    }

    /**
     * 返回的list，实际有效的只有list.get(0)第一个判断getIsSlotAll是否为1
     * @param advertId
     * @param planId
     * @param appId 媒体id
     * @return
     */
    @Override
    public List<AdvertLimitingDO> getAdvertLimitCache(Long advertId,Long planId,Long appId){
        try {
            String key = formatAdvertOrientationCacheKey(advertId,planId);
            Map<Long, List<AdvertLimitingDO>> longListMap = ADVERT_PLAN_CACHE.get(key);
            if (CollectionUtil.isEmpty(longListMap)) {
                // 判空处理
                return new ArrayList<>();
            }
            return longListMap.get(appId) != null ? longListMap.get(appId) : new ArrayList<>();
        } catch (Exception e) {
            logger.error("LimitingMaximunServiceImpl.getAdvertLimitCache",e);
        }
        return Collections.emptyList();
    }

    private List<AdvertLimitingDO> getAppLimitByAppId(Long advertId,Long planId,Long appId){
        return getAdvertLimitCache(advertId,planId,appId);
    }

    @Override
    public void updateLimitingMaximunCache(Long advertId, Long orientationPackageId, Long appId) {
        try{
            if (appId == null) {
                Long msgPackageId = orientationPackageId;
                String key = formatAdvertOrientationCacheKey(advertId,orientationPackageId);
                //更新广告配置包的限流信息缓存
                ADVERT_LIMITING_BUDGET_CACHE.refresh(key);
                //更新达到限流app的缓存
                ADVERT_PLAN_CACHE.refresh(key);
                //7.更新广告位缓存
                if(orientationPackageId.equals(AdvertConstants.DEFAULT_ORIENTATION_ID)) {
                    AdvertOrientationPackageDO packageDO = advertOrientationService.getOrientation(advertId, AdvertConstants.DEFAULT_ORIENTATION_ID);
                    orientationPackageId = packageDO.getId();
                }
                if (orientationPackageId == null) {
                    logger.info("updateLimitingMaximunCache getOrientation is null,advertId:{},packageId:{}", advertId, msgPackageId);
                    CatUtil.log(CatGroupEnum.CAT_107009.getCode());
                    return;
                }
                ADVERT_SLOT_LIMIT_CACHE.refresh(formatAdvertOrientationCacheKey(advertId,orientationPackageId));
            } else {//billing发送的消息，代表billing计算后已经达到限制
                String key = formatAdvertOrientationCacheKey(advertId,orientationPackageId);
                putHashMapCache(ADVERT_LIMITING_BUDGET_CACHE,key,appId,true);
            }
        }catch(Exception e){
            logger.error(e.getMessage(),e);
        }
    }

    @Override
    @java.lang.SuppressWarnings("squid:S3776")
    public void initLimiting(List<Long> advertIds) throws TuiaException {
        List<AdvertLimitingDO> advertLimitingDOS = limitingDAO.selectListByAdvertIdList(advertIds);

        String currentDay = DateUtils.getDayStr(new Date());

        Map<String,Map<Long, Boolean>> limitValidMap = Maps.newConcurrentMap();
        Map<String,List<AdvertLimitingDO>> limitListMap = Maps.newConcurrentMap();

        for (AdvertLimitingDO advertLimitingDO : advertLimitingDOS) {

            String key = formatAdvertOrientationCacheKey(advertLimitingDO.getAdvertId(), advertLimitingDO.getOrientationPackageId());

            List<AdvertLimitingDO> limitList = limitListMap.get(key);
            if(null == limitList){
                limitList = Lists.newArrayList();
                limitListMap.put(key, limitList);
            }
            limitList.add(advertLimitingDO);

            boolean valid = false;
            if (advertLimitingDO != null && this.isMaximun(advertLimitingDO, currentDay)) {
                valid = true;
            }
            Map<Long,Boolean> map = limitValidMap.get(key);
            if(null == map || map.isEmpty()){
                map = Maps.newConcurrentMap();
                limitValidMap.put(key, map);
            }
            map.put(advertLimitingDO.getAppId(), valid);

        }
        ADVERT_LIMITING_BUDGET_CACHE.putAll(limitValidMap);

        // 计算分组处理，将获取的list，按新的方案，进行计算分组
        Map<String,Map<Long, List<AdvertLimitingDO>>> limitListMapGroup = Maps.newConcurrentMap();
        for (String key : limitListMap.keySet()) {
            List<AdvertLimitingDO> limitDoS = limitListMap.get(key);
            Map<Long, List<AdvertLimitingDO>> longListMap = getLongListMap(limitDoS);
            limitListMapGroup.put(key, longListMap);
        }

        logger.info("LimitingMaximunServiceImpl.initLimiting limitListMapGroup.size(): {}", limitListMapGroup.size());

        ADVERT_PLAN_CACHE.putAll(limitListMapGroup);
    }


    @Override
    public boolean checkLimitingMaximunApps(AdvertLimitingDO advertLimitingDO) {
        try {
            if(null == advertLimitingDO){
                return false;
            }
            String currentDay = DateUtils.getDayStr(new Date());
            boolean valid = false;
            if (advertLimitingDO != null && this.isMaximun(advertLimitingDO, currentDay)) {
                valid = true;
            }
            return valid;
        } catch (TuiaException e) {
            logger.error(e.getMessage(), e);
            return false;
        }
    }

    /**
     * 媒体广告位是否被限流，true表示被限流。
     * @param advertId
     * @param orientationPackageId 配置id
     * @param appId
     * @param slotId
     * @param initialOrientationId 原始配置id
     * @return
     */
    @Override
    @SuppressWarnings("squid:S3776")
    public boolean hasInLimitingMaximunApps(Long advertId,final Long orientationPackageId, Long appId, Long slotId, Long initialOrientationId) {
        try {
            // 1. 根据媒体id获取，限流媒体信息
            List<AdvertLimitingDO> limits = getAppLimitByAppId(advertId, orientationPackageId, appId);
            if(limits == null || limits.isEmpty()){
                return false;
            }

            // 2. 广告是否到峰值，true表示达到预算限制
            boolean valid = getLimitingMaximunApps(advertId, orientationPackageId, appId);

            // 3. 广告位不为空，且限流媒体中，任意一个媒体所有的广告位都开启了限流
            if (slotId == null || limits.get(0).getIsSlotAll().equals(CommonConstant.YES)) {
                return valid;
            }

            // 4. 如果广告达到峰值，就返回默认配置包判断？？？
            if(valid){
                return getAdvertLimitSlotCache(advertId,initialOrientationId, slotId);
            }
            return valid;
        } catch (TuiaException e) {
            logger.error("判断广告限流峰值异常，app:{},advertId:{},异常：", appId, advertId, e);
            return false;
        }
    }

    /**
     * 获取广告限流时达到峰值的应用，true表示达到预算限制
     *
     * @param advertId
     * @return
     * @throws TuiaException
     */
    private boolean getLimitingMaximunApps(final Long advertId, final Long orientationPackageId, final Long appId)
            throws TuiaException {
        try {
            String key = formatAdvertOrientationCacheKey(advertId,orientationPackageId);
            Map<Long ,Boolean> limitAppBudgetMap = ADVERT_LIMITING_BUDGET_CACHE.getIfPresent(key);
            if (limitAppBudgetMap != null) {
                return limitAppBudgetMap.get(appId) == null ? false: limitAppBudgetMap.get(appId);
            }
            limitAppBudgetMap = CatUtils.executeInCatTransaction(()->getAppLimitBudget(advertId, orientationPackageId),"memoryFilter","getAppLimitBudget");
//            limitAppBudgetMap = getAppLimitBudget(advertId, orientationPackageId);
            ADVERT_LIMITING_BUDGET_CACHE.put(key, limitAppBudgetMap);
            return limitAppBudgetMap.get(appId) == null ? false: limitAppBudgetMap.get(appId);
        } catch (Throwable e) {
            // 如果发生异常，这次先限流。下次再正常判断，否则会出现无法限流的问题
            logger.error("获取广告限流时达到峰值的应用异常，app:{},advertId:{},orientationPackageId:{},异常：", appId, advertId,orientationPackageId, e);
            return true;
        }
    }

    /**
     * 判断是否达到限流峰值
     *
     * @param advertLimitingDO
     * @param currentDay
     * @return
     * @throws TuiaException
     */
    private boolean isMaximun(AdvertLimitingDO advertLimitingDO, String currentDay) throws TuiaException {

        if (!this.isTimeLimiting(advertLimitingDO))
            return false;

        if (advertLimitingDO.getDailyBudget() == AdvertConstants.ADVERT_ALL_DAY_DAILY_BUDGET) return true;

        Long totalConsume = consumeCurrentDAO.getTodayConsumeByAdvertApp(advertLimitingDO.getAdvertId(), advertLimitingDO.getAppId(), currentDay);
        return advertLimitingDO.getDailyBudget() == AdvertConstants.ADVERT_ALL_DAY_DAILY_BUDGET || advertLimitingDO.getDailyBudget().compareTo(totalConsume.intValue()) <= 0;
    }

    /**
     * 判断是否在限流时间范围内
     *
     * @param advertLimitingDO
     * @return
     */
    private boolean isTimeLimiting(AdvertLimitingDO advertLimitingDO) {

        //两个限流时间取交集
        boolean hasSetLimitingWeek = StringUtils.isNotBlank(advertLimitingDO.getCyclicityWeek());
        boolean hasSetLimitingDay = advertLimitingDO.getBeginTime() != null && advertLimitingDO.getEndTime() != null;

        //如果都没有设置，默认长期限流
        if (!hasSetLimitingDay && !hasSetLimitingWeek)
            return true;

        boolean isLimitingDay = false;
        boolean isLimitingWeek = false;

        if (hasSetLimitingDay) {
            isLimitingDay = isBetweenCurrentDate(advertLimitingDO.getBeginTime(), advertLimitingDO.getEndTime());
        }

        if (hasSetLimitingWeek) {
            isLimitingWeek = isBetweenCurrentWeek(advertLimitingDO.getCyclicityWeek());
        }

        if (hasSetLimitingDay && hasSetLimitingWeek)
            return isLimitingDay && isLimitingWeek;

        if (hasSetLimitingDay && !hasSetLimitingWeek)
            return isLimitingDay;

        if (!hasSetLimitingDay && hasSetLimitingWeek)
            return isLimitingWeek;

        return false;
    }

    /**
     * 当前时间是否在限流范围内
     *
     * @param begin
     * @param end
     * @return
     */
    private boolean isBetweenCurrentDate(Date begin, Date end) {
        Date currentDate = new Date();
        return currentDate.after(begin) && currentDate.before(end);
    }

    /**
     * 当前时间是否在周期时间范围内
     *
     * @param cyclicityWeek
     * @return
     */
    private boolean isBetweenCurrentWeek(String cyclicityWeek) {
        return cyclicityWeek.contains(String.valueOf(this.getCurrentWeek()));
    }

    /**
     * 获取当前日期星期几 从星期一开始下标为1
     *
     * @return
     */
    private int getCurrentWeek() {
        Calendar calendar = Calendar.getInstance();
        int week = calendar.get(Calendar.DAY_OF_WEEK);
        if (week == 1) return 7;
        return week - 1;
    }

    private String formatAdvertOrientationCacheKey(Long advertId,Long orientationPackageId){
        return String.format("%s-%s",advertId,orientationPackageId);
    }

}
