package cn.com.duiba.tuia.service.impl;

import cn.com.duiba.tuia.cache.ServiceManager;
import cn.com.duiba.tuia.dao.slot.OrderCustomAdvertDAO;
import cn.com.duiba.tuia.domain.dataobject.OrderCustomAdvertDO;
import cn.com.duiba.tuia.domain.enums.ABTestLayerCodeEnum;
import cn.com.duiba.tuia.service.OrderCustomAdvertService;
import cn.com.duiba.tuia.service.router.FlowRouterService;
import cn.com.tuia.advert.constants.SystemConfigKeyConstant;
import cn.com.tuia.advert.model.ObtainAdvertReq;
import com.alibaba.fastjson.JSON;
import com.duiba.tuia.abtest.api.dto.ABResponseDto;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * 券次序定制广告接口实现
 *
 * @author zhangbaiqiang
 * @date 2021/6/11
 */
@Service
public class OrderCustomAdvertServiceImpl implements OrderCustomAdvertService {

    private static final Logger log = LoggerFactory.getLogger(OrderCustomAdvertServiceImpl.class);

    @Autowired
    private ServiceManager serviceManager;
    @Autowired
    private OrderCustomAdvertDAO orderCustomAdvertDAO;
    @Resource
    private FlowRouterService flowRouterService;
    @Resource
    private ExecutorService executorService;

    /**
     * 券次序定制广告配置缓存
     */
    private final LoadingCache<Long, Map<Integer, OrderCustomAdvertDO>> orderCustomCache = CacheBuilder.newBuilder()
            .refreshAfterWrite(5, TimeUnit.MINUTES)
            .expireAfterWrite(30,TimeUnit.MINUTES)
            .build(new CacheLoader<Long, Map<Integer, OrderCustomAdvertDO>>() {

                @Override
                public Map<Integer, OrderCustomAdvertDO> load(Long key) throws Exception {
                    return queryOrderCustomMap(key);
                }

                @Override
                public ListenableFuture<Map<Integer, OrderCustomAdvertDO>> reload(Long key, Map<Integer, OrderCustomAdvertDO> oldValue) throws Exception {
                    ListenableFutureTask<Map<Integer, OrderCustomAdvertDO>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });

    @Override
    public boolean isSwitchOpen(ObtainAdvertReq req) {
        Long slotId = req.getSlotId();
        if (null == slotId) {
            return false;
        }

        try {
            String value = serviceManager.getStrValue(SystemConfigKeyConstant.ORDER_CUSTOM_ADVERT_OPEN_SLOTS);
            if (StringUtils.isBlank(value)) {
                return false;
            }
            List<Long> slotIds = JSON.parseArray(value, Long.class);
            return CollectionUtils.isNotEmpty(slotIds) && slotIds.contains(slotId) && isHitABTest(req);
        } catch (Exception e) {
            log.error("OrderCustomAdvertServiceImpl.isSwitchOpen error, slotId={}", slotId, e);
        }
        return false;
    }

    /**
     * 查询配置
     *
     * @param slotId
     * @param orderCount
     * @return
     */
    @Override
    public OrderCustomAdvertDO getConfigBySlotIdAndOrderCount(Long slotId, Integer orderCount) {
        Map<Integer, OrderCustomAdvertDO> map = queryOrderCustomMap(slotId);
        return null == map ? null : map.get(orderCount);
    }



    @Override
    public void refreshCache(Long slotId) {
        if (null != slotId) {
            orderCustomCache.refresh(slotId);
        }
    }

    /**
     * 查询券次序定制广告映射
     *
     * @param slotId 广告位Id
     * @return 发券次序-定制维度和值映射
     */
    private Map<Integer, OrderCustomAdvertDO> queryOrderCustomMap(Long slotId) {
        if (null == slotId) {
            return Collections.emptyMap();
        }
        List<OrderCustomAdvertDO> list = orderCustomAdvertDAO.selectBySlotId(slotId);
        if (CollectionUtils.isEmpty(list)) {
            return Collections.emptyMap();
        }
        Map<Integer, OrderCustomAdvertDO> map = new HashMap<>(list.size());
        for (OrderCustomAdvertDO config : list) {
            map.put(config.getCustomOrder(), config);
        }
        return map;
    }

    /**
     * 是否命中实验平台的实验组（没有实验也算命中）
     *
     * @param req 广告请求参数
     * @return 是否命中
     */
    private boolean isHitABTest(ObtainAdvertReq req) {
        // 调用实验平台，没有命中实验的情况下走正常配置的逻辑（也就是按照命中的来处理）
        ABResponseDto abResponse = flowRouterService.runABTest(req, ABTestLayerCodeEnum.ORDER_CUSTOM.getDefaultValue());

        if (null == abResponse || !abResponse.isSuccess() || CollectionUtils.isEmpty(abResponse.getResult())) {
            return true;
        }

        // 记录实验平台日志
        flowRouterService.abtestLog(req, abResponse.getResult());

        // 判断是否命中实验组
        String testValue = abResponse.getResult().get(0).getTestValue();
        return StringUtils.isNumeric(testValue) && Objects.equals(Integer.valueOf(testValue), 1);
    }
}
