package cn.com.duiba.tuia.service.impl;

import cn.com.duiba.tuia.dao.phone.PhoneDAO;
import cn.com.duiba.tuia.domain.model.PhoneCacheEntity;
import cn.com.duiba.tuia.exception.TuiaException;
import cn.com.duiba.tuia.service.PhoneLibraryService;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @author: panaihua
 * @date: 2017年02月27日 10:09
 * @descript:
 * @version: 1.0
 */
@Service("PhoneLibraryServiceAImpl")
public class PhoneLibraryServiceAImpl implements PhoneLibraryService {

    private static Logger log = LoggerFactory.getLogger(PhoneLibraryServiceAImpl.class);

    @Autowired
    private PhoneDAO phoneDAO;

    @Resource
    private ExecutorService executorService;

    /**
     * 2小时（在定时同步数据的任务之后1小时）没有读/写的数据会被清除
     */
    private final LoadingCache<String, PhoneCacheEntity> PHONE_CACHE = CacheBuilder.newBuilder().initialCapacity(1000).
            recordStats().refreshAfterWrite(2, TimeUnit.HOURS).expireAfterWrite(4, TimeUnit.HOURS).build(new CacheLoader<String, PhoneCacheEntity>() {
        @Override
        public PhoneCacheEntity load(String key) throws Exception {
            return selectPhoneInfo(key);
        }

        @Override
        public ListenableFuture<PhoneCacheEntity> reload(String key, PhoneCacheEntity oldValue) throws Exception {

            ListenableFutureTask<PhoneCacheEntity> task = ListenableFutureTask.create(() -> load(key));
            executorService.submit(task);
            return task;
        }
    });

    private PhoneCacheEntity selectPhoneInfo(String phoneType) {
        try {
            PhoneCacheEntity phone = phoneDAO.selectPhoneByType(phoneType);
            if (phone == null) {
                return new PhoneCacheEntity();
            }
            return phone;
        } catch (Exception e) {
            log.error("查询手机信息时数据库异常", e);
            return null;
        }
    }

    @Override
    public PhoneCacheEntity getPhoneInfoCacheByType(String phoneType) throws TuiaException {
        if (StringUtils.isBlank(phoneType)) {
            return null;
        }
        try {
            return PHONE_CACHE.get(phoneType);
        } catch (Exception e) {
            log.error("获取设备信息缓存异常", e);
            return null;
        }
    }

    @Override
    public Object flushPhoneInfoCache() {
        ConcurrentMap<String, PhoneCacheEntity> cacheValue = PHONE_CACHE.asMap();
        PHONE_CACHE.invalidateAll();
        return cacheValue;
    }

}
