package cn.com.duiba.tuia.service.impl;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;

import cn.com.duiba.tuia.dao.resource_tags.ResourceTagsDAO;
import cn.com.duiba.tuia.domain.dataobject.ResoureTagsDO;
import cn.com.tuia.advert.enums.ResourceTagsTypeEnum;
import cn.com.duiba.tuia.exception.TuiaException;
import cn.com.duiba.tuia.service.ResourceTagsService;

/**
 * ClassName: ResourceTagsImpl <br/>
 * Function: 资源对应标签Service <br/>
 * date: 2017年7月31日 下午5:03:00 <br/>
 *
 * @author chencheng
 */
@Service
public class ResourceTagsServiceImpl implements ResourceTagsService {

    private final Logger logger = LoggerFactory.getLogger(ResourceTagsServiceImpl.class);

    @Autowired
    private ResourceTagsDAO resourceTagsDAO;

    @Resource
    private ExecutorService executorService;

    private final static ResoureTagsDO EMPTY = new ResoureTagsDO();

    /**
     * 广告推广链接标签缓存
     * key:广告id value:标签列表
     */
    private final LoadingCache<Long, Set<String>> ADVERTS_TAGS_CACHE = CacheBuilder.newBuilder().initialCapacity(1000).
            recordStats().refreshAfterWrite(10, TimeUnit.MINUTES).expireAfterWrite(2, TimeUnit.HOURS).build(new CacheLoader<Long, Set<String>>() {
        @Override
        public Set<String> load(Long key) throws Exception {
            return getResourceTags(key, ResourceTagsTypeEnum.AD);
        }

        @Override
        public ListenableFuture<Set<String>> reload(Long key, Set<String> oldValue) throws Exception {

            ListenableFutureTask<Set<String>> task = ListenableFutureTask.create(new Callable<Set<String>>() {
                public Set<String> call() throws Exception {
                    return load(key);
                }
            });
            executorService.submit(task);
            return task;
        }
    });

    /**
     * 素材对应标签实体缓存
     *
     * key:素材id value:标签实体DO
     */
    private final LoadingCache<Long, Optional<ResoureTagsDO>> MATERIAL_TAGSDO_CACHE = CacheBuilder.newBuilder().initialCapacity(1000)
                                                                                                    .recordStats().refreshAfterWrite(10, TimeUnit.MINUTES).expireAfterWrite(2, TimeUnit.HOURS).build(this.materialTagsDOLoader());


    private CacheLoader<Long, Optional<ResoureTagsDO>>  materialTagsDOLoader() {
        return new CacheLoader<Long, Optional<ResoureTagsDO>>() {
            @Override
            public Optional<ResoureTagsDO> load(@NotNull Long key) throws Exception {
                return Optional.ofNullable(resourceTagsDAO.selectResoureTagsDOById(key, ResourceTagsTypeEnum.MATERIAL.getCode()));
            }

            @Override
            public ListenableFuture<Optional<ResoureTagsDO>> reload(Long key, Optional<ResoureTagsDO> oldValue)
                throws Exception {

                ListenableFutureTask<Optional<ResoureTagsDO>> task = ListenableFutureTask.create(() -> load(key));
                executorService.submit(task);
                return task;
            }
        };
    }

    /**
     * 根据素材id获取素材实体
     *
     * @param materialId
     * @return
     */
    @Override
    public Optional<ResoureTagsDO> getMaterialTagsDO(Long materialId){
        if(materialId == null){
            return Optional.empty();
        }

        try {
           return MATERIAL_TAGSDO_CACHE.get(materialId);
        } catch (ExecutionException e) {
            logger.error("getMaterialTagsDO error, materialId={}", materialId, e);
            return Optional.empty();
        }
    }



    /**
     * 素材对应标签缓存
     * key:素材id value:标签列表
     */
    private final LoadingCache<Long, Set<String>> MATERIAL_TAGS_CACHE = CacheBuilder.newBuilder().initialCapacity(1000).
            recordStats().refreshAfterWrite(10, TimeUnit.MINUTES).expireAfterWrite(2, TimeUnit.HOURS).build(new CacheLoader<Long, Set<String>>() {
        @Override
        public Set<String> load(Long key) throws Exception {
            Set<String> rtnTags = Sets.newConcurrentHashSet();
            rtnTags.addAll(getResourceTags(key, ResourceTagsTypeEnum.MATERIAL));
            rtnTags.addAll(getResourceTags(key, ResourceTagsTypeEnum.JIMU));
            return  rtnTags;
        }

        @Override
        public ListenableFuture<Set<String>> reload(Long key, Set<String> oldValue) throws Exception {

            ListenableFutureTask<Set<String>> task = ListenableFutureTask.create(new Callable<Set<String>>() {
                public Set<String> call() throws Exception {
                    return load(key);
                }
            });
            executorService.submit(task);
            return task;
        }
    });

    /**
     * 初始化资源对应标签列表
     */
    public void init() throws TuiaException {

        List<ResoureTagsDO> list = resourceTagsDAO.selectResoureTagsDOByType(null);

        list.forEach(resoureTagsDO -> {
            if (ResourceTagsTypeEnum.AD.getCode().equals(resoureTagsDO.getResoureType())) {
                ADVERTS_TAGS_CACHE.put(resoureTagsDO.getResoureId(), getTagNums(resoureTagsDO.getTagNums()));
            } else if (ResourceTagsTypeEnum.MATERIAL.getCode().equals(resoureTagsDO.getResoureType())
                    || ResourceTagsTypeEnum.JIMU.getCode().equals(resoureTagsDO.getResoureType())) {

                Set<String> ifPresent = MATERIAL_TAGS_CACHE.getIfPresent(resoureTagsDO.getResoureId());
                if(null == ifPresent){
                    MATERIAL_TAGS_CACHE.put(resoureTagsDO.getResoureId(), getTagNums(resoureTagsDO.getTagNums()));
                }else {
                    Set<String> data = Sets.newConcurrentHashSet();
                    data.addAll(ifPresent);
                    data.addAll(getTagNums(resoureTagsDO.getTagNums()));
                    MATERIAL_TAGS_CACHE.put(resoureTagsDO.getResoureId(),data);
                }
            }
        });
    }

    /**
     * 根据资源id和类型获取标签列表
     *
     * @param resourceId
     * @param typeEnum
     * @return
     * @throws TuiaException
     * @author chencheng
     */
    private Set<String> getResourceTags(Long resourceId, ResourceTagsTypeEnum typeEnum) throws TuiaException {
        ResoureTagsDO optional = Optional.ofNullable(resourceTagsDAO.selectResoureTagsDOById(resourceId,
                typeEnum.getCode())).orElse(EMPTY);
        return getTagNums(optional.getTagNums());
    }

    private static Set<String> getTagNums(String tagNums) {
        if (StringUtils.isBlank(tagNums)) {
            return Sets.newConcurrentHashSet();
        }
        return Arrays.asList(tagNums.split(",")).stream().distinct().collect(Collectors.toSet());
    }
    @Override
    public Set<String> getResoureTagsDOById(Long resourceId, ResourceTagsTypeEnum typeEnum) {
        try {
            switch (typeEnum) {
                case AD:
                    return ADVERTS_TAGS_CACHE.get(resourceId);
                case MATERIAL:
                    return MATERIAL_TAGS_CACHE.get(resourceId);
                default:
            }
        } catch (ExecutionException e) {
            logger.error("获取资源对应标签列表异常", e);
        }
        return null;
    }

    @Override
    public void refreshResourceCache(Long resourceId) {
        ADVERTS_TAGS_CACHE.invalidate(resourceId);
        // MATERIAL_TAGS_CACHE.invalidate(resourceId);
    }

    @Override
    public Map<Long, Set<String>> getResourceTagsMapByIdSet(Set<Long> resourceIdSet, ResourceTagsTypeEnum typeEnum) {
        try {

            Map<Long, Set<String>> resourceTagMap = Maps.newHashMapWithExpectedSize(resourceIdSet.size());
            if(CollectionUtils.isEmpty(resourceIdSet)){
                return resourceTagMap;
            }

            switch (typeEnum) {
                case AD:
                    for (Long resourceId : resourceIdSet) {
                        Set<String> adTags = ADVERTS_TAGS_CACHE.get(resourceId);
                        if (CollectionUtils.isNotEmpty(adTags)) {
                            resourceTagMap.put(resourceId, adTags);
                        }
                    }
                    return resourceTagMap;
                case MATERIAL:
                    for (Long resourceId : resourceIdSet) {
                        Set<String> materialsTags = MATERIAL_TAGS_CACHE.get(resourceId);
                        if (CollectionUtils.isNotEmpty(materialsTags)) {
                            resourceTagMap.put(resourceId, materialsTags);
                        }
                    }
                    return resourceTagMap;
                default:
                    return resourceTagMap;
            }
        } catch (Exception e) {
            logger.error("获取资源对应标签列表异常", e);
        }
        return null;
    }

}
