/**
 * Project Name:engine-service
 * File Name:ResourcesRepeatLunchServiceImpl.java
 * Package Name:cn.com.duiba.tuia.service.impl
 * Date:2019年1月11日上午10:24:57
 * Copyright (c) 2019, duiba.com.cn All Rights Reserved.
 *
*/

package cn.com.duiba.tuia.service.impl;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.tuia.cache.AdvertRepeatLunchConfigCacheService;
import cn.com.duiba.tuia.domain.dataobject.AdvertRepeatLunchConfigDO;
import cn.com.duiba.tuia.domain.model.FilterResult;
import cn.com.duiba.tuia.domain.vo.AdvertFilterVO;
import cn.com.duiba.tuia.domain.vo.ConsumerRecordVO;
import cn.com.duiba.tuia.service.ResourcesRepeatLunchService;
import cn.com.tuia.advert.enums.AdvertRepeatLunchTypeEnum;
import cn.com.tuia.advert.enums.ResourceRepeatLunchTypeEnum;


/**
 * ClassName:ResourcesRepeatLunchServiceImpl <br/>
 * Function: 资源维度重复发券逻辑. <br/>
 * Date:     2019年1月11日 上午10:24:57 <br/>
 * @author   chencheng
 * @version  
 * @since    JDK 1.8
 * @see 	 
 */
@Service
public class ResourcesRepeatLunchServiceImpl implements ResourcesRepeatLunchService {

    private final Logger logger = LoggerFactory.getLogger(ResourcesRepeatLunchServiceImpl.class);

    @Autowired
    private AdvertRepeatLunchConfigCacheService advertRepeatLunchConfigCacheService;


    private void buildResourcesRepeatLunchType(FilterResult filterResult) {

        Integer repeatLunchType;
        // 今日最后一条领券记录
        ConsumerRecordVO lastRecord = filterResult.getLastOfTodayConsumeRecord();
        // N小时内的领券记录
        List<ConsumerRecordVO> hoursConsumeList = filterResult.getHoursConsumeList();
        if (null == lastRecord || null == lastRecord.getConsumerRecordJsonVO() ||  null == lastRecord.getConsumerRecordJsonVO().getRepeatLunchType() || hoursConsumeList.isEmpty()){
            repeatLunchType = AdvertRepeatLunchTypeEnum.NEW_CUT.getCode();
        }else{
            repeatLunchType = lastRecord.getConsumerRecordJsonVO().getRepeatLunchType();
        }
        filterResult.setRepeatLunchType(Optional.ofNullable(repeatLunchType).orElse(AdvertRepeatLunchTypeEnum.NEW_CUT.getCode()));

    }

   
    @Override
    public Map<Long, AdvertFilterVO> resourcesRepeatLunch(FilterResult filterResult, Map<Long, AdvertFilterVO> validAdverts, String platform) {
        if (validAdverts.isEmpty()) {
            return validAdverts;
        }
        try {
            // 切量打标
            buildResourcesRepeatLunchType(filterResult);
            // 是否切量测试(老逻辑或今日释放，则不走资源过滤)
            if(!filterResult.getRepeatLunchType().equals(AdvertRepeatLunchTypeEnum.NEW_CUT.getCode())){
                return validAdverts;
            }
            // 资源过滤
            return filterResourceRepeatLunch(filterResult.getHoursConsumeList(), validAdverts, platform, filterResult);
        } catch (Exception e) {
            logger.warn("resourcesRepeatLunch error", e);
            return validAdverts;
        }
        
    }

    /**
     * 
     * filterResourceRepeatLunch:(资源过滤). <br/>
     *
     * @author chencheng
     * @param hoursConsumeList N小时内的发券记录
     * @param validAdverts 有效广告map
     * @param platform 操作平台类型
     * @param filterResult 
     * @return
     * @since JDK 1.8
     */
    private Map<Long, AdvertFilterVO> filterResourceRepeatLunch(List<ConsumerRecordVO> hoursConsumeList, Map<Long, AdvertFilterVO> validAdverts, String platform, FilterResult filterResult) {
        
        try {
            if (CollectionUtils.isEmpty(hoursConsumeList)) {
                return validAdverts;
            }

            // 领券记录资源对应的发券数量
            Map<String, Long> accountCountingMap = hoursConsumeList.stream().collect(Collectors.groupingBy(vo -> vo.getConsumerRecordJsonVO().getResourceTag(), Collectors.counting()));
            // 获取资源重复发券配置的规则
            Integer type = ResourceRepeatLunchTypeEnum.getByPlatform(platform);
            Map<String, AdvertRepeatLunchConfigDO> configMap = advertRepeatLunchConfigCacheService.getAdvertRepeatLunchConfigCache(type);
            // 已领取广告的资源，满足资源内发券数量限制的资源。必须用这个map去循环
            Set<String> limitResourceIds = accountCountingMap.entrySet().stream().map(map -> {
                AdvertRepeatLunchConfigDO config= getAdvertRepeatLunchConfigOrDefult(map.getKey(), configMap);
                if (null == config) {
                    return null;
                }
                if(map.getValue() >= config.getMaxLimit()){
                    return map.getKey();
                }
                // 前N个发券里有此资源，则需过滤
                Long lunchIntervalResourceSize  = hoursConsumeList.stream().limit(config.getLunchInterval()).
                    filter(vo -> vo.getConsumerRecordJsonVO().getResourceTag().equals(map.getKey()))
                    .collect(Collectors.counting());
                if (lunchIntervalResourceSize > 0 ){
                    return map.getKey();
                }
                return null;
            }).filter(Objects::nonNull).collect(Collectors.toSet());

            if(CollectionUtils.isEmpty(limitResourceIds)){
                return validAdverts;
            }
            // 过滤资源限制下的广告 todo toMap的value判空
            Map<Long, AdvertFilterVO> filterValidAdverts =  validAdverts.entrySet().stream()
                    .filter(map -> null != map && !limitResourceIds.contains(getResourceTag(map.getValue().getResourceTag())))
                    .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue, (oldVal, newVal) -> newVal));

            if (repeatLunchRelease(filterValidAdverts)) {
                filterResult.setRepeatLunchType(AdvertRepeatLunchTypeEnum.NEW_RELEASE.getCode());
                return validAdverts;
            }
            // 过滤后没有券可发时进行释放
            if (filterValidAdverts.isEmpty()) {
                filterResult.setRepeatLunchType(AdvertRepeatLunchTypeEnum.NEW_RELEASE.getCode());
                return validAdverts;
            }
            // 返回过滤后的有效广告
            return filterValidAdverts;
        } catch (Exception e) {
            logger.error("filterResourceRepeatLunch error", e);
            return validAdverts;
        }
        
       
    }

    /**
     * 
     * repeatLunchRelease:(判断过滤后广告是否满足过滤释放条件). <br/>
     *
     * @author chencheng
     * @param filterValidAdverts
     * @return
     * @since JDK 1.8
     */
    public boolean repeatLunchRelease(Map<Long, AdvertFilterVO> filterValidAdverts) {
        // 过滤后没有券可发时进行释放
        if (filterValidAdverts.isEmpty()) {
            return true;
        }
        // 过滤后是否还有付费券，如果只有免费券，则进行释放
        Long feeAdvertsCount = filterValidAdverts.values().stream().filter(item ->
            item.getManualAdvertSet().stream().anyMatch(x -> x.getFee() > 0)).collect(Collectors.counting());
        if (feeAdvertsCount.longValue() <= 0) {
            return true;
        } 
        // 返回过滤后的有效广告
        return false;
    }

    
    /**
     * 
     * getResourceTag:(获取资源标签对应的限制tag_num). <br/>
     *
     * @author chencheng
     * @param resourceTag
     * @return
     * @since JDK 1.8
     */
    private String getResourceTag(String resourceTag) {
       return StringUtils.isBlank(resourceTag) ? AdvertRepeatLunchConfigDO.BLANK_SOURCE_ID : resourceTag;
    }

    /**
     * 
     * getAdvertRepeatLunchConfigOrDefult:(获取特殊资源配置或默认资源配置). <br/>
     *
     * @author chencheng
     * @param key
     * @param configMap
     * @return
     * @since JDK 1.8
     */
    @Override
    public AdvertRepeatLunchConfigDO getAdvertRepeatLunchConfigOrDefult(String key, Map<String, AdvertRepeatLunchConfigDO> configMap) {
        AdvertRepeatLunchConfigDO config = configMap.get(key);
        if (null != config) {
            return config;
        }
        // 获取默认资源配置
        AdvertRepeatLunchConfigDO defultConfig = configMap.get(AdvertRepeatLunchConfigDO.DEFAULT_SOURCE_ID);
        if (defultConfig == null) {
            advertRepeatLunchConfigCacheService.invalidateAll();
            logger.info("advertRepeatLunchConfigCacheService get defult id null, invali date");
        }
        return defultConfig;
    }

}

