package cn.com.duiba.tuia.service.impl;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import cn.com.duiba.tuia.constants.AdvertReqLogExtKeyConstant;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.tuia.constants.CommonConstants;
import cn.com.duiba.tuia.dao.slot.SlotDirectionDAO;
import cn.com.duiba.tuia.domain.dataobject.ConsumerInteractiveRecordDO;
import cn.com.duiba.tuia.domain.dataobject.SlotDirectionDO;
import cn.com.duiba.tuia.risk.engine.api.remote.RemoteRuleEngineService;
import cn.com.duiba.tuia.risk.engine.api.req.RuleEngineParam;
import cn.com.duiba.tuia.risk.engine.api.rsp.RuleEngineResult;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.tuia.advert.model.ObtainAdvertReq;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;

/**
 * @Author:zhangmeng
 * @Date:2018/9/3
 * @Function:风控反作弊接口
 */

@Service
public class RiskChectServiceImpl {

    private static final Logger logger = LoggerFactory.getLogger(RiskChectServiceImpl.class);

    @Autowired
    private RemoteRuleEngineService remoteRuleEngineService;

    @Autowired
    private SlotDirectionDAO slotDirectionDAO;

    @Resource(name = "redisTemplate")
    private AdvancedCacheClient advancedCacheClient;

    @Resource
    private ExecutorService executorService;

    private static final String senceKey = "tuia_launch";
    private static final String source = "tuia-advert";
    private static final String rid = "_coll_rid";
    private static final String uuid6 = "_coll_uuid6";


    /**
     * 定向导量广告位缓存
     */
    private final LoadingCache<String, Map<Long, SlotDirectionDO>> SLOT_DIRCTION_CACHE = CacheBuilder.newBuilder()
            .refreshAfterWrite(5, TimeUnit.MINUTES).expireAfterWrite(1, TimeUnit.HOURS).build(new CacheLoader<String, Map<Long, SlotDirectionDO>>(){

                @Override
                public Map<Long, SlotDirectionDO> load(String key) throws Exception {
                    List<SlotDirectionDO> list = slotDirectionDAO.selectAllSlotDirection();
                    if (CollectionUtils.isEmpty(list)) {
                        return Maps.newHashMap();
                    }
                    return list.stream().filter(Objects::nonNull).collect(Collectors.toMap(SlotDirectionDO::getSlotId, Function.identity(), (k1, k2) -> {return k1;}));
                }

                @Override
                public ListenableFuture<Map<Long, SlotDirectionDO>> reload(String key, Map<Long, SlotDirectionDO> oldValue) throws Exception {
                    ListenableFutureTask<Map<Long, SlotDirectionDO>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });


    public boolean getRiskCheatStatus(ObtainAdvertReq req) {

        try {
            RuleEngineParam ruleEngineParam = new RuleEngineParam();
            //必传参数
            ruleEngineParam.setAdvertOrderId(req.getOrderId());
            ruleEngineParam.setConsumerId(req.getConsumerId().toString());
            ruleEngineParam.setDeviceId(req.getDeviceId());
            ruleEngineParam.setSceneKey(senceKey);

            ruleEngineParam.setIp(req.getIp());
            ruleEngineParam.setUa(req.getUa());
            ruleEngineParam.setPlatform(req.getOs());
            ruleEngineParam.setDate(new Date());
            ruleEngineParam.setMediaId(req.getAppId().toString());
            ruleEngineParam.setSoltId(req.getSlotId().toString());
            ruleEngineParam.setSource(source);
            Optional.ofNullable(req.getLogExtMap()).ifPresent(logExt -> {
                ruleEngineParam.setRid(logExt.get(rid));
                ruleEngineParam.setUuid6(logExt.get(uuid6));

                ruleEngineParam.setImei(logExt.get(AdvertReqLogExtKeyConstant.IMEI));
                ruleEngineParam.setImeiMd5(logExt.get(AdvertReqLogExtKeyConstant.IMEI_MD5));
                ruleEngineParam.setIdfa(logExt.get(AdvertReqLogExtKeyConstant.IDFA));
                ruleEngineParam.setIdfaMd5(logExt.get(AdvertReqLogExtKeyConstant.IDFA_MD5));
                ruleEngineParam.setOaid(logExt.get(AdvertReqLogExtKeyConstant.OAID));
                ruleEngineParam.setOaidMd5(logExt.get(AdvertReqLogExtKeyConstant.OAID_MD5));
            });
            RuleEngineResult ruleEngineResult = remoteRuleEngineService.operator(ruleEngineParam);
            //获取风控返回的结果
            if (ruleEngineResult != null) {
                if (!ruleEngineResult.isSuccess()) {
                    logger.info("风控失败原因:{}", ruleEngineResult.getDesc());
                }
                return ruleEngineResult.isReject();
            }
            return false;
        } catch (Exception e) {
            logger.error("调用风控反作弊接口异常！");
            return false;
        }
    }

    /**
     * 校验当前流量是否满足风控广告位定向导量条件，校验包含
     * 1、当前广告位在定向导量广告位列表
     * 2、该用户当天没有参与过定向导量广告
     * 3、命中定向导量广告位概率
     *
     * @param req                  请求参数
     * @param todayConsumerVOList  用户今日领取记录
     * @return ture-满足条件 false-不满足条件
     */
    public boolean validateSoltDirection(ObtainAdvertReq req, List<ConsumerInteractiveRecordDO> todayConsumerVOList){

        try {
            DBTimeProfile.enter("validateSoltDirection");

            // 1、判断当前广告位是否在定向导量广告位的列表
            Map<Long, SlotDirectionDO> slotDirectionMap = SLOT_DIRCTION_CACHE.getUnchecked("");
            SlotDirectionDO slotDirectionDO = slotDirectionMap.get(req.getSlotId());
            if (slotDirectionDO == null) {
                return false;
            }

            /* 2、判断用户今天是否参与过定向导量
            if (CollectionUtils.isNotEmpty(todayConsumerVOList)) {
                for (ConsumerInteractiveRecordDO consumerVO : todayConsumerVOList) {
                    if (StringUtils.equals(CommonConstants.HIT_SLOT_DIRECTION, 
                          Optional.ofNullable(consumerVO.getDirectSolt()).map(directSolt -> directSolt.toString())
                          .orElse(CommonConstants.NOT_SLOT_DIRECTION))) {
                        return false;
                    }
                }
            }*/

            // 3、命中定向导量广告位概率
            double rate = new BigDecimal(String.valueOf(slotDirectionDO.getRate()))
                    .divide(new BigDecimal("10000")).doubleValue();
            double randomNum = new Random().nextDouble();

            return rate >= randomNum;

        } catch (Exception e){
            logger.error("校验定向导量广告条件异常, slotId = {}, consumerId = {} ", req.getSlotId(), req.getConsumerId(), e);
            return false;
        } finally {
            DBTimeProfile.release();
        }
    }



}
