/**
 * Project Name:engine-service
 * File Name:TradeAppRepeatLunchServiceImpl.java
 * Package Name:cn.com.duiba.tuia.service.impl
 * Date:2019年3月6日下午5:47:19
 * Copyright (c) 2019, duiba.com.cn All Rights Reserved.
 *
*/

package cn.com.duiba.tuia.service.impl;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.tuia.cache.AdvertRepeatLunchConfigCacheService;
import cn.com.duiba.tuia.domain.dataobject.AdvertRepeatLunchConfigDO;
import cn.com.duiba.tuia.domain.model.FilterResult;
import cn.com.duiba.tuia.domain.vo.AdvertFilterVO;
import cn.com.duiba.tuia.domain.vo.ConsumerRecordVO;
import cn.com.duiba.tuia.service.ResourcesRepeatLunchService;
import cn.com.duiba.tuia.service.TradeAppRepeatLunchService;
import cn.com.tuia.advert.enums.AdvertRepeatLunchTypeEnum;
import cn.com.tuia.advert.enums.ResourceRepeatLunchTypeEnum;

import com.google.common.collect.Lists;

/**
 * ClassName:TradeAppRepeatLunchServiceImpl <br/>
 * Function: 行业分媒体维度重复发券逻辑. <br/>
 * Date:     2019年3月6日 下午5:47:19 <br/>
 * @author   chencheng
 * @version  
 * @since    JDK 1.8
 * @see 	 
 */
@Service
public class TradeAppRepeatLunchServiceImpl implements TradeAppRepeatLunchService{
    
    private final Logger logger = LoggerFactory.getLogger(TradeAppRepeatLunchServiceImpl.class);
    
    @Autowired
    private ResourcesRepeatLunchService resourcesRepeatLunchService;
    
    @Autowired
    private AdvertRepeatLunchConfigCacheService advertRepeatLunchConfigCacheService;

    public void buildTradeAppRepeatLunchType(FilterResult filterResult) {

        Integer repeatLunchType;
        // 今日最后一条领券记录
        ConsumerRecordVO lastRecord = filterResult.getLastOfTodayConsumeRecord();
        // N小时内的领券记录
        List<ConsumerRecordVO> hoursConsumeList = filterResult.getHoursConsumeList();
        if (null == lastRecord || null == lastRecord.getConsumerRecordJsonVO()
                ||  null == lastRecord.getConsumerRecordJsonVO().getTat() || hoursConsumeList.isEmpty()){
            repeatLunchType = AdvertRepeatLunchTypeEnum.NEW_CUT.getCode();
        }else{
            repeatLunchType = lastRecord.getConsumerRecordJsonVO().getTat();
        }
        filterResult.setTradeAppLunchType(Optional.ofNullable(repeatLunchType).orElse(AdvertRepeatLunchTypeEnum.NEW_CUT.getCode()));


    }

    @Override
    public Map<Long, AdvertFilterVO> tradeAppRepeatLunch(FilterResult filterResult,
                                                         Map<Long, AdvertFilterVO> validAdverts){
//        LogInfoPrintTools.log("0.1.开始校验");
        if (validAdverts.isEmpty()) {
            return validAdverts;
        }
//        LogInfoPrintTools.log("0.2.结束校验，开始判断是否开启切量测试");

        try {
           
            // 是否切量测试(老逻辑或今日释放，则不走行业标签过滤)
            if(!filterResult.getTradeAppLunchType().equals(AdvertRepeatLunchTypeEnum.NEW_CUT.getCode())){
                return validAdverts;
            }
//            LogInfoPrintTools.log("0.3.判断是否开启切量测试结束,");

            // 资源过滤
            return filterTradeAppLunch(filterResult.getHoursConsumeList(), validAdverts, filterResult);
        } catch (Exception e) {
            logger.warn("tradeAppRepeatLunch error", e);
            return validAdverts;
        }
    }

    /**
     * 
     * filterTradeAppLunch:行业标签过滤. <br/>
     *
     * @author chencheng
     * @param hoursConsumeList N小时内的发券记录
     * @param validAdverts 有效广告map
     * @param filterResult 
     * @return
     * @since JDK 1.8
     */
    private Map<Long, AdvertFilterVO> filterTradeAppLunch(List<ConsumerRecordVO> hoursConsumeList, Map<Long, AdvertFilterVO> validAdverts, FilterResult filterResult) {
        
        try {
            if (CollectionUtils.isEmpty(hoursConsumeList)) {
                return validAdverts;
            }
//            LogInfoPrintTools.log("0.3.1");
            // 获取行业标签重复发券配置的规则
            Map<String, AdvertRepeatLunchConfigDO> configMap = advertRepeatLunchConfigCacheService.getAdvertRepeatLunchConfigCache(ResourceRepeatLunchTypeEnum.TRADE_APP_DEFULT.getCode());
            AdvertRepeatLunchConfigDO repeatLunchConfig = resourcesRepeatLunchService.getAdvertRepeatLunchConfigOrDefult(filterResult.getAppId().toString(), configMap);
            if (repeatLunchConfig == null) {
                return validAdverts;
            }
//            LogInfoPrintTools.log("0.3.2");

            //log
            // 领券记录对应的行业标签发券数量
            Map<String, Long> matchTagLunchCountMap = hoursConsumeList.stream().collect(Collectors.groupingBy(vo -> vo.getTags(), Collectors.counting()));
            //

//            LogInfoPrintTools.log("0.3.3");
            // 要过滤的行业
            List<String> limitTradeTag = Lists.newArrayList();
            // 到达发券上限的行业
            limitTradeTag.addAll(getMaxLimitTarde(matchTagLunchCountMap, repeatLunchConfig.getMaxLimit()));
//            LogInfoPrintTools.log("0.3.4");
            // 超过发券间隔行业。（领券记录倒序后，前N个广告对应的行业不能投放）
            limitTradeTag.addAll(getLunchInterval(hoursConsumeList, repeatLunchConfig.getLunchInterval()));
//            LogInfoPrintTools.log("0.3.5");
            if(CollectionUtils.isEmpty(limitTradeTag)){
                return validAdverts;
            }
            
            // 过滤行业标签受限的广告
            Map<Long, AdvertFilterVO> filterValidAdverts =  validAdverts.entrySet().stream()
                    .filter(map -> null != map && null != map.getValue() && !limitTradeTag.contains(map.getValue().getMatchTags()))
                    .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue, (oldVal, newVal) -> newVal));
//            LogInfoPrintTools.log("0.3.6");
            if (resourcesRepeatLunchService.repeatLunchRelease(filterValidAdverts)) {
                filterResult.setTradeAppLunchType(AdvertRepeatLunchTypeEnum.NEW_RELEASE.getCode());
                return validAdverts;
            }
//            LogInfoPrintTools.log("0.3.7");
            // 返回过滤后的有效广告
            return filterValidAdverts;
        } catch (Exception e) {
            logger.error("filterResourceRepeatLunch error", e);
            return validAdverts;
        }


    }

    /**
     * 
     * getLunchInterval:(获取超过发券间隔行业). <br/>
     *
     * @author chencheng
     * @param hoursConsumeList
     * @param lunchInterval
     * @return
     * @since JDK 1.8
     */
    private List<String> getLunchInterval(List<ConsumerRecordVO> hoursConsumeList, Integer lunchInterval) {
        // 不限
        if (lunchInterval == null || lunchInterval == 0) {
            return Lists.newArrayList();
        }
        return hoursConsumeList.stream().limit(lunchInterval).map(ConsumerRecordVO::getTags).collect(Collectors.toList());
    }

    /**
     * 
     * getMaxLimitTarde:(到达发券上限的行业). <br/>
     *
     * @author chencheng
     * @param matchTagLunchCountMap
     * @param maxLimit
     * @return
     * @since JDK 1.8
     */
    private List<String> getMaxLimitTarde(Map<String, Long> matchTagLunchCountMap, Integer maxLimit) {
        // 不限
        if (maxLimit == null || maxLimit == 0) {
            return Lists.newArrayList();
        }
        return matchTagLunchCountMap.entrySet().stream().filter(entity -> entity.getValue().intValue() >= maxLimit).map(Map.Entry::getKey).collect(Collectors.toList());   
    }
    
}

