package cn.com.duiba.tuia.service.impl;

import static java.util.stream.Collectors.toList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.tuia.dao.engine.TradeTagRuleDAO;
import cn.com.duiba.tuia.dao.tag.NewTagDAO;
import cn.com.duiba.tuia.domain.dataobject.NewTagDO;
import cn.com.duiba.tuia.domain.dataobject.TradeTagRuleScopeDO;
import cn.com.duiba.tuia.domain.model.TagRuleScope;
import cn.com.duiba.tuia.enums.NewTagsNumEnum;
import cn.com.duiba.tuia.service.BaseService;
import cn.com.duiba.tuia.service.TradeTagRuleService;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;

/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2018年02月09日 14:52
 * @descript:
 * @version: 1.0
 */
@Service
public class TradeTagRuleServiceImpl extends BaseService implements TradeTagRuleService, InitializingBean {

    @Autowired
    private TradeTagRuleDAO tradeTagRuleDAO;

    @Autowired
    private NewTagDAO newTagDAO;
    
    private final Long CACHE_KEY = 1L;

    @Resource
    private ExecutorService executorService;

    private final LoadingCache<Long, TagRuleScope> TAG_RULE_SCOPE_CACHE = CacheBuilder.newBuilder().initialCapacity(100).
            recordStats().refreshAfterWrite(30, TimeUnit.MINUTES).expireAfterWrite(40, TimeUnit.MINUTES).build(new CacheLoader<Long, TagRuleScope>() {
        @Override
        public TagRuleScope load(Long key) {

            TagRuleScope tagRuleScope = new TagRuleScope();
            List<TradeTagRuleScopeDO> tags = selectTagScopeAll();
            if (CollectionUtils.isEmpty(tags)) {
                tagRuleScope.setTags(Collections.emptyList());
                return tagRuleScope;
            }
            
            tagRuleScope.setTags(tags);
            Map<String, TradeTagRuleScopeDO> tagRuleDOMap = tags.stream().filter(Objects::nonNull)
                    .collect(Collectors.toMap(TradeTagRuleScopeDO::getTagId, ruleDO -> ruleDO, (oldVal, newVal) -> newVal));
            tagRuleScope.setTagRuleDOMap(tagRuleDOMap);
            
            // 区分一级、二级标签
            Map<Boolean, List<TradeTagRuleScopeDO>> tagLevelMap = tags.stream().collect(Collectors.partitioningBy(scoptDO -> scoptDO.getTagLevel() == 1));
            tagRuleScope.setLevelOneTagNum(tagLevelMap.get(true).stream().map(TradeTagRuleScopeDO::getTagId).collect(toList()));
            tagRuleScope.setLevelTwoTagNum(tagLevelMap.get(false).stream().map(TradeTagRuleScopeDO::getTagId).collect(toList()));
            // 获取一级标签对应的二级标签关系
            Map<String, List<String>> allTagNum = getAallTagNum();
            
            //获取其他无规则分，用户对应所有的标签对应的标签规则，含有空规则的标签
            tagRuleScope.setUserAllRoleTagNum(getUserAllRoleOtherInsert(allTagNum, tagRuleScope));
            
            // 用来查询规则的标签，就是数据库里的tagnum,不再区分一二级
            List<String> tagNums = Lists.newArrayList();
            tagNums.addAll(tagRuleScope.getLevelOneTagNum());
            tagNums.addAll(tagRuleScope.getLevelTwoTagNum());
            
            tagRuleScope.setTagNums(tagNums);
            
            return tagRuleScope;
        }
       
        /**
         * 
         * getOtherRoleTagNums:(用户对应所有的标签对应的标签规则，含有空规则的标签). <br/>
         *
         * @author chencheng
         * @param allTagNum
         * @param tagRuleScope
         * @return
         * @since JDK 1.8
         */
        private List<String> getUserAllRoleOtherInsert(Map<String, List<String>> allTagNum, TagRuleScope tagRuleScope) {
            
            // 除了有规则的一级标签，其他的用二级标签打分
            List<String> allLevelTowTag = allTagNum.entrySet().stream().filter(tagMap -> !tagRuleScope.getLevelOneTagNum().contains(tagMap.getKey()))
            .map(Map.Entry::getValue).flatMap(List::stream).collect(toList());
            // 加上一级标签的tagNum
            allLevelTowTag.addAll(tagRuleScope.getLevelOneTagNum());
            return allLevelTowTag;
        }

        
        @Override
        public ListenableFuture<TagRuleScope> reload(Long key, TagRuleScope oldValue) {

            ListenableFutureTask<TagRuleScope> task = ListenableFutureTask.create(() -> load(key));
            executorService.submit(task);
            return task;
        }
    });
    
    
    /**
     * 
     * getAallTagNum:(查询所有的二级行业标签,构建一级行业标签tagNum对应的二级行业标签tagNum). <br/>
     *
     * @author chencheng
     * @return
     * @since JDK 1.8
     */
    private Map<String, List<String>> getAallTagNum() {
        
        // 查询所有的二级行业标签，构建一级行业标签tagNum对应的二级行业标签tagNum
        List<NewTagDO> listTag = newTagDAO.selectNewTagList(NewTagsNumEnum.ADVERT_TAG.getCode(), 2);
        Map<String, List<String>> tagMap = Maps.newHashMap();
        if (CollectionUtils.isEmpty(listTag)) {
            return tagMap;
        }
        listTag.forEach(tag -> {
            String parentNum = tag.getTagNum().substring(0, 5);
            List<String> tagTwoLevelNums = tagMap.get(parentNum);
            if (CollectionUtils.isEmpty(tagTwoLevelNums)) {
                tagTwoLevelNums =Lists.newArrayList();
            }
            tagTwoLevelNums.add(tag.getTagNum());
            tagMap.put(parentNum, tagTwoLevelNums);
        });
        
        return tagMap;
    }

    @Override
    public TagRuleScope getTagScopeAll() {
        return TAG_RULE_SCOPE_CACHE.getUnchecked(CACHE_KEY);
    }

    @Override
    public void afterPropertiesSet() {
        TAG_RULE_SCOPE_CACHE.getUnchecked(CACHE_KEY);
    }

    private List<TradeTagRuleScopeDO> selectTagScopeAll() {
        return tradeTagRuleDAO.selectTagScopeAll();
    }
   
}
