/**
 * Project Name:tuia-core-api<br>
 * File Name:DuiBaItemClient.java<br>
 * Package Name:cn.com.duiba.tuia.core.api.client<br>
 * Date:2016年11月11日下午6:21:04<br>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.<br>
 */

package cn.com.duiba.tuia.media.api.client;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import cn.com.duiba.tuia.media.api.dto.MediaAppDataDto;
import cn.com.duiba.tuia.media.api.dto.SlotCacheDto;
import cn.com.duiba.tuia.media.api.dto.StrategyCacheDto;
import cn.com.duiba.tuia.media.api.remoteservice.RemoteMediaService;
import cn.com.duiba.tuia.media.api.utils.CacheKeyUtils;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.redis.RedisClient;

import com.alibaba.fastjson.JSONObject;

/**
 * ClassName: DuiBaItemClient <br/>
 * Function: 兑吧ItemClient. <br/>
 * date: 2016年11月11日 下午6:21:04 <br/>
 *
 * @author leiliang
 * @version
 * @since JDK 1.6
 */
public class TuiaMediaClient {

    /** 一周. */
    protected static final int  ONE_WEEK   = 60 * 60 * 24 * 7;

    /** 不存在的id, 说明库里没有该记录. */
    protected static final long DEFAULT_ID = -1;

    @Autowired
    protected RedisClient       redisClient;

    @Autowired
    private RemoteMediaService  remoteMediaService;

    /**
     * 查询广告位选择的屏蔽策略信息.<br>
     * [使用说明]
     * <ol>
     * <li>当该广告位没有选择屏蔽策略时，返回null</li>
     * <li>其他情况，返回屏蔽策略信息</li>
     * </ol>
     *
     * @param slotId the slot id
     * @return the strategy by slot id
     */
    public StrategyCacheDto getStrategyBySlotId(Long slotId) {

        SlotCacheDto slotCacheDto = getSlotBySlotId(slotId);

        if (slotCacheDto == null) {
            // 广告位缓存失效, 需要去更新缓存
            slotCacheDto = doUpdateSlot(slotId);
        }
        if (slotCacheDto.getId() == DEFAULT_ID || slotCacheDto.getStrategyId() == null) {
            // 库里没有改广告位的情况, 或者改广告位没有选择屏蔽策略的情况
            return null;
        }

        return doGetStrategyBySlotId(slotCacheDto, slotId);
    }

    /**
     * Do get strategy by slot id.
     *
     * @param slotCacheDto the slot cache dto
     * @param slotId the slot id
     * @return the strategy cache dto
     */
    private StrategyCacheDto doGetStrategyBySlotId(SlotCacheDto slotCacheDto, Long slotId) {
        Long strategyId = slotCacheDto.getStrategyId();
        String strategy = redisClient.get(CacheKeyUtils.getStrategyKey(strategyId));
        StrategyCacheDto strategyCacheDto;
        if (StringUtils.isEmpty(strategy)) {
            // 屏蔽策略缓存失效, 需要去更新缓存
            DubboResult<StrategyCacheDto> dubboResult = remoteMediaService.updateStrategyById(strategyId);
            if (!dubboResult.isSuccess()) {
                // 调用失败，返回null
                return null;
            }
            strategyCacheDto = dubboResult.getResult();
            if (strategyCacheDto == null) {
                // 数据库里没有该屏蔽策略信息
                redisClient.setex(CacheKeyUtils.getStrategyKey(strategyId), ONE_WEEK,
                                  JSONObject.toJSONString(new StrategyCacheDto(DEFAULT_ID)));
            }
        } else {
            strategyCacheDto = JSONObject.parseObject(strategy, StrategyCacheDto.class);
            strategyCacheDto = (strategyCacheDto.getId() != null && strategyCacheDto.getId() == DEFAULT_ID) ? null : strategyCacheDto;
        }

        return strategyCacheDto;
    }

    /**
     * 通过Appkey 获取app信息.
     *
     * @param appKey the app key
     * @return the media app by key
     */
    public MediaAppDataDto getMediaAppByKey(String appKey) {
        String strMedaiAppData = redisClient.get(CacheKeyUtils.getMediaAppByKey(appKey));
        MediaAppDataDto mediaAppDataDto;
        if (StringUtils.isNotBlank(strMedaiAppData)) {
            mediaAppDataDto = JSONObject.parseObject(strMedaiAppData, MediaAppDataDto.class);

            // 如果是默认的appId，则说明该app不存在，返回null
            return mediaAppDataDto.getAppId() == DEFAULT_ID ? null : mediaAppDataDto;
        } else {
            DubboResult<MediaAppDataDto> dubboResult = remoteMediaService.updateMediaAppByKey(appKey);
            if (!dubboResult.isSuccess()) {
                // 调用失败，返回null
                return null;
            }
            mediaAppDataDto = dubboResult.getResult();
            // 库里没有该app key对应的app，塞入一个默认值，防止下次再去查询数据库
            if (mediaAppDataDto == null) {
                redisClient.setex(CacheKeyUtils.getMediaAppByKey(appKey), ONE_WEEK,
                                  JSONObject.toJSONString(new MediaAppDataDto(DEFAULT_ID)));
            }
            return mediaAppDataDto;
        }
    }

    /**
     * 查询广告位信息.
     *
     * @param slotId the slot id
     * @return the strategy id
     */
    private SlotCacheDto getSlotBySlotId(Long slotId) {
        String strSlot = redisClient.get(CacheKeyUtils.getSlotKey(slotId));

        if (StringUtils.isBlank(strSlot)) {
            return null;
        }
        return JSONObject.parseObject(strSlot, SlotCacheDto.class);
    }

    /**
     * Gets the slot by id.
     *
     * @param slotId the slot id
     * @return the slot by id
     */
    public SlotCacheDto getSlotById(Long slotId) {
        SlotCacheDto slotCacheDto = getSlotBySlotId(slotId);

        if (slotCacheDto == null) {
            // 广告位缓存失效, 需要去更新缓存
            slotCacheDto = doUpdateSlot(slotId);
        } else {
            if (DEFAULT_ID == slotCacheDto.getId()) {
                slotCacheDto = null;
            }
        }

        return slotCacheDto;
    }

    /**
     * Do update slot.
     *
     * @param slotId the slot id
     * @return the slot cache dto
     */
    private SlotCacheDto doUpdateSlot(Long slotId) {
        SlotCacheDto slotCacheDto = null;
        DubboResult<SlotCacheDto> dubboResult = remoteMediaService.updateSlot(slotId);
        if (dubboResult.isSuccess()) {
            slotCacheDto = dubboResult.getResult();
            if (slotCacheDto == null) {
                // 数据库中也没用该广告位信息, 则塞入一个默认值，防止下次再去缓存中查询
                redisClient.setex(CacheKeyUtils.getSlotKey(slotId), ONE_WEEK,
                                  JSONObject.toJSONString(new SlotCacheDto(DEFAULT_ID)));
            }
        }
        return slotCacheDto;
    }
}
