/**
 * Project Name:media-biz<br>
 * File Name:StatisticsBackendBOImpl.java<br>
 * Package Name:cn.com.duiba.tuia.media.bo.impl<br>
 * Date:2016年10月10日下午4:25:07<br>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.<br>
 */
package cn.com.duiba.tuia.media.bo.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.common.tool.DataTool;
import cn.com.duiba.tuia.media.api.dto.MediaAppStatisticsDto;
import cn.com.duiba.tuia.media.api.dto.PageResultDto;
import cn.com.duiba.tuia.media.api.dto.SlotStatisticsDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqMediaAppStatisticsDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqSlotStatisticsDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspMediaAppStatisticsDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspSlotStatisticsDto;
import cn.com.duiba.tuia.media.bo.SlotStatisticsBackendBO;
import cn.com.duiba.tuia.media.model.IdAndName;
import cn.com.duiba.tuia.media.service.MediaAppService;
import cn.com.duiba.tuia.media.service.SlotService;
import cn.com.duiba.tuia.media.service.SlotStatisticsService;

/**
 * ClassName: StatisticsBackendBOImpl <br/>
 * date: 2016年10月10日 下午4:25:07 <br/>
 *
 * @author xiawei
 * @version
 * @since JDK 1.6
 */
@Service
public class SlotStatisticsBackendBOImpl implements SlotStatisticsBackendBO {

    /** 分页查询每页10条. */
    private static final int      PAGE_SIZE_TEN = 10;

    @Autowired
    private SlotStatisticsService slotStatisticsService;

    @Autowired
    private MediaAppService       mediaAppService;

    @Autowired
    private SlotService           slotService;

    @Override
    public PageResultDto<RspSlotStatisticsDto> getAdsenseData(ReqSlotStatisticsDto req) throws TuiaMediaException {
        String slotName = req.getSlotName();
        List<Long> slotIdsList = null;
        if (StringUtils.isNotBlank(slotName)) {
            // 通过广告位名称模糊查询广告位ID
            slotIdsList = slotService.selectIdsByName(slotName);
            if (CollectionUtils.isEmpty(slotIdsList)) {
                return new PageResultDto<>(0, null, req.getPageSize());
            }
        }

        String appName = req.getAppName();
        List<Long> appIdsList = null;
        if (StringUtils.isNotBlank(appName)) {
            // 通过应用名称模糊查询应用ID
            appIdsList = mediaAppService.selectIdsByName(appName);
            if (CollectionUtils.isEmpty(appIdsList)) {
                return new PageResultDto<>(0, null, req.getPageSize());
            }
        }

        Integer totalAmount = slotStatisticsService.getAdsenseDataAmount(req, slotIdsList, appIdsList);
        List<SlotStatisticsDto> slotStatisticsDtos;
        List<RspSlotStatisticsDto> rspDtos = null;
        List<Long> slotIds;
        List<Long> appIds;
        // 总数查询有结果，查询分页数据
        req.setPageSize(PAGE_SIZE_TEN);
        if ((totalAmount > 0) && (totalAmount >= req.getRowStart())) {
            req.setRowStart(req.getPageSize() * (req.getCurrentPage() - 1));
            slotStatisticsDtos = slotStatisticsService.getAdsenseDataByPage(req, slotIdsList, appIdsList);

            int size = slotStatisticsDtos.size();
            appIds = new ArrayList<>(size);
            slotIds = new ArrayList<>(size);
            for (SlotStatisticsDto slotStatisticsDto : slotStatisticsDtos) {
                appIds.add(slotStatisticsDto.getAppId());
                slotIds.add(slotStatisticsDto.getSlotId());
            }
            rspDtos = new ArrayList<>(size);

            // 通过slotId批量查询广告位信息
            Map<Long, String> slotMap = getSlotMap(slotIds);

            // 通过appId批量查询应用信息
            Map<Long, String> appMap = getMediaMap(appIds);
            //
            doRsp(slotStatisticsDtos, rspDtos, slotMap, appMap);
        }
        return new PageResultDto<>(totalAmount, rspDtos, req.getPageSize());
    }

    /**
     * 生成返回集合. <br/>
     *
     * @author xiawei
     * @param slotStatisticsDtos
     * @param rspDtos
     * @param slotMap
     * @param appMap
     * @since JDK 1.6
     */
    private void doRsp(List<SlotStatisticsDto> slotStatisticsDtos, List<RspSlotStatisticsDto> rspDtos,
                       Map<Long, String> slotMap, Map<Long, String> appMap) {
        for (SlotStatisticsDto dto : slotStatisticsDtos) {
            RspSlotStatisticsDto rspDto = new RspSlotStatisticsDto();
            rspDto.setSlotId(dto.getSlotId());
            rspDto.setSlotType(dto.getSlotType());
            rspDto.setClickCount(dto.getClickCount());
            rspDto.setConsumeTotal(dto.getConsumeTotal());
            rspDto.setExposureCount(dto.getExposureCount());
            rspDto.setAppName(appMap.get(dto.getAppId()));
            rspDto.setSlotName(slotMap.get(dto.getSlotId()));
            // 数据计算
            calculateStatisticsData(rspDto);
            rspDtos.add(rspDto);
        }
    }

    /**
     * 获取appId和appName的map. <br/>
     *
     * @author xiawei
     * @param appIds
     * @return
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    private Map<Long, String> getMediaMap(List<Long> appIds) throws TuiaMediaException {
        List<IdAndName> appIdAndNameList;
        Map<Long, String> appMap = new HashMap<>(1);
        if (CollectionUtils.isNotEmpty(appIds)) {
            appIdAndNameList = mediaAppService.selectAppIdAndName(appIds);
            for (IdAndName appVO : appIdAndNameList) {
                appMap.put(appVO.getId(), appVO.getName());
            }
        }
        return appMap;
    }

    /**
     * 获取slotId和slotName的map. <br/>
     *
     * @author xiawei
     * @param slotIds
     * @return
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    private Map<Long, String> getSlotMap(List<Long> slotIds) throws TuiaMediaException {
        List<IdAndName> slotIdAndNameList;
        Map<Long, String> slotMap = new HashMap<>(1);
        if (CollectionUtils.isNotEmpty(slotIds)) {
            slotIdAndNameList = slotService.selectAppIdAndName(slotIds);
            for (IdAndName slotVO : slotIdAndNameList) {
                slotMap.put(slotVO.getId(), slotVO.getName());
            }
        }
        return slotMap;
    }

    /**
     * 计算统计数据.
     *
     * @param data the data
     */
    private void calculateStatisticsData(RspSlotStatisticsDto data) {
        // eCPM=(预计收入/曝光量*1000)
        if (data.getExposureCount() != null) {
            data.seteCpm(DataTool.calculateEcpm(data.getConsumeTotal(), data.getExposureCount()));
        }
        // CPC=预计收入/点击量
        data.setCpc(DataTool.calculateAveragePrice(data.getConsumeTotal(), data.getClickCount()));
    }

    @Override
    public PageResultDto<RspMediaAppStatisticsDto> getMediaData(ReqMediaAppStatisticsDto req) throws TuiaMediaException {

        String appName = req.getAppName();
        List<Long> appIdsByName = null;
        if (appName != null) {
            // 通过应用名称模糊查询应用ID
            appIdsByName = mediaAppService.selectIdsByName(appName);
            if (CollectionUtils.isEmpty(appIdsByName)) {
                return new PageResultDto<>(0, null, req.getPageSize());
            }
        }

        Integer totalAmount = slotStatisticsService.getMediaDataAmount(req, appIdsByName);
        List<MediaAppStatisticsDto> mediaAppStatisticsDtos;
        List<RspMediaAppStatisticsDto> rspDtos = null;
        List<Long> appIds;
        // 总数查询有结果，查询分页数据
        req.setPageSize(PAGE_SIZE_TEN);
        if ((totalAmount > 0) && (totalAmount >= req.getRowStart())) {
            req.setRowStart(req.getPageSize() * (req.getCurrentPage() - 1));
            mediaAppStatisticsDtos = slotStatisticsService.getMediaDataByPage(req, appIdsByName);
            int size = mediaAppStatisticsDtos.size();
            appIds = new ArrayList<>(size);
            for (MediaAppStatisticsDto MediaAppStatisticsDto : mediaAppStatisticsDtos) {
                appIds.add(MediaAppStatisticsDto.getAppId());
            }
            rspDtos = new ArrayList<>(size);

            // 通过appId批量查询应用信息
            List<IdAndName> appIdAndNameList;
            Map<Long, String> appMap = null;
            if (CollectionUtils.isNotEmpty(appIds)) {
                appIdAndNameList = mediaAppService.selectAppIdAndName(appIds);
                appMap = new HashMap<>(appIds.size());
                for (IdAndName appVO : appIdAndNameList) {
                    appMap.put(appVO.getId(), appVO.getName());
                }
            }

            for (MediaAppStatisticsDto dto : mediaAppStatisticsDtos) {
                RspMediaAppStatisticsDto rspDto = new RspMediaAppStatisticsDto();
                rspDto.setAppId(dto.getAppId());
                rspDto.setClickCount(dto.getClickCount());
                rspDto.setConsumeTotal(dto.getConsumeTotal());
                rspDto.setExposureCount(dto.getExposureCount());
                rspDto.setPlatform(dto.getPlatform());
                if (appMap != null) {
                    rspDto.setAppName(appMap.get(dto.getAppId()));
                }
                rspDto.seteCpm(DataTool.calculateEcpm(dto.getConsumeTotal(), dto.getExposureCount()));
                rspDto.setCpc(DataTool.calculateAveragePrice(dto.getConsumeTotal(), dto.getClickCount()));
                rspDtos.add(rspDto);
            }
        }

        return new PageResultDto<>(totalAmount, rspDtos, req.getPageSize());
    }

}
