package cn.com.duiba.tuia.media.message.consumer;

/**
 * @author xuyenan
 * @createTime 2016/10/27
 */

import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.domain.ActivityDto;
import cn.com.duiba.tuia.media.message.MessageBody.AdActivityMessage;
import cn.com.duiba.tuia.media.service.ActivityService;
import cn.com.duiba.tuia.media.service.ActivitySortService;
import cn.com.duiba.tuia.media.utils.JsonUtils;
import com.google.common.collect.Lists;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.util.Arrays;
import java.util.List;

import static cn.com.duiba.tuia.media.message.MessageBody.AdActivityMessage.Action_Delete_Type;
import static cn.com.duiba.tuia.media.message.MessageBody.AdActivityMessage.Action_Insert_Type;
import static cn.com.duiba.tuia.media.message.MessageBody.AdActivityMessage.Action_Update_Type;

/**
 * 兑吧活动信息变更消息接收器。
 * <p>
 * 在兑吧活动中心,活动信息修改后，会通过kafka发送消息，这个类接受消息后会同步到广告系统中。
 */
@Component
public class ActivityKafkaConsumer extends AbstractKafkaConsumer {

    private Logger              log = LoggerFactory.getLogger(getClass());

    @Autowired
    private ActivityService     activityService;

    @Autowired
    private ActivitySortService activitySortService;

    @Value("${media.kafka.activity.topic}")
    private String              topic;

    protected List<String> getTopics() {
        return Arrays.asList(this.topic);
    }

    @Override
    protected void readMessage(ConsumerRecord<String, String> record) {
        String message = record.value();
        AdActivityMessage adActivityMessage = JsonUtils.jsonToObject(AdActivityMessage.class, message);
        if (adActivityMessage != null) {
            try {
                switch (adActivityMessage.getAction()) {
                    case Action_Insert_Type: {
                        ActivityDto activityDto = new ActivityDto();
                        activityDto.setActivityId(adActivityMessage.getActivityId());
                        activityDto.setActivityStatus(adActivityMessage.getActivityStatus());
                        activityDto.setActivityType(adActivityMessage.getActivityType());
                        activityDto.setActivityName(adActivityMessage.getActivityName());
                        activityDto.setTag(adActivityMessage.getTag());
                        activityService.insertActivity(Lists.newArrayList(activityDto));
                        break;
                    }
                    case Action_Update_Type: {
                        ActivityDto activityDto = activityService.selectByActivityIdAndType(adActivityMessage.getActivityId(),
                                                                                            adActivityMessage.getActivityType());
                        if (activityDto == null) {
                            throw new TuiaMediaException(ErrorCode.E0501001);
                        }
                        if (adActivityMessage.getActivityStatus() != null) {
                            activityDto.setActivityStatus(adActivityMessage.getActivityStatus());
                        }
                        if (adActivityMessage.getActivityType() != null) {
                            activityDto.setActivityType(adActivityMessage.getActivityType());
                        }
                        if (adActivityMessage.getActivityName() != null) {
                            activityDto.setActivityName(adActivityMessage.getActivityName());
                        }
                        if (adActivityMessage.getTag() != null) {
                            activityDto.setTag(adActivityMessage.getTag());
                        }
                        activityService.updateActivity(activityDto);
                        break;
                    }
                    case Action_Delete_Type: {
                        ActivityDto activityDto = activityService.selectByActivityIdAndType(adActivityMessage.getActivityId(),
                                                                                            adActivityMessage.getActivityType());
                        if (activityDto == null) {
                            throw new TuiaMediaException(ErrorCode.E0501001);
                        }
                        activityDto.setDelete(1);
                        activityService.updateActivity(activityDto);
                        // 删除活动时需要移除所有活动列表的该活动
                        activitySortService.deleteActivity(0L, activityDto.getActivityId(),
                                                           activityDto.getActivityType());
                        break;
                    }
                }
            } catch (TuiaMediaException e) {
                log.error("ActivityKafkaConsumer error:", e);
            }
        }
    }
}
