/**
 * Project Name:media-biz<br>
 * File Name:AccountServiceImpl.java<br>
 * Package Name:cn.com.duiba.tuia.media.service.impl<br>
 * Date:2016年9月26日上午10:52:36<br>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.<br>
 */
package cn.com.duiba.tuia.media.service.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import cn.com.duiba.tuia.media.api.dto.AccountDto;
import cn.com.duiba.tuia.media.api.dto.PageResultDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqGetAccountByPageDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspAccountBaseDto;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.constants.TuiaDomainConstant;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.common.tool.IDGeneratorTool;
import cn.com.duiba.tuia.media.common.utils.BlowfishUtils;
import cn.com.duiba.tuia.media.common.utils.MessageDigestUtils;
import cn.com.duiba.tuia.media.dao.AccountBankDAO;
import cn.com.duiba.tuia.media.dao.AccountCheckRecordDAO;
import cn.com.duiba.tuia.media.dao.AccountDAO;
import cn.com.duiba.tuia.media.model.Email;
import cn.com.duiba.tuia.media.model.req.ReqUpdateAccountBank;
import cn.com.duiba.tuia.media.model.req.ReqUpdateAccountBase;
import cn.com.duiba.tuia.media.model.rsp.TokenRsp;
import cn.com.duiba.tuia.media.service.AccountService;
import cn.com.duiba.tuia.media.service.EmailService;
import cn.com.duiba.tuia.media.utils.CacheKeyUtils;
import cn.com.duiba.tuia.media.utils.RequestLocal;

/**
 * ClassName: AccountServiceImpl <br/>
 * date: 2016年9月26日 上午10:52:36 <br/>
 *
 * @author xiawei
 * @version
 * @since JDK 1.6
 */
@Service
public class AccountServiceImpl extends BaseCacheService implements AccountService {

    /** 邮箱验证码位数. */
    private static final int      EMAIL_CODE_LENGTH = 6;

    /** 10分钟. */
    private static final int      THIRTY_MINUTES    = 3 * 10 * 60;

    /** 1小时. */
    private static final int      ONE_HOUR          = 3600;

    /** 48小时. */
    private static final int      FORTYEIGHT_HOURS  = 48 * 60 * 60;

    /** The login encrypt key. */
    @Value("${media.login.encrypt.key}")
    private String                loginEncryptKey;

    /** The email verify key. */
    @Value("${media.email.verify.key}")
    private String                emailVerifyKey;

    /** The email service. */
    @Autowired
    private EmailService          emailService;

    @Autowired
    private AccountDAO            accountDAO;

    @Autowired
    private AccountBankDAO        accountBankDAO;

    @Autowired
    private AccountCheckRecordDAO accountCheckRecordDao;

    @Autowired
    private MediaCacheService     mediaCacheService;

    @Override
    public boolean isEmailExists(String email) throws TuiaMediaException {
        return accountDAO.selectByEmail(email) != null ? true : false;
    }

    @Override
    public boolean sendResetPasswdEmail(final String email) throws TuiaMediaException {

        // 查询账号信息
        final AccountDto accountDto = accountDAO.selectByEmail(email);
        accountDtoIsNull(accountDto);
        // 发送邮件
        doSendResetPasswdEmail(accountDto.getLinkman(), email);

        return true;

    }

    @Override
    public boolean sendReplaceEmail(String email, Long userId) throws TuiaMediaException {

        AccountDto user = selectByIdNotNull(userId);
        // 校验该邮箱是否被使用
        AccountDto accountDto = accountDAO.selectByEmail(email);
        if (accountDto != null) {
            if (accountDto.getMediaId().longValue() == userId) {
                // 邮箱没有修改
                throw new TuiaMediaException(ErrorCode.E0104017);
            } else {
                // 邮箱被占用
                throw new TuiaMediaException(ErrorCode.E0102016);
            }
        }

        doSendReplaceEmail(user.getLinkman(), email, userId);

        return true;
    }

    @Override
    public boolean replaceEmail(String token, Long userId) throws TuiaMediaException {
        String cacheEmail = redisClient.get(CacheKeyUtils.getReplaceEmailKey(userId, token));
        if (StringUtils.isEmpty(cacheEmail)) {
            // 验证码失效
            throw new TuiaMediaException(ErrorCode.E0102003);
        }
        if (accountDAO.updateEmail(userId, cacheEmail) != 1) {
            // 更换邮箱失败
            throw new TuiaMediaException(ErrorCode.E0102018);
        }
        redisClient.del(CacheKeyUtils.getReplaceEmailKey(userId, token));
        return true;
    }

    @Override
    public AccountDto selectByIdNotNull(Long mediaId) throws TuiaMediaException {
        AccountDto user = accountDAO.selectById(mediaId);
        if (user == null) {
            throw new TuiaMediaException(ErrorCode.E0102001);
        }

        return user;
    }

    @Override
    public TokenRsp resetPasswdRedirect(String email, String verificationCode) throws TuiaMediaException {
        String cacheCode = redisClient.get(CacheKeyUtils.getResetPwEmailKey(email));
        if (!StringUtils.equals(verificationCode, cacheCode)) {
            // 验证码失效
            logger.error("the verificationCode invalid, the email=[{}]", email);
            throw new TuiaMediaException(ErrorCode.E0102003);
        }
        String key = new Date().getTime() + "-" + email;
        String token = BlowfishUtils.encryptBlowfish(key, emailVerifyKey);
        redisClient.setex(token, THIRTY_MINUTES, email);
        TokenRsp tokenRsp = new TokenRsp();
        tokenRsp.setEmail(email);
        tokenRsp.setToken(token);
        return tokenRsp;
    }

    @Override
    public boolean resetPassword(String email, String newPasswd, String token) throws TuiaMediaException {
        // 校验重置密码的toekn
        String cacheToken = redisClient.get(token);
        if (null == cacheToken) {
            // token失效
            logger.error("the token invalid, the email=[{}]", email);
            throw new TuiaMediaException(ErrorCode.E0102009);
        }

        String newPd = BlowfishUtils.encryptBlowfish(MessageDigestUtils.SHA(newPasswd), loginEncryptKey);
        // 查询账号
        AccountDto accountDto = accountDAO.selectByEmail(email);
        accountDtoIsNull(accountDto);
        if (newPd.equals(accountDto.getPassword())) {
            // 新密码和旧密码不能一致
            throw new TuiaMediaException(ErrorCode.E0102013);
        }

        int count = accountDAO.updatePasswdByEmail(email, newPd);

        if (count > 0) {
            // 重置成功， 删除缓存验证码,token
            redisClient.del(CacheKeyUtils.getResetPwEmailKey(email));
            redisClient.del(token);
            return true;
        }
        return false;
    }

    @Override
    public void verifyEmail(String token) throws TuiaMediaException {
        // 从缓存里查询该token
        String value = redisClient.get(token);
        if (value == null) {
            logger.error("token has invalid, the token=[{}]", token);
            throw new TuiaMediaException(ErrorCode.E0102003);
        }

        // 查询账号
        AccountDto accountDto = accountDAO.selectByEmail(value);
        accountDtoIsNull(accountDto);
        // 更新邮箱审核状态
        accountDAO.updateEmailStatus(accountDto.getMediaId(), AccountDto.EMAIL_STATUS_CHECKED);

        redisClient.del(token);
    }

    /**
     * 构建和发送重置密码邮件.
     *
     * @param userId 用户ID
     * @param linkman 用户名称
     * @param userEmail the user email
     */
    private void doSendResetPasswdEmail(final String linkman, final String userEmail) {
        // 发送邮件
        executorService.submit(new Runnable() {

            @Override
            public void run() {
                // 生成验证码
                String verificationCode = IDGeneratorTool.getRandomString(EMAIL_CODE_LENGTH);

                Map<String, Object> model = new HashMap<>(3);
                model.put("linkman", linkman);
                model.put("email", userEmail);
                model.put("verificationCode", verificationCode);

                // 拼装邮件内容
                Email email = new Email();
                email.setTo(userEmail);
                email.setTemplateLocation("/templates/mail/resetpasswd.vm");
                email.setSubject("推啊媒体账号找回密码-邮箱验证码");
                email.setModel(model);

                emailService.sendWithTemplate(email);

                // 存入缓存
                redisClient.setex(CacheKeyUtils.getResetPwEmailKey(userEmail), THIRTY_MINUTES, verificationCode);
            }
        });
    }

    /**
     * 发生更换邮箱的邮件.
     *
     * @param linkman the linkman
     * @param userEmail the user email
     */
    private void doSendReplaceEmail(final String linkman, final String userEmail, final Long userId) {
        // 发送邮件
        executorService.submit(new Runnable() {

            @Override
            public void run() {
                // 生成验证码
                String verificationCode = IDGeneratorTool.getRandomString(EMAIL_CODE_LENGTH);

                Map<String, Object> model = new HashMap<>(3);
                model.put("linkman", linkman);
                model.put("email", userEmail);
                model.put("verificationCode", verificationCode);

                // 拼装邮件内容
                Email email = new Email();
                email.setTo(userEmail);
                email.setTemplateLocation("/templates/mail/replaceEmail.vm");
                email.setSubject("推啊媒体更换邮箱通知");
                email.setModel(model);

                emailService.sendWithTemplate(email);

                // 存入缓存
                redisClient.setex(CacheKeyUtils.getReplaceEmailKey(userId, verificationCode), ONE_HOUR, userEmail);
            }
        });
    }

    /**
     * Do send check email.
     *
     * @param accountDO the account do
     */
    @Override
    public void doSendVerifyEmail(final AccountDto accountDto) {
        executorService.submit(new Runnable() {

            @Override
            public void run() {
                String key = new Date().getTime() + "-" + accountDto.getMediaId();
                String code = BlowfishUtils.encryptBlowfish(key, emailVerifyKey);
                String email = accountDto.getEmail();
                String url = TuiaDomainConstant.getTuia_domain_name() + "/account/verifyEmail?email=" + email
                             + "&token=" + code;
                Map<String, Object> model = new HashMap<>();
                model.put("linkman", accountDto.getLinkman());
                model.put("email", email);
                model.put("url", url);

                // 拼装邮件内容
                Email sendEmail = new Email();
                sendEmail.setTo(accountDto.getEmail());
                sendEmail.setTemplateLocation("/templates/mail/active.vm");
                sendEmail.setSubject("推啊媒体账号邮箱验证");
                sendEmail.setModel(model);

                emailService.sendWithTemplate(sendEmail);

                redisClient.setex(code, FORTYEIGHT_HOURS, accountDto.getEmail());
            }
        });
    }

    @Override
    public void repeatVerifyEmail(String email) throws TuiaMediaException {
        AccountDto accountDto = accountDAO.selectByEmail(email);
        accountDtoIsNull(accountDto);
        int emailStatus = accountDto.getEmailStatus();
        if (AccountDto.EMAIL_STATUS_CHECKED == emailStatus) {
            // 邮箱已验证,审核中
            if (AccountDto.CHECK_STATUS_ING == accountDto.getCheckStatus()) {
                throw new TuiaMediaException(ErrorCode.E0102012.getErrorCode(), "该邮箱账号已验证,账号审核中");
            }
            // 邮箱已验证,审核通过
            if (AccountDto.CHECK_STATUS_PASS == accountDto.getCheckStatus()) {
                throw new TuiaMediaException(ErrorCode.E0102012.getErrorCode(), "该邮箱账号已验证,账号审核通过");
            }

            // 邮箱已验证,审核拒绝
            if (AccountDto.CHECK_STATUS_REFUSE == accountDto.getCheckStatus()) {
                throw new TuiaMediaException(ErrorCode.E0102012.getErrorCode(), "该邮箱账号已验证,账号审核拒绝");
            }

        }
        // 重新发送验证邮件
        doSendVerifyEmail(accountDto);
    }

    @Override
    public boolean updateAccountBase(ReqUpdateAccountBase req) throws TuiaMediaException {
        // 判断手机号是否存在
        boolean flag = repeatRegisterVerifyPhone(req.getLinkPhone(), RequestLocal.get().getCid());
        if (flag) {
            logger.error("the phone is exist");
            throw new TuiaMediaException(ErrorCode.E0102021);
        }
        accountDAO.updateAccount(req);
        // 更新缓存
        mediaCacheService.updateMediaCache(req.getMediaId(), false);
        return true;
    }

    @Override
    public boolean updateAccountBank(ReqUpdateAccountBank req) throws TuiaMediaException {
        accountBankDAO.update(req);

        // 更新缓存
        mediaCacheService.updateMediaCache(req.getMediaId(), false);
        return true;
    }

    @Override
    public PageResultDto<RspAccountBaseDto> pageQuery(ReqGetAccountByPageDto req) throws TuiaMediaException {
        // 查询总数
        Integer totalAmount = accountDAO.selectMediaDataAmount(req);
        List<AccountDto> accountDtos = null;
        // 总数查询有结果，查询分页数据
        List<RspAccountBaseDto> mediaRsps = new ArrayList<>();
        if ((totalAmount != 0) && (totalAmount >= req.getRowStart())) {
            req.setRowStart(req.getPageSize() * (req.getCurrentPage() - 1));
            accountDtos = accountDAO.selectMediaDataByPage(req);
        }
        if (CollectionUtils.isNotEmpty(accountDtos)) {
            for (AccountDto accountDto : accountDtos) {
                RspAccountBaseDto rspAccountBase = new RspAccountBaseDto();
                rspAccountBase.setMediaId(accountDto.getMediaId());
                rspAccountBase.setEmail(accountDto.getEmail());
                rspAccountBase.setEmailStatus(accountDto.getEmailStatus());
                rspAccountBase.setCompanyName(accountDto.getCompanyName());
                rspAccountBase.setBusinessLicenseId(accountDto.getBusinessLicenseId());
                rspAccountBase.setBusinessLicenseUrl(accountDto.getBusinessLicenseUrl());
                rspAccountBase.setLinkman(accountDto.getLinkman());
                rspAccountBase.setLinkmanPhone(accountDto.getLinkPhone());
                rspAccountBase.setCheckStatus(accountDto.getCheckStatus());
                rspAccountBase.setFreezeStatus(accountDto.getFreezeStatus());
                mediaRsps.add(rspAccountBase);
            }
        }

        return new PageResultDto<>(totalAmount, mediaRsps, req.getPageSize());
    }

    @Override
    public boolean updateCheckStatus(Long mediaId, Integer status) throws TuiaMediaException {
        boolean flag = accountDAO.updateCheckStatus(mediaId, status) == 1 ? true : false;
        // 更新缓存
        if (flag && AccountDto.CHECK_STATUS_PASS == status) {
            mediaCacheService.updateMediaCache(mediaId, true);
        }
        return flag;

    }

    @Override
    public boolean updateFreezeStatus(Long mediaId, Integer status) throws TuiaMediaException {
        boolean flag = accountDAO.updateFreezeStatus(mediaId, status) == 1 ? true : false;
        if (flag) {
            // 更新缓存
            if (AccountDto.UNFREEZED_STATUS == status) {
                mediaCacheService.updateMediaCache(mediaId, true);
            }
            if (AccountDto.FREEZED_STATUS == status) {
                mediaCacheService.updateMediaCache(mediaId, false);
            }
        }
        return flag;
    }

    @Override
    public boolean insertAccountBase(AccountDto accountDto) throws TuiaMediaException {
        return accountDAO.insert(accountDto) == 1 ? true : false;
    }

    @Override
    public AccountDto selectByMediaId(Long mediaId) throws TuiaMediaException {
        return accountDAO.selectById(mediaId);
    }

    @Override
    public AccountDto selectByEmail(String email) throws TuiaMediaException {
        return accountDAO.selectByEmail(email);
    }

    @Override
    public boolean updateAuditData(AccountDto req) throws TuiaMediaException {
        return accountDAO.updateAuditData(req) == 1 ? true : false;
    }

    @Override
    public void accountDtoIsNull(AccountDto dto) throws TuiaMediaException {
        if (dto == null) {
            logger.error("the account is not exist");
            throw new TuiaMediaException(ErrorCode.E0102001);
        }

    }

    @Override
    public AccountDto isPhoneExists(String linkPhone) throws TuiaMediaException {
        return accountDAO.selectByPhone(linkPhone);
    }

    @Override
    public boolean repeatRegisterVerifyPhone(String linkPhone, Long mediaId) throws TuiaMediaException {
        AccountDto targetAccount = isPhoneExists(linkPhone);
        if (mediaId != null) {
            // 表示重新提交注册信息时验证手机号
            AccountDto sourceAccount = accountDAO.selectById(mediaId);
            // 判断账户是否存在
            accountDtoIsNull(sourceAccount);
            if (targetAccount != null && !sourceAccount.getLinkPhone().equals(targetAccount.getLinkPhone())) {
                return true;
            }
            return false;
        }
        // 表示注册信息时验证手机号
        if (targetAccount != null) {
            return true;
        }
        return false;
    }
}
