/**
 * Project Name:media-biz File Name:MediaServiceImpl.java Package Name:cn.com.duiba.tuia.media.service.impl
 * Date:2016年9月26日上午11:25:40 Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 */
package cn.com.duiba.tuia.media.service.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.UUID;

import cn.com.duiba.tuia.media.dao.ActivityAppDAO;
import cn.com.duiba.tuia.media.domain.ActivityAppDto;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import cn.com.duiba.developer.center.api.domain.dto.AppSimpleDto;
import cn.com.duiba.developer.center.api.domain.paramquery.CreateAppQueryParams;
import cn.com.duiba.developer.center.api.remoteservice.RemoteAppService;
import cn.com.duiba.tuia.media.api.dto.MediaAppCheckRecordDto;
import cn.com.duiba.tuia.media.api.dto.MediaAppDto;
import cn.com.duiba.tuia.media.api.dto.PageResultDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqCreatMediaAppDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqUpdateMediaAppStatusDto;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.common.utils.BlowfishUtils;
import cn.com.duiba.tuia.media.dao.BaseDAO;
import cn.com.duiba.tuia.media.dao.MediaAppCheckRecordDao;
import cn.com.duiba.tuia.media.dao.MediaAppDao;
import cn.com.duiba.tuia.media.model.IdAndName;
import cn.com.duiba.tuia.media.model.req.ReqPageQueryMediaApp;
import cn.com.duiba.tuia.media.model.req.ReqUpdateMediaApp;
import cn.com.duiba.tuia.media.model.rsp.RspIdAndValue;
import cn.com.duiba.tuia.media.model.rsp.RspMediaApp;
import cn.com.duiba.tuia.media.service.MediaAppService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.utils.Base58;
import cn.com.duiba.wolf.utils.SecurityUtils;

/**
 * ClassName:MediaAppServiceImpl <br/>
 * Function: 媒体管理. <br/>
 * Date: 2016年9月26日 上午11:25:40 <br/>
 * 
 * @author guyan
 * @version
 * @since JDK 1.6
 * @see
 */
@Service
public class MediaAppServiceImpl implements MediaAppService {

    protected Logger               logger = LoggerFactory.getLogger(MediaAppServiceImpl.class);

    /** The mediaApp dao */
    @Autowired
    private MediaAppDao            mediaAppDao;

    /** The remoteAppService service */
    @Autowired
    private RemoteAppService       remoteAppService;

    @Autowired
    private MediaCacheService      mediaCacheService;

    @Autowired
    private MediaAppCheckRecordDao mediaAppCheckRecordDao;

    @Autowired
    private ActivityAppDAO activityAppDAO;

    /** app加密秘钥. */
    @Value("${media.consumer.encrypt.key}")
    private String                 consumerEncryptKey;

    @Override
    public Boolean createMediaApp(ReqCreatMediaAppDto param) throws TuiaMediaException {
        CreateAppQueryParams createApp = new CreateAppQueryParams();
        Long mediaId = param.getMediaId();
        if (StringUtils.isEmpty(param.getAppName())) {
            throw new TuiaMediaException(ErrorCode.E0102001);
        }

        MediaAppDto mediaApp = this.mediaAppDao.getMediaAppByName(param.getAppName());

        if (null != mediaApp) {
            throw new TuiaMediaException(ErrorCode.E0401008);
        }
        createApp.setDeveloperId(mediaId);
        createApp.setName(param.getAppName());
        createApp.setMallCredits(true);

        MediaAppDto mediaAppDto = getMediaAppByRemote(createApp);
        mediaAppDto.setMediaId(mediaId);
        mediaAppDto.setAppName(param.getAppName());
        mediaAppDto.setPlatform(param.getPlatform());
        mediaAppDto.setIndustry(param.getIndustry());
        mediaAppDto.setCheckStatus(ReqCreatMediaAppDto.MEDIA_APP_CHECKING); // 初始状态'待审核'
        mediaAppDto.setAppPackage(param.getAppPackage());
        mediaAppDto.setDescription(param.getDescription());
        mediaAppDto.setDownloadUrl(param.getDownloadUrl());
        mediaAppDto.setAppKeyWords(param.getAppKeyWords());
        this.mediaAppDao.insert(mediaAppDto);
        mediaCacheService.updateMediaAppCache(mediaAppDto.getAppId());
        return true;
    }

    @SuppressWarnings("unchecked")
    @Override
    public PageResultDto<RspMediaApp> getMediaAppList(ReqPageQueryMediaApp mediaAppQuery) throws TuiaMediaException {
        mediaAppQuery.setPageSize(ReqPageQueryMediaApp.PAGE_SIZE_50);
        mediaAppQuery.setRowStart((mediaAppQuery.getCurrentPage() - 1) * mediaAppQuery.getPageSize());
        mediaAppQuery.setSort(ReqPageQueryMediaApp.SORT_TYPE);
        mediaAppQuery.setOrder(ReqPageQueryMediaApp.ORDER_TYPE);
        List<MediaAppDto> mediaAppList;
        List<RspMediaApp> rspMediaAppList = null;

        int totalNum = mediaAppDao.selectAmountByPageQuery(mediaAppQuery);
        if (totalNum > 0 && totalNum >= mediaAppQuery.getRowStart()) {
            mediaAppList = mediaAppDao.selectMediaAppList(mediaAppQuery);
            if (CollectionUtils.isNotEmpty(mediaAppList)) {
                rspMediaAppList = new ArrayList<>(mediaAppList.size());
                for (MediaAppDto dto : mediaAppList) {
                    RspMediaApp rspMediaApp = new RspMediaApp();
                    rspMediaApp.setAppId(dto.getAppId());
                    rspMediaApp.setAppKey(dto.getAppKey());
                    String appSecret = dto.getAppSecret();
                    rspMediaApp.setAppSecret(BlowfishUtils.decryptBlowfish(StringUtils.isNotEmpty(dto.getAppSecret()) == true ? appSecret : "",
                                                                           consumerEncryptKey));
                    rspMediaApp.setAppSecret(dto.getAppSecret());
                    rspMediaApp.setMediaId(dto.getMediaId());
                    rspMediaApp.setAppName(dto.getAppName());
                    rspMediaApp.setIndustry(dto.getIndustry());
                    rspMediaApp.setCheckStatus(dto.getCheckStatus());
                    rspMediaApp.setPlatform(dto.getPlatform());
                    rspMediaApp.setCreateDate(new DateTime(dto.getGmtCreate()).toString("yyyy-MM-dd"));
                    rspMediaAppList.add(rspMediaApp);
                }
            } else {
                rspMediaAppList = Collections.EMPTY_LIST;
            }
        }
        return new PageResultDto<RspMediaApp>(totalNum, rspMediaAppList, mediaAppQuery.getPageSize());
    }

    @Override
    public Boolean updateMediaApp(ReqUpdateMediaApp param) throws TuiaMediaException {
        MediaAppDto mediaAppDto = mediaAppDao.selectMediaAppById(param.getAppId());
        if (mediaAppDto == null) {
            logger.error("the mediaAppDto is not exist, the appId = [{}]", param.getAppId());
            throw new TuiaMediaException(ErrorCode.E0401001);
        }
        if (mediaAppDto.getCheckStatus() != null
            && mediaAppDto.getCheckStatus() == ReqUpdateMediaApp.MEDIA_APP_CHECK_FAIL) {
            param.setCheckStatus(ReqUpdateMediaApp.MEDIA_APP_CHECKING);
            mediaAppDao.update(param);
        }
        if (mediaAppDto.getCheckStatus() != null
            && mediaAppDto.getCheckStatus() == ReqUpdateMediaApp.MEDIA_APP_CHECK_PASS) {
            param.setCheckStatus(ReqUpdateMediaApp.MEDIA_APP_CHECKING);
            mediaAppDao.update(param);
            //修改媒体时需要同步到活动定制媒体表
            ActivityAppDto activityAppDto = new ActivityAppDto();
            activityAppDto.setActivityAppId(param.getAppId());
            activityAppDto.setActivityAppName(param.getAppName());
            activityAppDAO.update(activityAppDto);
        }
        mediaCacheService.updateMediaAppCache(param.getAppId());
        return true;
    }

    @Override
    public MediaAppDto getMediaAppDto(Long appId) throws TuiaMediaException {
        MediaAppDto mediaAppDto = this.mediaAppDao.selectMediaAppById(appId);
        if (mediaAppDto == null) {
            logger.error("the mediaAppDto is not exist, the appId = [{}]", appId);
            throw new TuiaMediaException(ErrorCode.E0401001);
        }
        return mediaAppDto;
    }

    @Override
    public Boolean updateMediaAppStatus(ReqUpdateMediaAppStatusDto param) throws TuiaMediaException {

        int result = this.mediaAppDao.updateMediaAppStatus(param);
        if (result == ReqUpdateMediaAppStatusDto.UPDATE_SUCCESS
            && (param.getCheckStatus() == ReqUpdateMediaAppStatusDto.MEDIA_APP_CHECK_FAIL || param.getCheckStatus() == ReqUpdateMediaAppStatusDto.MEDIA_APP_CHECK_PASS)) {
            MediaAppDto mediaAppDto = mediaAppDao.selectMediaAppById(param.getAppId());
            MediaAppCheckRecordDto checkRecordDto = new MediaAppCheckRecordDto();
            checkRecordDto.setAppId(mediaAppDto.getAppId());
            checkRecordDto.setCheckType(mediaAppDto.getCheckStatus());
            checkRecordDto.setMediaId(mediaAppDto.getMediaId());
            mediaAppCheckRecordDao.insert(checkRecordDto);
        }
        mediaCacheService.updateMediaAppCache(param.getAppId());
        return true;
    }

    /**
     * 获取appId、appKey、appSecret方法
     * 
     * @param param
     * @return
     * @throws TuiaMediaException
     */
    private MediaAppDto getMediaAppByRemote(CreateAppQueryParams param) throws TuiaMediaException {
        MediaAppDto mediaApp = new MediaAppDto();
        DubboResult<Long> appResult = this.remoteAppService.createApp(param);
        if (!appResult.isSuccess() && appResult.getResult() != null) {
            logger.error("the remoteAppService.createApp is error", appResult.getMsg());
            throw new TuiaMediaException(ErrorCode.E0401003);
        }
        DubboResult<AppSimpleDto> appSimpleResult = this.remoteAppService.getSimpleApp(appResult.getResult());

        AppSimpleDto app = appSimpleResult.getResult();
        if (!appSimpleResult.isSuccess() && app != null) {
            logger.error("the remoteAppService.getSimpleApp is error", appSimpleResult.getMsg());
            throw new TuiaMediaException(ErrorCode.E0401004);
        }

        if (null == app.getAppKey() || null == app.getId()) {
            throw new TuiaMediaException(ErrorCode.E0401003);
        }
        mediaApp.setAppKey(app.getAppKey());
        mediaApp.setAppId(app.getId());
        mediaApp.setAppSecret(app.getAppSecret());
        String appSecret = generate();
        mediaApp.setAppSecret(BlowfishUtils.encryptBlowfish(StringUtils.isNotEmpty(appSecret) == true ? appSecret : "",
                                                            consumerEncryptKey));
        return mediaApp;
    }

    @Override
    public List<Long> selectIdsByName(String appName) throws TuiaMediaException {
        return mediaAppDao.selectIdsByName(appName);
    }

    @Override
    public List<IdAndName> selectAppIdAndName(List<Long> appIds) throws TuiaMediaException {
        // 计算列表长度(考虑为空的情况)
        int size = (appIds == null ? 0 : appIds.size());
        List<IdAndName> result = new ArrayList<>(size);

        // 批量最大执行长度为100
        for (int i = 1, j = 0; i <= size; i++) {
            if (i % BaseDAO.BATCH_EXECUTE_MAX_LENGTH == 0 || i == size) {
                result.addAll(mediaAppDao.selectAppIdAndName(appIds.subList(j, i)));
                j = i;
            }
        }
        return result;
    }

    @SuppressWarnings("unchecked")
    @Override
    public List<RspIdAndValue> selectAppIdAndNameByMediaId(Long mediaId) throws TuiaMediaException {
        List<IdAndName> dtoList = mediaAppDao.selectAppIdAndNameByMediaId(mediaId);

        if (CollectionUtils.isNotEmpty(dtoList)) {
            List<RspIdAndValue> rspList = new ArrayList<>(dtoList.size());

            for (IdAndName dto : dtoList) {
                RspIdAndValue rspDto = new RspIdAndValue();
                if (null != dto.getId()) {
                    rspDto.setId(Long.toString(dto.getId()));
                }
                rspDto.setValue(dto.getName());
                rspList.add(rspDto);
            }
            return rspList;
        } else {
            return Collections.EMPTY_LIST;
        }
    }

    /**
     * generate:通过加密算法生成表示位 <br/>
     *
     * @author guyan
     * @return
     * @since JDK 1.6
     */
    private String generate() {
        byte[] bs = SecurityUtils.encode2BytesBySHA(UUID.randomUUID().toString());
        return Base58.encode(bs);
    }

}
