package cn.com.duiba.tuia.media.utils;

import java.util.Date;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import cn.com.duiba.tuia.media.common.constants.CommonConstants.COOKIE_KEY;
import cn.com.duiba.tuia.media.common.tool.SecureTool;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;

/**
 * The Class RequestLocal.
 */
public class RequestLocal {

    /** The logger. */
    private Logger                           logger             = LoggerFactory.getLogger(RequestLocal.class);

    /** 一天的毫秒数. */
    private static final int                 DAY_OF_MILLISECOND = 86400000;

    /** The local. */
    private static ThreadLocal<RequestLocal> local              = new ThreadLocal<>();

    /** 用户ID. */
    private Long                             cid;

    /** cookie中的账户内容. */
    private JSONObject                       accountInfo;

    /** The request. */
    private HttpServletRequest               request;

    /** The response. */
    private HttpServletResponse              response;

    /**
     * The Constructor.
     */
    private RequestLocal() {
    }

    /**
     * Gets the.
     *
     * @return the request local
     */
    public static RequestLocal get() {
        RequestLocal rl = local.get();
        if (rl == null) {
            rl = new RequestLocal();
            local.set(rl);
        }
        return rl;
    }

    /**
     * Clear.
     */
    public static void clear() {
        local.set(null);
    }

    /**
     * 根据key获取cookie中对应的值.<br>
     * 如果cookie为空或者不存在改key则返回null<br>
     *
     * @param key 键
     * @return cookie中对应的值
     */
    private String getCookie(String key) {
        Cookie[] cookies = request.getCookies();
        if (cookies != null) {
            for (Cookie cookie : cookies) {
                if (key.equals(cookie.getName())) {
                    String value = cookie.getValue();
                    if (StringUtils.isNotBlank(value)) {
                        return value;
                    }
                }
            }
        }

        return null;
    }

    /**
     * 获取cookie中关于账号信息部分的数据.
     *
     * @return the cookie value
     */
    public JSONObject getAccountInfo() {
        if (accountInfo == null) {
            try {
                // 获取账号信息
                String accound = getCookie(COOKIE_KEY.ACCOUNT_KEY);
                if (accound != null) {
                    String content = SecureTool.decryptConsumerCookie(accound);
                    if (StringUtils.isNotBlank(content)) {
                        JSONObject json = JSON.parseObject(content);
                        if (new Date().getTime() - json.getLong(COOKIE_KEY.LOGIN_TIME_KEY) < DAY_OF_MILLISECOND) {
                            accountInfo = json;
                        } else {
                            // 24小时过期, 删除cookie
                            CookieUtil.deleteCookie(COOKIE_KEY.ACCOUNT_KEY);
                        }
                    }
                }
            } catch (Exception e) {
                CookieUtil.deleteCookie(COOKIE_KEY.ACCOUNT_KEY);
                logger.error("parse cookie happen error", e);
            }
        }
        return accountInfo;
    }

    /**
     * 得到当前登录用户ID.
     *
     * @return the user id
     */
    public Long getCid() {
        if (cid == null && getAccountInfo() != null) {
            cid = accountInfo.getLong(COOKIE_KEY.ACCOUNT_ID_KEY);
        }
        return cid;
    }

    public HttpServletRequest getRequest() {
        return request;
    }

    public void setRequest(HttpServletRequest request) {
        this.request = request;
    }

    public HttpServletResponse getResponse() {
        return response;
    }

    public void setResponse(HttpServletResponse response) {
        this.response = response;
    }

}
