/**
 * Project Name:media-biz<br>
 * File Name:AccountBackendBOImpl.java<br>
 * Package Name:cn.com.duiba.tuia.media.bo.impl<br>
 * Date:2016年9月30日下午3:45:49<br>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.<br>
 */
package cn.com.duiba.tuia.media.bo.impl;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import cn.com.duiba.tuia.media.common.constants.AccCheckRecordStatus;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.common.utils.BlowfishUtils;
import cn.com.duiba.tuia.media.common.utils.MessageDigestUtils;
import cn.com.duiba.developer.center.api.domain.dto.DeveloperDto;
import cn.com.duiba.developer.center.api.domain.paramquery.CreateDeveloperParams;
import cn.com.duiba.developer.center.api.remoteservice.RemoteDeveloperService;
import cn.com.duiba.tuia.media.api.dto.AccountDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqUpdateAccountCheckStatus;
import cn.com.duiba.tuia.media.api.dto.req.ReqUpdateAccountFreezeStatus;
import cn.com.duiba.tuia.media.bo.AccountBackendBO;
import cn.com.duiba.tuia.media.domain.AccountBankDto;
import cn.com.duiba.tuia.media.domain.AccountCheckRecordDto;
import cn.com.duiba.tuia.media.model.Email;
import cn.com.duiba.tuia.media.model.req.ReqRegister;
import cn.com.duiba.tuia.media.model.req.ReqUpdateAccountBank;
import cn.com.duiba.tuia.media.model.req.ReqUpdateAuditData;
import cn.com.duiba.tuia.media.model.rsp.AccountInfoRsp;
import cn.com.duiba.tuia.media.model.rsp.RegisterRsp;
import cn.com.duiba.tuia.media.service.AccountBankService;
import cn.com.duiba.tuia.media.service.AccountCheckRecordService;
import cn.com.duiba.tuia.media.service.AccountService;
import cn.com.duiba.tuia.media.service.EmailService;
import cn.com.duiba.tuia.media.service.SmsService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.redis.RedisClient;

/**
 * ClassName: AccountBackendBOImpl <br/>
 * date: 2016年9月30日 下午3:45:49 <br/>
 * .
 *
 * @author xiawei
 * @version
 * @since JDK 1.6
 */
@Service
public class AccountBackendBOImpl implements AccountBackendBO {

    /** The logger. */
    protected Logger                  logger          = LoggerFactory.getLogger(getClass());

    /** The login encrypt key. */
    @Value("${media.login.encrypt.key}")
    private String                    loginEncryptKey;

    /** The executor service. */
    private ExecutorService           executorService = Executors.newFixedThreadPool(3);

    /** The account service. */
    @Autowired
    private AccountService            accountService;

    /** The account bank service. */
    @Autowired
    private AccountBankService        accountBankService;

    /** The email service. */
    @Autowired
    private EmailService              emailService;

    @Autowired
    protected RedisClient             redisClient;

    /** The account check record service. */
    @Autowired
    private AccountCheckRecordService accountCheckRecordService;

    /** The remote developer service. */
    @Autowired
    private RemoteDeveloperService    remoteDeveloperService;

    /** The email verify key. */
    @Value("${media.email.verify.key}")
    private String                    emailVerifyKey;

    @Autowired
    private SmsService                smsService;

    /**
     * Register.
     *
     * @param req the req
     * @return the register rsp
     * @throws TuiaMediaException the tuia media exception
     */
    @Override
    public RegisterRsp register(ReqRegister req) throws TuiaMediaException {
        AccountDto accountDto = new AccountDto();
        // 判断邮箱是否存在
        if (accountService.isEmailExists(req.getEmail())) {
            logger.error("the account is exist, the email=[{}]", req.getEmail());
            throw new TuiaMediaException(ErrorCode.E0102011);
        }
        if (smsService.verify(req.getType(), req.getLinkPhone(), req.getCode())) {

            // 获取媒体用户ID
            Long mediaId;
            CreateDeveloperParams params = new CreateDeveloperParams();
            params.setEmail(req.getEmail());
            params.setCompany(req.getCompanyName());
            params.setName(req.getLinkman());
            params.setPassword(req.getPassword());
            params.setPhone(req.getLinkPhone());

            DubboResult<Long> mediaIdResult = remoteDeveloperService.createDeveloper(params, false);
            if (!mediaIdResult.isSuccess()) {
                // 捕获异常
                logger.error("the developer account is exist", mediaIdResult.getMsg());
                DubboResult<DeveloperDto> developerDtoResult = remoteDeveloperService.findDeveloperByEmail(req.getEmail());
                if (!developerDtoResult.isSuccess() || developerDtoResult.getResult() == null) {
                    // 抛出异常
                    logger.error("can not get developer account", developerDtoResult.getMsg());
                    throw new TuiaMediaException(ErrorCode.E0102001);
                }
                mediaId = developerDtoResult.getResult().getId();
            } else {
                mediaId = mediaIdResult.getResult();
            }

            // 密码加密
            String srePassword = BlowfishUtils.encryptBlowfish(MessageDigestUtils.SHA(req.getPassword()),
                                                               loginEncryptKey);

            accountDto.setMediaId(mediaId);
            accountDto.setPassword(srePassword);
            accountDto.setEmail(req.getEmail());
            accountDto.setBusinessLicenseId(req.getBusinessLicenseId());
            accountDto.setBusinessLicenseName(req.getBusinessLicenseName());
            accountDto.setBusinessLicenseUrl(req.getBusinessLicenseUrl());
            accountDto.setCompanyName(req.getCompanyName());
            accountDto.setLinkman(req.getLinkman());
            accountDto.setLinkPhone(req.getLinkPhone());

            // 插入账户信息
            boolean flag = accountService.insertAccountBase(accountDto);

            if (!flag) {
                // 抛出异常
                logger.error("add account error");
                throw new TuiaMediaException(ErrorCode.E0102020);
            }
            // 插入媒体银行账号
            AccountBankDto accountBankDto = new AccountBankDto();
            accountBankDto.setMediaId(accountDto.getMediaId());
            accountBankDto.setBankName(req.getBankName());
            accountBankDto.setCardNumber(req.getCardNumber());
            accountBankDto.setProvince(req.getProvince());
            accountBankDto.setCity(req.getCity());
            accountBankDto.setBranchName(req.getBranchName());
            accountBankService.insertAccountBank(accountBankDto);

        }
        // 发送激活邮件
        accountService.doSendVerifyEmail(accountDto);
        RegisterRsp rsp = new RegisterRsp();
        rsp.setcName(accountDto.getLinkman());
        rsp.setEmail(accountDto.getEmail());
        rsp.setMediaId(accountDto.getMediaId());
        return rsp;
    }

    /**
     * Login.
     *
     * @param email the email
     * @param password the password
     * @return the long
     * @throws TuiaMediaException the tuia media exception
     */
    @Override
    public Long login(String email, String password) throws TuiaMediaException {
        AccountDto accountDto = accountService.selectByEmail(email);
        accountService.accountDtoIsNull(accountDto);
        String srePassword = BlowfishUtils.encryptBlowfish(MessageDigestUtils.SHA(password), loginEncryptKey);
        if (!StringUtils.equals(srePassword, accountDto.getPassword())) {
            logger.error("the password is error, email =[{}]", password);
            throw new TuiaMediaException(ErrorCode.E0102002);
        }
        // 账号状态校验
        Long getMediaId = accountDto.getMediaId();
        accountStatusAuth(getMediaId, accountDto.getCheckStatus(), accountDto.getFreezeStatus());

        return accountDto.getMediaId();
    }

    /**
     * 校验账号状态.
     *
     * @param getMediaId the get media id
     * @param auditStatus 账号审核状态
     * @param freezeStatus 冻结状态
     * @throws TuiaMediaException the tuia media exception
     */
    private void accountStatusAuth(long getMediaId, Integer auditStatus, Integer freezeStatus)
                                                                                              throws TuiaMediaException {
        doCheckAudit(getMediaId, auditStatus);
        doCheckFreeze(getMediaId, freezeStatus);
    }

    /**
     * 校验账户是否冻结. <br/>
     *
     * @param mediaId the media id
     * @param freezeStatus the freeze status
     * @throws TuiaMediaException the tuia media exception
     * @author xiawei
     * @since JDK 1.6
     */
    private void doCheckFreeze(long mediaId, Integer freezeStatus) throws TuiaMediaException {
        if (AccountDto.FREEZED_STATUS == freezeStatus) {

            // 已冻结，查询冻结原因
            AccountCheckRecordDto accountCheckRecordDto = accountCheckRecordService.selectLatelyCheckRecord(mediaId,
                                                                                                            AccCheckRecordStatus.FREEZE);

            if (accountCheckRecordDto == null) {
                logger.error(" the getAccountCheckRecord is null, the accountId=[{}]", mediaId);
                throw new TuiaMediaException(ErrorCode.E0001005);
            }
            // 被冻结状态
            logger.error("the account is freeze, the freeze status=[{}]", freezeStatus);
            throw new TuiaMediaException(ErrorCode.E0102004.getErrorCode(), "账号被冻结，因为："
                                                                            + accountCheckRecordDto.getReason());
        }
        if (AccountDto.UNFREEZED_STATUS != freezeStatus) {
            // 非法的冻结状态(既不是冻结也不是未冻结)
            logger.error("the account freeze status is illegal,  the freeze status=[{}]", freezeStatus);
            throw new TuiaMediaException(ErrorCode.E0102005);
        }
    }

    /**
     * 校验账户是否审核. <br/>
     *
     * @param mediaId the media id
     * @param auditStatus the audit status
     * @throws TuiaMediaException the tuia media exception
     * @author xiawei
     * @since JDK 1.6
     */
    private void doCheckAudit(long mediaId, Integer auditStatus) throws TuiaMediaException {
        if (AccountDto.CHECK_STATUS_ING == auditStatus) {
            // 审核中状态
            logger.error("user account in review status, the audit status=[{}]", auditStatus);
            throw new TuiaMediaException(ErrorCode.E0102006);
        }
        if (AccountDto.CHECK_STATUS_REFUSE == auditStatus) {
            // 审核驳回状态
            AccountCheckRecordDto accountCheckRecordDto = accountCheckRecordService.selectLatelyCheckRecord(mediaId,
                                                                                                            AccCheckRecordStatus.CHECK_REFUSE);

            if (accountCheckRecordDto == null) {
                logger.error(" the getAccountCheckRecord is null, the accountId=[{}]", mediaId);
                throw new TuiaMediaException(ErrorCode.E0001005);
            }
            // 被拒绝状态
            logger.error("user account in refuse status, the audit status=[{}]", auditStatus);
            throw new TuiaMediaException(ErrorCode.E0102007.getErrorCode(), "账号被审核拒绝，因为："
                                                                            + accountCheckRecordDto.getReason());
        }
        if (AccountDto.CHECK_STATUS_PASS != auditStatus) {
            // 非法的审核状态
            logger.error("the account status is illegal, the audit status=[{}]", auditStatus);
            throw new TuiaMediaException(ErrorCode.E0102008);
        }
    }

    /**
     * Gets the account info.
     *
     * @param mediaId the media id
     * @return the account info
     * @throws TuiaMediaException the tuia media exception
     */
    @Override
    public AccountInfoRsp getAccountInfo(Long mediaId) throws TuiaMediaException {
        // 获取媒体账户基本信息
        AccountDto accountDto = accountService.selectByMediaId(mediaId);
        accountService.accountDtoIsNull(accountDto);
        AccountInfoRsp accountInfoRsp = new AccountInfoRsp();
        doAccountBaseInfo(accountDto, accountInfoRsp);
        // 获取媒体账户财务信息
        AccountBankDto accountBankDto = accountBankService.selectByMediaId(mediaId);
        if (accountBankDto == null) {
            logger.error("account not exist, the media=[{}]", mediaId);
            throw new TuiaMediaException(ErrorCode.E0102001);
        }
        doAccountBankInfo(accountInfoRsp, accountBankDto);

        return accountInfoRsp;
    }

    @Override
    public AccountInfoRsp getUncheckAccountInfo(String email) throws TuiaMediaException {
        // 获取媒体账户基本信息
        AccountDto accountDto = accountService.selectByEmail(email);
        accountService.accountDtoIsNull(accountDto);
        AccountInfoRsp accountInfoRsp = new AccountInfoRsp();
        doAccountBaseInfo(accountDto, accountInfoRsp);
        // 获取媒体账户财务信息
        AccountBankDto accountBankDto = accountBankService.selectByMediaId(accountDto.getMediaId());
        if (accountBankDto == null) {
            logger.error("account not exist, the media=[{}]", accountDto.getMediaId());
            throw new TuiaMediaException(ErrorCode.E0102001);
        }
        doAccountBankInfo(accountInfoRsp, accountBankDto);

        return accountInfoRsp;
    }

    /**
     * 账户基本信息. <br/>
     * 
     * @author xiawei
     * @param accountInfoRsp
     * @param accountBankDto
     * @since JDK 1.6
     */
    private void doAccountBankInfo(AccountInfoRsp accountInfoRsp, AccountBankDto accountBankDto) {
        accountInfoRsp.setCardNumber(accountBankDto.getCardNumber());
        accountInfoRsp.setBankName(accountBankDto.getBankName());
        accountInfoRsp.setProvince(accountBankDto.getProvince());
        accountInfoRsp.setCity(accountBankDto.getCity());
        accountInfoRsp.setBranchName(accountBankDto.getBranchName());
    }

    /**
     * 账户财务信息. <br/>
     *
     * @author xiawei
     * @param accountDto
     * @param accountInfoRsp
     * @since JDK 1.6
     */
    private void doAccountBaseInfo(AccountDto accountDto, AccountInfoRsp accountInfoRsp) {
        accountInfoRsp.setMediaId(accountDto.getMediaId());
        accountInfoRsp.setCompanyName(accountDto.getCompanyName());
        accountInfoRsp.setBusinessLicenseId(accountDto.getBusinessLicenseId());
        accountInfoRsp.setBusinessLicenseName(accountDto.getBusinessLicenseName());
        accountInfoRsp.setBusinessLicenseUrl(accountDto.getBusinessLicenseUrl());
        accountInfoRsp.setEmail(accountDto.getEmail());
        accountInfoRsp.setLinkman(accountDto.getLinkman());
        accountInfoRsp.setLinkPhone(accountDto.getLinkPhone());
    }

    /**
     * Update audit data.
     *
     * @param accountInfo the account info
     * @return true, if update audit data
     * @throws TuiaMediaException the tuia media exception
     */
    @Override
    public boolean updateAuditData(ReqUpdateAuditData accountInfo) throws TuiaMediaException {
        if (smsService.verify(accountInfo.getType(), accountInfo.getLinkPhone(), accountInfo.getCode())) {
            AccountDto currentData = accountService.selectByMediaId(accountInfo.getMediaId());
            if (AccountDto.CHECK_STATUS_REFUSE != currentData.getCheckStatus()) {
                // 只有处于审核拒绝状态时才可以进行修改审核资料
                throw new TuiaMediaException(ErrorCode.E0102015);
            }
            AccountDto baseReq = new AccountDto();
            baseReq.setMediaId(accountInfo.getMediaId());
            baseReq.setBusinessLicenseId(accountInfo.getBusinessLicenseId());
            baseReq.setBusinessLicenseName(accountInfo.getBusinessLicenseName());
            baseReq.setBusinessLicenseUrl(accountInfo.getBusinessLicenseUrl());
            baseReq.setCompanyName(accountInfo.getCompanyName());
            baseReq.setLinkman(accountInfo.getLinkman());
            baseReq.setLinkPhone(accountInfo.getLinkPhone());
            // 更新媒体账户基本信息
            boolean flag = accountService.updateAuditData(baseReq);

            if (flag) {
                ReqUpdateAccountBank bankReq = new ReqUpdateAccountBank();
                bankReq.setMediaId(accountInfo.getMediaId());
                bankReq.setCardNumber(accountInfo.getCardNumber());
                bankReq.setBankName(accountInfo.getBankName());
                bankReq.setProvince(accountInfo.getProvince());
                bankReq.setCity(accountInfo.getCity());
                bankReq.setBranchName(accountInfo.getBranchName());
                // 更新媒体账户信息
                accountBankService.updateAccountBank(bankReq);

            }
        }
        return true;
    }

    /**
     * 是否为相同邮箱.
     *
     * @param accountInfo the account info
     * @param currentData the current data
     * @return true, if checks if is same email
     * @author xiawei
     * @since JDK 1.6
     */
    // private boolean isSameEmail(ReqUpdateAuditData accountInfo, AccountDto currentData) {
    // return StringUtils.equals(currentData.getEmail(), accountInfo.getEmail());
    // }

    /**
     * Update check status.
     *
     * @param mediaId the media id
     * @param checkType the check type
     * @param reason the reason
     * @return true, if update check status
     * @throws TuiaMediaException the tuia media exception
     */
    @Override
    public boolean updateCheckStatus(Long mediaId, Integer checkType, String reason) throws TuiaMediaException {

        // 是否能更新审核状态
        AccountDto accountDto = canUpdateCheckStatus(mediaId);

        int status = -1;
        String checkTypeMessage = "审核通过";

        // 如果是审核通过操作
        if (ReqUpdateAccountCheckStatus.PARAM_CHECK_PASS.intValue() == checkType) {
            status = AccountDto.CHECK_STATUS_PASS;
        }

        // 如果是审核拒绝操作
        if (ReqUpdateAccountCheckStatus.PARAM_CHECK_REFUSE.intValue() == checkType) {
            checkTypeMessage = "审核拒绝";
            status = AccountDto.CHECK_STATUS_REFUSE;
        }

        // 修改媒体账户审核状态
        boolean flag = accountService.updateCheckStatus(mediaId, status);

        // 增加账号审核状态记录
        if (flag) {
            AccountCheckRecordDto checkRecord = new AccountCheckRecordDto();
            checkRecord.setMediaId(mediaId);
            checkRecord.setCheckType(checkType);
            checkRecord.setReason(reason);
            checkRecord.setGmtCreate(checkRecord.getGmtCreate());
            checkRecord.setGmtModified(checkRecord.getGmtModified());
            accountCheckRecordService.insertCheckRecord(checkRecord);

            // 发送审核邮件
            doSendEmail(checkTypeMessage, accountDto);
            return true;
        }
        return false;
    }

    /**
     * 是否能更新审核状态. <br/>
     *
     * @author xiawei
     * @param mediaId
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    private AccountDto canUpdateCheckStatus(Long mediaId) throws TuiaMediaException {
        AccountDto accountDto = accountService.selectByMediaId(mediaId);

        // 邮箱未激活
        if (accountDto != null && accountDto.getEmailStatus() == AccountDto.EMAIL_STATUS_UNCHECK) {
            logger.error("the media is not active, email = [{}]", accountDto.getEmail());
            throw new TuiaMediaException(ErrorCode.E0102019);
        }
        // 已经审核过
        if (isChecked(accountDto)) {
            logger.error("the media is checked ,mediaId = [{}]", mediaId);
            throw new TuiaMediaException(ErrorCode.E0304007);
        }
        return accountDto;
    }

    /**
     * 是否已经审核. <br/>
     *
     * @author xiawei
     * @param accountDto
     * @return
     * @since JDK 1.6
     */
    private boolean isChecked(AccountDto accountDto) {
        return AccountDto.CHECK_STATUS_PASS == accountDto.getCheckStatus()
               || AccountDto.CHECK_STATUS_REFUSE == accountDto.getCheckStatus();
    }

    /**
     * Update freeze status.
     *
     * @param mediaId the media id
     * @param checkType the check type
     * @param reason the reason
     * @return true, if update freeze status
     * @throws TuiaMediaException the tuia media exception
     */
    @Override
    public boolean updateFreezeStatus(Long mediaId, Integer checkType, String reason) throws TuiaMediaException {
        int status = -1;
        String checkTypeMessage = "未冻结";
        // 解冻操作
        if (ReqUpdateAccountFreezeStatus.PARAM_UNFREEZE.intValue() == checkType) {
            status = AccountDto.UNFREEZED_STATUS;

        }
        // 如果是冻结操作
        if (ReqUpdateAccountFreezeStatus.PARAM_FREEZE.intValue() == checkType) {
            checkTypeMessage = "已冻结";
            status = AccountDto.FREEZED_STATUS;
        }

        // 修改媒体账户冻结状态
        boolean flag = accountService.updateFreezeStatus(mediaId, status);

        // 增加账号审核状态记录
        if (flag) {

            AccountCheckRecordDto checkRecord = new AccountCheckRecordDto();
            checkRecord.setMediaId(mediaId);
            checkRecord.setCheckType(checkType);
            checkRecord.setReason(reason);
            checkRecord.setGmtCreate(checkRecord.getGmtCreate());
            checkRecord.setGmtModified(checkRecord.getGmtModified());
            accountCheckRecordService.insertCheckRecord(checkRecord);

            // 查询账号信息
            AccountDto mediaDto = accountService.selectByMediaId(mediaId);
            if (mediaDto == null || mediaDto.getEmail() == null) {
                logger.error("the media is not exist, the mediaId=[{}]", mediaId);
                throw new TuiaMediaException(ErrorCode.E0102001);
            }
            // 发送审核邮件
            doSendEmail(checkTypeMessage, mediaDto);
            return true;
        }
        return false;
    }

    /**
     * 发送审核邮件.
     *
     * @param checkTypeMessage the check type message
     * @param accountDto the account dto
     * @author xiawei
     * @since JDK 1.6
     */
    private void doSendEmail(String checkTypeMessage, AccountDto accountDto) {
        Map<String, Object> model = new HashMap<>();
        model.put("linkman", accountDto.getLinkman());
        model.put("email", accountDto.getEmail());
        model.put("checkTypeMessage", checkTypeMessage);

        // 拼装邮件内容
        final Email email = new Email();
        email.setTo(accountDto.getEmail());
        email.setTemplateLocation("/templates/mail/checkStatus.vm");
        email.setSubject("账号状态变化通知");
        email.setModel(model);
        // 放入线程池中执行
        executorService.submit(new Runnable() {

            @Override
            public void run() {
                emailService.sendWithTemplate(email);
            }
        });
    }

}
