/**
 * Project Name:media-biz File Name:DataStatisticsBOImpl.java Package Name:cn.com.duiba.tuia.media.bo.impl
 * Date:2016年10月12日下午5:20:36 Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 */

package cn.com.duiba.tuia.media.bo.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.common.tool.DataTool;
import cn.com.duiba.tuia.media.api.dto.DataStatisticsDto;
import cn.com.duiba.tuia.media.api.dto.MediaAppStatisticsDto;
import cn.com.duiba.tuia.media.api.dto.PageResultDto;
import cn.com.duiba.tuia.media.api.dto.SlotStatisticsDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqSlotLineStatisticsByPageDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspSlotLineStatisticsDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspMediaAppDataStatisticsDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspSlotDataStatisticsDto;
import cn.com.duiba.tuia.media.bo.SlotLineStatisticsBO;
import cn.com.duiba.tuia.media.dao.MediaAppDao;
import cn.com.duiba.tuia.media.dao.SlotDAO;
import cn.com.duiba.tuia.media.model.IdAndName;
import cn.com.duiba.tuia.media.service.SlotLineStatisticsService;

/**
 * ClassName:DataStatisticsBOImpl <br/>
 * Function: 结算数据. <br/>
 * Date: 2016年10月12日 下午5:20:36 <br/>
 * 
 * @author guyan
 * @version
 * @since JDK 1.6
 * @see
 */
@Service
public class SlotLineStatisticsBOImpl implements SlotLineStatisticsBO {

    @Autowired
    SlotLineStatisticsService slotLineStatisticsService;

    @Autowired
    SlotDAO                   slotDao;

    @Autowired
    MediaAppDao               mediaAppDao;

    @Override
    public PageResultDto<RspMediaAppDataStatisticsDto> getMediaAppStatisticsData(ReqSlotLineStatisticsByPageDto param)
                                                                                                                      throws TuiaMediaException {
        int totalNum = this.slotLineStatisticsService.getMediaAppDataAmount(param);
        List<Long> mediaAppIds;
        List<MediaAppStatisticsDto> mediaAppList;
        List<RspMediaAppDataStatisticsDto> rspMediaAppList = null;
        if ((totalNum > 0) && (totalNum >= param.getRowStart())) {
            param.setPageSize(ReqSlotLineStatisticsByPageDto.PAGE_SIZE_10);
            param.setRowStart(param.getPageSize() * (param.getCurrentPage() - 1));
            mediaAppList = this.slotLineStatisticsService.getMediaAppStatisticsData(param);
            mediaAppIds = new ArrayList<>(mediaAppList.size());
            for (MediaAppStatisticsDto mediaAppDto : mediaAppList) {
                mediaAppIds.add(mediaAppDto.getAppId());
            }
            Map<Long, String> mediaAppMap = getMediaAppMapByIds(mediaAppIds);
            rspMediaAppList = rspMediaAppList(mediaAppList, mediaAppMap);
        }
        return new PageResultDto<RspMediaAppDataStatisticsDto>(totalNum, rspMediaAppList, param.getPageSize());
    }

    @Override
    public PageResultDto<RspSlotDataStatisticsDto> getSlotStatisticsData(ReqSlotLineStatisticsByPageDto param)
                                                                                                              throws TuiaMediaException {
        int totalNum = this.slotLineStatisticsService.getSlotDataAmount(param);
        List<Long> soltIds;
        List<SlotStatisticsDto> slotList;
        List<RspSlotDataStatisticsDto> rspSlotList = null;
        if ((totalNum > 0) && (totalNum >= param.getRowStart())) {
            param.setPageSize(ReqSlotLineStatisticsByPageDto.PAGE_SIZE_10);
            param.setRowStart(param.getPageSize() * (param.getCurrentPage() - 1));
            slotList = this.slotLineStatisticsService.getSlotStatisticsData(param);
            soltIds = new ArrayList<>(slotList.size());
            for (SlotStatisticsDto slotDto : slotList) {
                soltIds.add(slotDto.getSlotId());
            }
            Map<Long, String> soltMap = getSoltMapByIds(soltIds);
            rspSlotList = rspSlotList(slotList, soltMap);
        }
        return new PageResultDto<RspSlotDataStatisticsDto>(totalNum, rspSlotList, param.getPageSize());
    }

    @Override
    public RspSlotLineStatisticsDto selectStatisticsDataByDate(ReqSlotLineStatisticsByPageDto param)
                                                                                                    throws TuiaMediaException {
        param.setOrder(ReqSlotLineStatisticsByPageDto.ORDER_TYPE);
        param.setSort(ReqSlotLineStatisticsByPageDto.SORT_TYPE);
        List<DataStatisticsDto> dataStatisticsList = this.slotLineStatisticsService.getStatisticsDataByDate(param);
        RspSlotLineStatisticsDto rspDto = dataFormat(dataStatisticsList);
        return rspDto;
    }

    /**
     * 广告信息List转换成Map
     * 
     * @param soltIds
     * @return
     * @throws TuiaMediaException
     */
    private Map<Long, String> getSoltMapByIds(List<Long> soltIds) throws TuiaMediaException {
        List<IdAndName> list;
        Map<Long, String> slotMap = null;
        if (CollectionUtils.isNotEmpty(soltIds)) {
            slotMap = new HashMap<>(soltIds.size());
            list = slotDao.selectAppIdAndName(soltIds);
            for (IdAndName slotVO : list) {
                // key:广告位Id value:广告位名称
                slotMap.put(slotVO.getId(), slotVO.getName());
            }
        }
        return slotMap;
    }

    /**
     * 媒体app List转换成map
     * 
     * @param mediaAppIds
     * @return
     * @throws TuiaMediaException
     */
    private Map<Long, String> getMediaAppMapByIds(List<Long> mediaAppIds) throws TuiaMediaException {
        List<IdAndName> list;
        Map<Long, String> mediaAppMap = null;
        if (CollectionUtils.isNotEmpty(mediaAppIds)) {
            mediaAppMap = new HashMap<>(mediaAppIds.size());
            list = mediaAppDao.selectAppIdAndName(mediaAppIds);
            for (IdAndName mediaAppVO : list) {
                // key:媒体app Id value:媒体app 名称
                mediaAppMap.put(mediaAppVO.getId(), mediaAppVO.getName());
            }
        }
        return mediaAppMap;

    }

    /**
     * 拼装广告统计数据
     * 
     * @param slotList
     * @param slotMap
     * @return
     */
    private List<RspSlotDataStatisticsDto> rspSlotList(List<SlotStatisticsDto> slotList, Map<Long, String> slotMap) {
        List<RspSlotDataStatisticsDto> rspSoltList = new ArrayList<>(slotList.size());
        for (SlotStatisticsDto dto : slotList) {
            RspSlotDataStatisticsDto rspDto = new RspSlotDataStatisticsDto();
            rspDto.setSlotId(dto.getSlotId());
            rspDto.setSlotType(dto.getSlotType());
            rspDto.setClickCount(dto.getClickCount());
            rspDto.setConsumeTotal(dto.getConsumeTotal());
            rspDto.setExposureCount(dto.getExposureCount());
            rspDto.setSlotName(slotMap.get(dto.getSlotId()));
            rspSoltList.add(rspDto);
        }
        return rspSoltList;
    }

    /**
     * 拼装媒体app统计数据
     * 
     * @param mediaAppList
     * @param mediaAppMap
     * @return
     */
    private List<RspMediaAppDataStatisticsDto> rspMediaAppList(List<MediaAppStatisticsDto> mediaAppList,
                                                               Map<Long, String> mediaAppMap) {
        List<RspMediaAppDataStatisticsDto> rspSoltList = new ArrayList<>(mediaAppList.size());
        for (MediaAppStatisticsDto dto : mediaAppList) {
            RspMediaAppDataStatisticsDto rspDto = new RspMediaAppDataStatisticsDto();
            rspDto.setAppId(dto.getAppId());
            rspDto.setAppName(mediaAppMap.get(dto.getAppId()));
            rspDto.setClickCount(dto.getClickCount());
            rspDto.setConsumeTotal(dto.getConsumeTotal());
            rspDto.setExposureCount(dto.getExposureCount());
            rspDto.setPlatform(dto.getPlatform());
            rspSoltList.add(rspDto);
        }
        return rspSoltList;
    }

    /**
     * 拼装成折线图的数据格式
     * 
     * @param dataList
     * @return
     */
    private RspSlotLineStatisticsDto dataFormat(List<DataStatisticsDto> dataList) {
        List<String> curDateList;
        List<Long> exposureCountList;
        List<Long> clickCountList;
        List<Long> consumeTotalList;
        List<Long> ecpmList;
        List<Long> clickRateList;
        RspSlotLineStatisticsDto rspDto = new RspSlotLineStatisticsDto();
        if (CollectionUtils.isNotEmpty(dataList)) {
            curDateList = new ArrayList<>(dataList.size());
            exposureCountList = new ArrayList<>(dataList.size());
            clickCountList = new ArrayList<>(dataList.size());
            consumeTotalList = new ArrayList<>(dataList.size());
            ecpmList = new ArrayList<>(dataList.size());
            clickRateList = new ArrayList<>(dataList.size());
            for (DataStatisticsDto dto : dataList) {
                curDateList.add(dto.getCurDate());
                clickCountList.add(dto.getClickCount());
                exposureCountList.add(dto.getExposureCount());
                ecpmList.add(calculateEcpmData(dto));
                consumeTotalList.add(dto.getConsumeTotal());
                clickRateList.add(calculateClickRateData(dto));
            }
            rspDto.setClickCount(clickCountList);
            rspDto.setConsumeTotal(consumeTotalList);
            rspDto.setCurDate(curDateList);
            rspDto.seteCpm(ecpmList);
            rspDto.setExposureCount(exposureCountList);
            rspDto.setClickRate(clickRateList);
        }
        return rspDto;
    }

    /**
     * 计算ecpm统计数据.
     *
     * @param data the data
     */
    private Long calculateEcpmData(DataStatisticsDto data) {
        // eCPM=(预计收入/曝光量*1000)
        return DataTool.calculateEcpm(data.getConsumeTotal(), data.getExposureCount().longValue());
    }

    /**
     * 计算点击率 点击量/曝光量 *100
     * 
     * @param data
     * @return
     */
    private Long calculateClickRateData(DataStatisticsDto data) {
        // clickRate=(点击量/曝光量 *100)
        return DataTool.calculateClickRade(data.getClickCount().longValue(), data.getExposureCount().longValue());
    }

}
