/**
 * Project Name:media-biz<br>
 * File Name:MediaCacheServiceImpl.java<br>
 * Package Name:cn.com.duiba.tuia.media.service.impl<br>
 * Date:2016年9月30日下午3:00:03<br>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.<br>
 */

package cn.com.duiba.tuia.media.service.impl;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.api.dto.AccountDto;
import cn.com.duiba.tuia.media.api.dto.MediaAppDataDto;
import cn.com.duiba.tuia.media.api.dto.MediaAppDto;
import cn.com.duiba.tuia.media.dao.AccountDAO;
import cn.com.duiba.tuia.media.dao.MediaAppDao;
import cn.com.duiba.tuia.media.utils.CacheKeyUtils;

import com.alibaba.fastjson.JSONObject;

/**
 * ClassName: MediaCacheServiceImpl <br/>
 * Function: 媒体、APP缓存. <br/>
 * date: 2016年9月30日 下午3:00:03 <br/>
 *
 * @author leiliang
 * @version
 * @since JDK 1.6
 */
@Service
public class MediaCacheService extends BaseCacheService {

    /** The media app dao. */
    @Autowired
    MediaAppDao        mediaAppDao;

    /** The account dao. */
    @Autowired
    private AccountDAO accountDAO;

    /**
     * 更新媒体用户缓存.<br>
     * [使用说明]
     * <ol>
     * <li>当修改媒体账号信息时：isValid=false</li>
     * <li>当媒体账号审核通过时：isValid=true</li>
     * <li>当媒体账号审核拒绝时：不用更新</li>
     * <li>当媒体账号冻结时：isValid=false</li>
     * <li>当媒体账号解冻时：isValid=true</li>
     * </ol>
     * 
     * @param mediaId 媒体用户ID
     * @param isValid the is valid
     */
    public void updateMediaCache(final long mediaId, final boolean isValid) {
        executorService.submit(new Runnable() {

            @Override
            public void run() {
                redisClient.setex(CacheKeyUtils.getMediaStatusKey(mediaId), ONE_WEEK, JSONObject.toJSONString(isValid));
            }
        });
    }

    /**
     * 更新媒体APP缓存.
     *
     * @param mediaAppId 媒体APP ID
     */
    public void updateMediaAppCache(final long mediaAppId) {
        executorService.submit(new Runnable() {

            @Override
            public void run() {
                try {
                    doGetMediaApp(mediaAppId);
                } catch (Exception e) {
                    logger.error("updateMediaAppCache error", e);
                }
            }
        });
    }

    /**
     * 获取媒体APP缓存.
     *
     * @param mediaAppId 媒体APPID
     * @return the media app
     * @throws TuiaMediaException the tuia media exception
     */
    public MediaAppDataDto getMediaApp(long mediaAppId) throws TuiaMediaException {
        String strMedaiAppData = redisClient.get(CacheKeyUtils.getMediaAppKey(mediaAppId));
        MediaAppDataDto cacheData = null;

        if (StringUtils.isEmpty(strMedaiAppData)) {
            cacheData = doGetMediaApp(mediaAppId);

        } else {
            cacheData = JSONObject.parseObject(strMedaiAppData, MediaAppDataDto.class);
        }
        if (cacheData == null) {
            // 媒体应用不存在
            throw new TuiaMediaException(ErrorCode.E0401001);
        }
        return cacheData;
    }

    /**
     * 通过app key查询.
     *
     * @param appKey the app key
     * @return the media app by key
     * @throws TuiaMediaException the tuia media exception
     */
    public MediaAppDataDto getMediaAppByKey(String appKey) throws TuiaMediaException {
        String strMedaiAppData = redisClient.get(CacheKeyUtils.getMediaAppByKey(appKey));
        MediaAppDataDto cacheData = null;

        if (StringUtils.isEmpty(strMedaiAppData)) {
            cacheData = doGetMediaAppByKey(appKey);

        } else {
            cacheData = JSONObject.parseObject(strMedaiAppData, MediaAppDataDto.class);
        }
        if (cacheData == null) {
            // 媒体应用不存在
            throw new TuiaMediaException(ErrorCode.E0401001);
        }
        return cacheData;
    }

    /**
     * 执行获取媒体应用操作.
     *
     * @param mediaAppId 媒体应用ID
     * @return the media app data dto
     * @throws TuiaMediaException the tuia media exception
     */
    private MediaAppDataDto doGetMediaApp(long mediaAppId) throws TuiaMediaException {
        MediaAppDataDto cacheData = null;
        MediaAppDto mediaAppDto = mediaAppDao.selectMediaAppById(mediaAppId);
        if (mediaAppDto != null) {
            cacheData = buildMediaAppDataDto(mediaAppDto);
            // 更新缓存
            redisClient.setex(CacheKeyUtils.getMediaAppKey(mediaAppId), ONE_WEEK, JSONObject.toJSONString(cacheData));
            redisClient.setex(CacheKeyUtils.getMediaAppByKey(mediaAppDto.getAppKey()), ONE_WEEK,
                              JSONObject.toJSONString(cacheData));
        }
        return cacheData;
    }

    /**
     * 获取应用缓存.
     *
     * @param appkey the appkey
     * @return the media app data dto
     * @throws TuiaMediaException the tuia media exception
     */
    private MediaAppDataDto doGetMediaAppByKey(String appkey) throws TuiaMediaException {
        MediaAppDataDto cacheData = null;
        MediaAppDto mediaAppDto = mediaAppDao.selectByAppKey(appkey);
        if (mediaAppDto != null) {
            cacheData = buildMediaAppDataDto(mediaAppDto);
            // 更新缓存
            redisClient.setex(CacheKeyUtils.getMediaAppByKey(appkey), ONE_WEEK, JSONObject.toJSONString(cacheData));
        }
        return cacheData;
    }

    /**
     * 构建app缓存存储对象.
     *
     * @param mediaAppDto the media app dto
     * @return the media app data dto
     */
    private MediaAppDataDto buildMediaAppDataDto(MediaAppDto mediaAppDto) {
        MediaAppDataDto cacheData = new MediaAppDataDto(mediaAppDto.getMediaId(), mediaAppDto.getAppKey(),
                                                        mediaAppDto.getAppSecret(), mediaAppDto.getIndustry());
        cacheData.setValid(MediaAppDto.MEDIA_APP_CHECK_PASS == mediaAppDto.getCheckStatus() ? true : false);

        cacheData.setAppId(mediaAppDto.getAppId());
        return cacheData;
    }

    /**
     * 媒体APP是否有效.
     *
     * @param mediaAppId 媒体APPID
     * @return true：有效(只有媒体APP为审核通过并且所属媒体账号为邮箱验证通过、审核通过、未冻结的情况下才有效)
     * @throws TuiaMediaException the tuia media exception
     */
    public Boolean isValidMediaApp(long mediaAppId) throws TuiaMediaException {
        MediaAppDataDto cacheData = getMediaApp(mediaAppId);
        if (cacheData != null && cacheData.isValid() && doGetMediaValid(cacheData.getMediaId())) {
            return true;
        }
        return false;
    }

    /**
     * 查询媒体有效状态.<br>
     *
     * @param mediaId 媒体ID
     * @return true：审核通过、邮箱验证通过、未冻结
     * @throws TuiaMediaException the tuia media exception
     */
    private Boolean doGetMediaValid(long mediaId) throws TuiaMediaException {
        String mediaValid = redisClient.get(CacheKeyUtils.getMediaStatusKey(mediaId));
        boolean isValid = false;
        if (mediaValid == null) {
            AccountDto accountDto = accountDAO.selectById(mediaId);
            if (accountDto == null) {
                throw new TuiaMediaException(ErrorCode.E0102001);
            }
            if (AccountDto.CHECK_STATUS_PASS == accountDto.getCheckStatus()
                && AccountDto.EMAIL_STATUS_CHECKED == accountDto.getEmailStatus()
                && AccountDto.UNFREEZED_STATUS == accountDto.getFreezeStatus()) {
                isValid = true;
            }

            redisClient.setex(CacheKeyUtils.getMediaStatusKey(mediaId), ONE_WEEK, JSONObject.toJSONString(isValid));
        }

        return isValid;
    }

}
