/**
 * Project Name:media-biz<br>
 * File Name:SlotCacheServiceImpl.java<br>
 * Package Name:cn.com.duiba.tuia.media.service.impl<br>
 * Date:2016年9月30日下午4:48:25<br>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.<br>
 */

package cn.com.duiba.tuia.media.service.impl;

import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.tuia.media.api.dto.SlotCacheDto;
import cn.com.duiba.tuia.media.api.dto.SlotDto;
import cn.com.duiba.tuia.media.api.dto.StrategyDto;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.constants.SlotPictureSizeEnmu;
import cn.com.duiba.tuia.media.common.constants.SlotPictureSizeEnmu.PictureSize;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.dao.ShieldStrategyDAO;
import cn.com.duiba.tuia.media.dao.SlotDAO;
import cn.com.duiba.tuia.media.domain.ShieldStrategyDto;
import cn.com.duiba.tuia.media.utils.CacheKeyUtils;

import com.alibaba.fastjson.JSONObject;

/**
 * ClassName: SlotCacheServiceImpl <br/>
 * Function: 广告位缓存层. <br/>
 * date: 2016年9月30日 下午4:48:25 <br/>
 *
 * @author leiliang
 * @version
 * @since JDK 1.6
 */
@Service
public class SlotCacheService extends BaseCacheService {

    @Autowired
    private SlotDAO           slotDAO;

    @Autowired
    private ShieldStrategyDAO shieldStrategyDAO;

    /**
     * 获取广告位信息.
     *
     * @param slotId 广告位ID
     * @return 广告位信息
     * @throws TuiaMediaException the tuia media exception
     */
    public SlotCacheDto getSlot(long slotId) throws TuiaMediaException {
        String strSlot = redisClient.get(CacheKeyUtils.getSlotKey(slotId));
        SlotCacheDto slotCacheDto = null;
        if (StringUtils.isEmpty(strSlot)) {
            slotCacheDto = doGetSlot(slotId);
        } else {
            slotCacheDto = JSONObject.parseObject(strSlot, SlotCacheDto.class);
        }
        if (slotCacheDto == null) {
            // 广告位不存在
            throw new TuiaMediaException(ErrorCode.E0304008);
        }
        return slotCacheDto;
    }

    /**
     * 更新广告位信息.
     *
     * @param slotId 广告位ID
     */
    public void updateSlotCache(final long slotId) {
        executorService.submit(new Runnable() {

            @Override
            public void run() {
                try {
                    doGetSlot(slotId);
                } catch (TuiaMediaException e) {
                    logger.error("updateSlotCache error", e);
                }

            }
        });
    }

    /**
     * 批量更新广告位信息.
     *
     * @param slotIds 广告位ID列表
     */
    public void updatebatchSlotCache(final List<Long> slotIds) {
        executorService.submit(new Runnable() {

            @Override
            public void run() {
                try {
                    if (CollectionUtils.isNotEmpty(slotIds)) {
                        for (Long slotId : slotIds) {
                            doGetSlot(slotId);
                        }
                    }

                } catch (TuiaMediaException e) {
                    logger.error("updateSlotCache error", e);
                }

            }
        });
    }

    /**
     * 执行查询广告位信息.
     *
     * @param slotId 广告位id
     * @return 广告位信息
     * @throws TuiaMediaException
     */
    private SlotCacheDto doGetSlot(long slotId) throws TuiaMediaException {
        SlotDto slotDto = slotDAO.selectById(slotId);
        SlotCacheDto slotCacheDto = null;
        if (slotDto != null) {
            PictureSize pictureSize = SlotPictureSizeEnmu.getPictureSizeById(slotDto.getPictureSize());
            if (pictureSize == null) {
                // 广告位尺寸异常
                logger.error(" the slot=[{}] pictureSizeId=[{}] non-existent", slotId, slotDto.getPictureSize());
                return null;
            }
            slotCacheDto = new SlotCacheDto(slotId, slotDto.getSlotType(), pictureSize.getWidth(),
                                            pictureSize.getHeight(), slotDto.getStrategyId());

            redisClient.setex(CacheKeyUtils.getSlotKey(slotId), ONE_WEEK, JSONObject.toJSONString(slotCacheDto));
        }
        return slotCacheDto;
    }

    /**
     * 获取广告位的屏蔽策略信息.
     *
     * @param slotId 广告位ID
     * @return 广告位的屏蔽策略信息
     * @throws TuiaMediaException the tuia media exception
     */
    public StrategyDto getStrategy(long slotId) throws TuiaMediaException {
        Long strategyId = getStrategyIdBySlotId(slotId);
        if (strategyId == null) {
            // 该广告位没有屏蔽策略
            return null;
        }
        StrategyDto strategyDto = null;

        String strategy = redisClient.get(CacheKeyUtils.getStrategyKey(strategyId));
        if (StringUtils.isEmpty(strategy)) {
            // 查询屏蔽策略信息
            strategyDto = doGetStrategy(strategyId);
        } else {
            strategyDto = JSONObject.parseObject(strategy, StrategyDto.class);
        }
        return strategyDto;
    }

    /**
     * 更新屏蔽策略信息.
     *
     * @param slotId 广告位id
     * @return 屏蔽策略信息
     * @throws TuiaMediaException the tuia media exception
     */
    public StrategyDto updateStrategy(long slotId) throws TuiaMediaException {
        Long strategyId = getStrategyIdBySlotId(slotId);
        if (strategyId == null) {
            // 该广告位没有屏蔽策略
            return null;
        }

        return doGetStrategy(strategyId);
    }

    /**
     * 通过广告位ID查询该广告位的屏蔽策略ID.
     *
     * @param slotId 广告位ID
     * @return 屏蔽策略ID
     * @throws TuiaMediaException the tuia media exception
     */
    private Long getStrategyIdBySlotId(Long slotId) throws TuiaMediaException {
        // 查询出屏蔽策略ID
        SlotCacheDto slot = getSlot(slotId);
        if (slot == null) {
            // 广告位不存在
            throw new TuiaMediaException(ErrorCode.E0304008);
        }
        return slot.getStrategyId();
    }

    /**
     * 执行查询屏蔽策略操作.
     *
     * @param strategyId 屏蔽策略ID
     * @return the strategy dto
     * @throws TuiaMediaException the tuia media exception
     */
    private StrategyDto doGetStrategy(long strategyId) throws TuiaMediaException {
        StrategyDto strategyDto = null;

        ShieldStrategyDto shieldStrategyDto = shieldStrategyDAO.selectById(strategyId);
        if (shieldStrategyDto != null) {
            strategyDto = new StrategyDto(shieldStrategyDto.getShieldIndustries(), shieldStrategyDto.getShieldUrls());
            // 塞入缓存
            redisClient.setex(CacheKeyUtils.getStrategyKey(strategyId), ONE_WEEK, JSONObject.toJSONString(strategyDto));
        }
        return strategyDto;
    }
}
