/**
 * Project Name:media-biz<br/>
 * File Name:AccountChangeServiceImpl.java<br/>
 * Package Name:cn.com.duiba.tuia.media.service.impl<br/>
 * Date:2016年12月6日下午8:41:52<br/>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 */

package cn.com.duiba.tuia.media.service.impl;

import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.ListUtils;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.tuia.media.api.dto.AccountChangeDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqAccountChange;
import cn.com.duiba.tuia.media.api.dto.rsp.AccountChangeRsp;
import cn.com.duiba.tuia.media.api.dto.rsp.RspAccountChangeResult;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.common.tool.DataTool;
import cn.com.duiba.tuia.media.dao.AccountChangeDAO;
import cn.com.duiba.tuia.media.service.AccountChangeService;

/**
 * ClassName:AccountChangeServiceImpl <br/>
 * Date: 2016年12月6日 下午8:41:52 <br/>
 * 
 * @author ZFZ
 * @version
 * @since JDK 1.6
 * @see
 */
@Service
public class AccountChangeServiceImpl implements AccountChangeService {
    /** The logger. */
    protected Logger                  logger          = LoggerFactory.getLogger(getClass());
    
    /** The Constant WORD_SEPARATOR. */
    private static final String  WORD_SEPARATOR = "\",\"";

    /** The Constant START_PREFIX. */
    private static final String  START_PREFIX   = "\" ";

    /** The Constant END_POSTFIX. */
    private static final String  END_POSTFIX    = "\"\n";
    
    @Autowired
    private AccountChangeDAO accountChangeDAO;

    @SuppressWarnings("unchecked")
    @Override
    public RspAccountChangeResult<AccountChangeRsp> selectAccountChangeByPage(ReqAccountChange req)
                                                                                                 throws TuiaMediaException {
        int row = accountChangeDAO.selectAccountChangeCount(req);
        // 当总数查询结果有数据时，再查询具体数据
        List<AccountChangeDto> accountChangeDtos = null;
        List<AccountChangeRsp> accountChangeRsps = new ArrayList<>();
        Long balance = 0L;
        if (row > 0 && row >= req.getRowStart()) {
            req.setRowStart(req.getPageSize() * (req.getCurrentPage() - 1));
            accountChangeDtos = accountChangeDAO.selectAccountChangeByPage(req);
            balance = accountChangeDAO.selectAccountBalance(req.getMediaId());
        }

        if (CollectionUtils.isNotEmpty(accountChangeDtos)) {
            for (AccountChangeDto dto : accountChangeDtos) {
                AccountChangeRsp rsp = new AccountChangeRsp();
                rsp.setAmount(dto.getAmount());
                rsp.setBalance(dto.getBalance());
                rsp.setChangeType(dto.getChangeType());
                rsp.setNote(dto.getRemark());
                rsp.setCurrentDate(new DateTime(dto.getCurDate()).toString("yyyy-MM-dd HH:mm:ss"));
                accountChangeRsps.add(rsp);
            }
        } else {
            accountChangeRsps = ListUtils.EMPTY_LIST;
        }
        return new RspAccountChangeResult<>(row, accountChangeRsps, req.getPageSize(), balance);
    }

    @Override
    public void insertAccountChange(AccountChangeDto accountChangeDto) throws TuiaMediaException {
        accountChangeDAO.insertAccountChange(accountChangeDto);
    }

    @Override
    public List<AccountChangeDto> selectAccountChangeListByIds(List<Long> ids) throws TuiaMediaException {
        return accountChangeDAO.selectAccountChangeListByIds(ids);
    }

    @Override
    public AccountChangeDto selectAccountChangeById(Long id) throws TuiaMediaException {
        return accountChangeDAO.selectAccountChangeById(id);
    }

    @Override
    public AccountChangeDto selectAccountChangeByMediaId(Long mediaId) throws TuiaMediaException {
        return accountChangeDAO.selectAccountChangeByMediaId(mediaId);
    }

    @Override
    public List<AccountChangeDto> selectAccountChangeList(ReqAccountChange req) throws TuiaMediaException {
        return accountChangeDAO.selectAccountChangeList(req);
    }
    
    
    @Override
    public void exportAccountChange(ReqAccountChange req, OutputStream out)throws TuiaMediaException {
        StringBuilder content = new StringBuilder();
        content.append(" 时间 ,明细,金额（元）,余额（元）\n");
        try {
            List<AccountChangeDto> datas = accountChangeDAO.selectAccountChangeList(req);
            if (CollectionUtils.isNotEmpty(datas)) {
                for (AccountChangeDto data : datas) {
                    StringBuilder row = new StringBuilder();
                    row.append(START_PREFIX);
                    row.append(new DateTime(data.getCurDate()).toString("yyyy-MM-dd HH:mm:ss"));
                    row.append(WORD_SEPARATOR);
                    row.append(data.getRemark());
                    row.append(WORD_SEPARATOR);
                    row.append("+" + DataTool.calculateFenToYuan(data.getAmount()));
                    row.append(WORD_SEPARATOR);
                    row.append(DataTool.calculateFenToYuan(data.getBalance()));
                    row.append(END_POSTFIX);
                    content.append(row);
                }
            }
            // 导出文件
            doExportFile(out, content);
        } catch (TuiaMediaException e) {
            logger.error("exportAccountChange error!", e);
            throw new TuiaMediaException(ErrorCode.E0002006);
        }
    }

    /**
     * 执行导出文件操作.
     *
     * @param out 输出流
     * @param content 带导出的内容
     * @throws TuiaException the tuia exception
     */
    private void doExportFile(OutputStream out, StringBuilder content) throws TuiaMediaException {
        try {
            out.write(content.toString().getBytes("GBK"));
            out.flush();
        } catch (Exception e) {
            logger.error("doExportFile error!", e);
            throw new TuiaMediaException(ErrorCode.E0002006, e);
        }
    }

}
