/**
 * Project Name:media-biz<br>
 * File Name:ActivityStatisticsDayBOImpl.java<br>
 * Package Name:cn.com.duiba.tuia.media.bo.impl<br>
 * Date:2016年10月18日下午3:30:48<br>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.<br>
 */

package cn.com.duiba.tuia.media.bo.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import cn.com.duiba.tuia.media.api.constant.ActivityTypeConstant;
import cn.com.duiba.tuia.media.common.tool.DataTool;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import cn.com.duiba.activity.center.api.dto.activity.AdvActivityDto;
import cn.com.duiba.activity.center.api.remoteservice.activity.RemoteOperatingActivityServiceNew;
import cn.com.duiba.tuia.media.api.dto.BaseActivityDataDto;
import cn.com.duiba.tuia.media.api.dto.BaseQueryDto;
import cn.com.duiba.tuia.media.api.dto.PageResultDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqGetActivityData;
import cn.com.duiba.tuia.media.api.dto.rsp.RspGetActivityData;
import cn.com.duiba.tuia.media.api.dto.rsp.RspGetActivityDetailData;
import cn.com.duiba.tuia.media.bo.ActivityStatisticsDayBO;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.domain.ActivityStatisticsDayDto;
import cn.com.duiba.tuia.media.model.IdAndName;
import cn.com.duiba.tuia.media.model.req.ReqActivityData;
import cn.com.duiba.tuia.media.service.ActivityStatisticsDayService;
import cn.com.duiba.tuia.media.service.MediaAppService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import org.springframework.stereotype.Service;

/**
 * ClassName: ActivityStatisticsDayBOImpl <br/>
 * Function: 活动数据. <br/>
 * date: 2016年10月18日 下午3:30:48 <br/>
 *
 * @author leiliang
 * @version
 * @since JDK 1.6
 */
@Service
public class ActivityStatisticsDayBOImpl implements ActivityStatisticsDayBO {

    /** 活动中心支持的最大批量查询长度为20. */
    private static final int                  ACTIVITY_CENTER_MAX_BACTCH_LENGTH = 20;

    /** The logger. */
    private Logger                            logger                            = LoggerFactory.getLogger(ActivityStatisticsDayBOImpl.class);

    @Autowired
    private ActivityStatisticsDayService      activityStatisticsDayService;

    @Autowired
    private RemoteOperatingActivityServiceNew remoteOperatingActivityServiceNew;

    @Autowired
    private MediaAppService                   mediaAppService;

    @Override
    public PageResultDto<RspGetActivityData> getActivityDataByPage(ReqGetActivityData req) throws TuiaMediaException {
        // 请求参数转换
        ReqActivityData queryParam = buildReqActivityData(req);

        // 待返回的结果
        List<RspGetActivityData> resultActivityDatas = null;
        // 查询活动记录总数
        int totalAmount = activityStatisticsDayService.selectAmountByPage(queryParam);
        if (totalAmount > 0 && totalAmount >= queryParam.getRowStart()) {

            List<ActivityStatisticsDayDto> activityDataList = activityStatisticsDayService.selectListByPage(queryParam);

            Map<Long, String> activityNameMap = buildActivityNameMap(activityDataList);
            resultActivityDatas = new ArrayList<>(activityDataList.size());

            for (ActivityStatisticsDayDto activityData : activityDataList) {
                RspGetActivityData rspGetActivityData = new RspGetActivityData();
                resultActivityDatas.add(rspGetActivityData);

                rspGetActivityData.setActivityId(activityData.getDuibaActivityId());
                rspGetActivityData.setActivityName(activityNameMap.get(activityData.getDuibaActivityId()));
                rspGetActivityData.setActivityType(activityData.getDuibaActivityType());
                rspGetActivityData.setActivityTypeName(ActivityTypeConstant.getTypeText(activityData.getDuibaActivityType()));
                buidBaseActivityData(rspGetActivityData, activityData);
            }
        }

        // 组装数据
        return new PageResultDto<>(totalAmount, resultActivityDatas, queryParam.getPageSize());
    }

    /**
     * 构建查询参数.
     *
     * @param req the req
     * @return the req activity data
     */
    private ReqActivityData buildReqActivityData(ReqGetActivityData req) {
        ReqActivityData queryParam = new ReqActivityData(req.getActivityType(), req.getActivityId(), req.getStartTime(),
                                                         req.getEndTime());

        queryParam.setPageSize(req.getPageSize() == null ? BaseQueryDto.PAGE_SIZE_20 : req.getPageSize());
        queryParam.setRowStart((req.getCurrentPage() - 1) * req.getPageSize());
        queryParam.setOrder(req.getOrder());
        queryParam.setSort(req.getSort());

        return queryParam;
    }

    /**
     * 拼装活动数据.
     *
     * @param baseActivityDataDto the rsp get activity data
     * @param activityData the activity data
     */
    private void buidBaseActivityData(BaseActivityDataDto baseActivityDataDto, ActivityStatisticsDayDto activityData) {
        baseActivityDataDto.setActivityUv(activityData.getActivityUv());
        baseActivityDataDto.setClickAdvertCount(activityData.getClickAdvertCount());
        baseActivityDataDto.setClickCount(activityData.getClickCount());
        baseActivityDataDto.setConsumeTotal(activityData.getConsumeTotal());
        baseActivityDataDto.setInvolveCounts(activityData.getInvolveCounts());
        baseActivityDataDto.setParticipantCounts(activityData.getParticipantCounts());
        // 活动每日曝光数
        long exposureCount = activityData.getExposureCount();
        baseActivityDataDto.setExposureCount(exposureCount);
        // 活动发券量
        long launchAdvertCount = activityData.getLaunchAdvertCount();
        baseActivityDataDto.setLaunchAdvertCount(launchAdvertCount);

        // eCPM(活动每千次展示获得的广告收入)=每日消费/(活动每日曝光数*1000)
        baseActivityDataDto.seteCPM(DataTool.calculateEcpm(activityData.getConsumeTotal(), exposureCount));
        // 曝光点击率=点击量/曝光量
        baseActivityDataDto.setExposureCTR(DataTool.extendHundred(DataTool.calculateRatio(activityData.getClickCount(),
                                                                                          exposureCount)));
        // 发券成功率=活动发券量/活动参与次数
        baseActivityDataDto.setLaunchRate(DataTool.extendHundred(DataTool.calculateRatio(launchAdvertCount,activityData.getInvolveCounts())));
        
        // 人均参与次数=活动参与次数/活动参与人数
        baseActivityDataDto.setPerParticipate(DataTool.calculateAveragePrice(activityData.getInvolveCounts(),
                                                                             activityData.getParticipantCounts()));
        // 参与率 =活动参人数/活动UV
        baseActivityDataDto.setRate(DataTool.extendHundred(DataTool.calculateRatio(activityData.getParticipantCounts(),
                                                                                   activityData.getActivityUv())));
        // spm=活动发券数量/活动曝光*1000
        baseActivityDataDto.setSpm(DataTool.calculateEcpm(launchAdvertCount, exposureCount));
    }

    /**
     * 构建Map<活动ID, 活动名称>.
     *
     * @param activityDataList the activity data list
     * @return the map< long, string>
     * @throws TuiaMediaException the tuia media exception
     */
    private Map<Long, String> buildActivityNameMap(List<ActivityStatisticsDayDto> activityDataList) throws TuiaMediaException {
        int size = activityDataList.size();
        List<Long> activityIds = new ArrayList<>(size);
        for (ActivityStatisticsDayDto activityData : activityDataList) {
            activityIds.add(activityData.getDuibaActivityId());
        }

        Map<Long, String> activityNameMap = new HashMap<>(size);

        // 计算列表长度(考虑为空的情况)
        List<AdvActivityDto> advActivityDtoList = new ArrayList<>(size);
        // 批量最大执行长度为100
        for (int i = 1, j = 0; i <= size; i++) {
            if (i % ACTIVITY_CENTER_MAX_BACTCH_LENGTH == 0 || i == size) {
                List<AdvActivityDto> activityDtos = doGetAdvActivtyDto(activityIds.subList(j, i));
                if (CollectionUtils.isNotEmpty(activityDtos)) {
                    advActivityDtoList.addAll(activityDtos);
                }
                j = i;
            }
        }

        if (CollectionUtils.isNotEmpty(advActivityDtoList)) {
            for (AdvActivityDto advActivityDto : advActivityDtoList) {
                activityNameMap.put(advActivityDto.getOperatingActiivtyId(), advActivityDto.getName());
            }
        }

        return activityNameMap;
    }

    /**
     * 执行去活动中心根据活动ID列表批量查询活动名称的操作.
     *
     * @param activityIds the activity ids
     * @return the list< adv activity dto>
     * @throws TuiaMediaException the tuia media exception
     */
    private List<AdvActivityDto> doGetAdvActivtyDto(List<Long> activityIds) throws TuiaMediaException {
        DubboResult<List<AdvActivityDto>> dubboResult = remoteOperatingActivityServiceNew.batchQueryAdvActivty(activityIds);
        if (dubboResult.isResultSuccess()) {
            logger.error("[AC] remoteOperatingActivityServiceNew.batchQueryAdvActivty error!, the reason=[{}]",
                         dubboResult.getMsg());
            throw new TuiaMediaException(ErrorCode.E9999999);
        }
        return dubboResult.getResult();
    }

    /**
     * 构建Map<媒体ID, 媒体名称>.
     *
     * @param activityDataList the activity data list
     * @return the map< long, string>
     * @throws TuiaMediaException the tuia media exception
     */
    private Map<Long, String> buildAppNameMap(List<ActivityStatisticsDayDto> activityDataList) throws TuiaMediaException {
        int size = activityDataList.size();
        List<Long> appIds = new ArrayList<>(size);
        for (ActivityStatisticsDayDto activityData : activityDataList) {
            appIds.add(activityData.getAppId());
        }

        List<IdAndName> appList = mediaAppService.selectAppIdAndName(appIds);

        Map<Long, String> appNameMap = new HashMap<>(size);
        for (IdAndName app : appList) {
            appNameMap.put(app.getId(), app.getName());
        }

        return appNameMap;
    }

    @Override
    public PageResultDto<RspGetActivityDetailData> getActivityDetailDataByPage(ReqGetActivityData req) throws TuiaMediaException {
        // 请求参数转换
        ReqActivityData queryParam = buildReqActivityData(req);

        // 待返回的结果
        List<RspGetActivityDetailData> resultActivityDatas = null;
        // 查询活动记录总数
        int totalAmount = activityStatisticsDayService.selectAmountByDetail(queryParam);
        if (totalAmount > 0 && totalAmount >= queryParam.getRowStart()) {

            List<ActivityStatisticsDayDto> activityDataList = activityStatisticsDayService.selectListByDetail(queryParam);

            Map<Long, String> appNameMap = buildAppNameMap(activityDataList);
            resultActivityDatas = new ArrayList<>(activityDataList.size());

            for (ActivityStatisticsDayDto activityData : activityDataList) {
                RspGetActivityDetailData rspGetActivityDetailData = new RspGetActivityDetailData();
                resultActivityDatas.add(rspGetActivityDetailData);

                rspGetActivityDetailData.setAppName(appNameMap.get(activityData.getAppId()));
                rspGetActivityDetailData.setAppId(activityData.getAppId());
                buidBaseActivityData(rspGetActivityDetailData, activityData);
            }
        }

        // 组装数据
        return new PageResultDto<>(totalAmount, resultActivityDatas, queryParam.getPageSize());
    }

    @Override
    public List<RspGetActivityData> getActivityDataList(ReqGetActivityData req) throws TuiaMediaException {
        // 请求参数转换
        ReqActivityData queryParam = new ReqActivityData(req.getActivityType(), req.getActivityId(), req.getStartTime(),
                                                         req.getEndTime());
        queryParam.setPageSize(BaseQueryDto.PAGE_SIZE_MAX);

        List<ActivityStatisticsDayDto> activityDataList = activityStatisticsDayService.selectListByPage(queryParam);

        Map<Long, String> activityNameMap = buildActivityNameMap(activityDataList);
        List<RspGetActivityData> resultActivityDatas = new ArrayList<>(activityDataList.size());

        for (ActivityStatisticsDayDto activityData : activityDataList) {
            RspGetActivityData rspGetActivityData = new RspGetActivityData();
            resultActivityDatas.add(rspGetActivityData);

            rspGetActivityData.setActivityId(activityData.getDuibaActivityId());
            rspGetActivityData.setActivityName(activityNameMap.get(activityData.getDuibaActivityId()));
            rspGetActivityData.setActivityType(activityData.getDuibaActivityType());
            rspGetActivityData.setActivityTypeName(ActivityTypeConstant.getTypeText(activityData.getDuibaActivityType()));
            buidBaseActivityData(rspGetActivityData, activityData);
        }

        return resultActivityDatas;
    }

    @Override
    public List<RspGetActivityDetailData> getActivityDetailDataList(ReqGetActivityData req) throws TuiaMediaException {
        // 请求参数转换
        ReqActivityData queryParam = new ReqActivityData(req.getActivityType(), req.getActivityId(), req.getStartTime(),
                                                         req.getEndTime());
        queryParam.setPageSize(BaseQueryDto.PAGE_SIZE_MAX);

        List<ActivityStatisticsDayDto> activityDataList = activityStatisticsDayService.selectListByDetail(queryParam);

        Map<Long, String> appNameMap = buildAppNameMap(activityDataList);
        List<RspGetActivityDetailData> resultActivityDatas = new ArrayList<>(activityDataList.size());

        for (ActivityStatisticsDayDto activityData : activityDataList) {
            RspGetActivityDetailData rspGetActivityDetailData = new RspGetActivityDetailData();
            resultActivityDatas.add(rspGetActivityDetailData);

            rspGetActivityDetailData.setAppName(appNameMap.get(activityData.getAppId()));
            rspGetActivityDetailData.setAppId(activityData.getAppId());
            buidBaseActivityData(rspGetActivityDetailData, activityData);
        }

        return resultActivityDatas;
    }
}
