package cn.com.duiba.tuia.media.message.consumer;

import java.util.Collections;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;

import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.TopicPartition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Value;

import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;

import com.alibaba.ttl.threadpool.TtlExecutors;

/**
 * Kafka消息消费模板类。
 */
public abstract class AbstractKafkaConsumer implements InitializingBean {

    /**
     * 默认消费线程数，可以通过覆盖 getPoolSize修改。
     */
    public static final int DEFAULT_POOL_SIZE = 1;

    protected Logger        log               = LoggerFactory.getLogger("message");

    @Value("${media.kafka.group.id}")
    private String          kafkaGroupId;

    @Value("${media.kafka.bootstrap.servers}")
    private String          kafkaServers;

    private ExecutorService executorService;

    private boolean         running;

    private void createMessageConsumer() {
        Properties props = new Properties();
        props.put("bootstrap.servers", this.kafkaServers);
        props.put("group.id", this.kafkaGroupId);
        props.put("enable.auto.commit", "false");
        props.put("session.timeout.ms", "30000");
        props.put("auto.offset.reset", "earliest");
        props.put("max.partition.fetch.bytes", "2048");
        props.put("key.deserializer", "org.apache.kafka.common.serialization.StringDeserializer");
        props.put("value.deserializer", "org.apache.kafka.common.serialization.StringDeserializer");
        props.put("max.poll.records", "10");

        KafkaConsumer<String, String> consumer = new KafkaConsumer<>(props);
        consumer.subscribe(getTopics());

        try {
            while (this.running) {
                ConsumerRecords<String, String> records = consumer.poll(100);
                for (TopicPartition pt : records.partitions()) {
                    for (ConsumerRecord<String, String> record : records.records(pt)) {
                        try {
                            readMessage(record);
                            long lastOffset = record.offset();
                            consumer.commitSync(Collections.singletonMap(pt, new OffsetAndMetadata(lastOffset + 1)));
                        } catch (KafkaException e) {
                            log.error(String.format("Kafka Error, offset=[%s]", record.offset()), e);
                        } catch (Exception e) {
                            log.error(String.format("Consumer Error, offset=[%s]", record.offset()), e);
                        }
                    }
                }
            }
        } finally {
            consumer.close();
        }
    }

    /**
     * start:(开启线程). <br/>
     *
     * @author ZFZ
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    public synchronized void start() throws TuiaMediaException {
        if (this.running) {
            return;
        }
        this.running = true;

        this.executorService = TtlExecutors.getTtlExecutorService(Executors.newFixedThreadPool(getPoolSize(), new ThreadFactory() {

            private int i = 0;

            @Override
            public Thread newThread(Runnable runnable) {
                return new Thread(runnable, "Thread-" + getClass().getName() + "-Consumer-" + (i++));
            }
        }));

        for (int i = 0; i < getPoolSize(); i++) {
            this.executorService.submit(new Runnable() {

                @Override
                public void run() {
                    try {
                        createMessageConsumer();
                    } catch (Exception e) {
                        log.error("error:", e);
                    }
                }
            });
        }
        log.info("Consumer started");
    }

    /**
     * stop:(关闭线程). <br/>
     *
     * @author ZFZ
     * @since JDK 1.6
     */
    public synchronized void stop() {
        this.running = false;
        if (this.executorService != null) {
            this.executorService.shutdown();
            this.executorService = null;
        }
    }

    protected int getPoolSize() {
        return DEFAULT_POOL_SIZE;
    }

    protected abstract List<String> getTopics();

    protected abstract void readMessage(ConsumerRecord<String, String> record);

    @Override
    public void afterPropertiesSet() throws Exception {
        start();
    }
}
