package cn.com.duiba.tuia.media.message.consumer;

/**
 * @author xuyenan
 * @createTime 2016/10/27
 */

import java.util.Arrays;
import java.util.List;

import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.domain.ActivityDto;
import cn.com.duiba.tuia.media.message.body.AdActivityMessage;
import cn.com.duiba.tuia.media.service.ActivityService;
import cn.com.duiba.tuia.media.service.ActivitySortService;
import cn.com.duiba.tuia.media.utils.JsonUtils;

/**
 * 兑吧活动信息变更消息接收器。
 * <p>
 * 在兑吧活动中心,活动信息修改后，会通过kafka发送消息，这个类接受消息后会同步到广告系统中。
 */
@Component
public class ActivityKafkaConsumer extends AbstractKafkaConsumer {

    @Autowired
    private ActivityService     activityService;

    @Autowired
    private ActivitySortService activitySortService;

    @Value("${media.kafka.activity.topic}")
    private String              topic;

    @Override
    protected List<String> getTopics() {
        return Arrays.asList(this.topic);
    }

    @Override
    protected void readMessage(ConsumerRecord<String, String> record) {
        String key = record.key();
        String message = record.value();
        Long offset = record.offset();
        try {
            log.info("ActivityKafkaConsumer begin message consume, key=[{}], offset=[{}], message=[{}]", key, offset, message);
            AdActivityMessage adActivityMessage = JsonUtils.jsonToObject(AdActivityMessage.class, message);
            if (adActivityMessage != null) {
                switch (adActivityMessage.getAction()) {
                    case AdActivityMessage.ACTION_INSERT_TYPE:
                        createActivityDto(adActivityMessage);
                        break;
                    case AdActivityMessage.ACTION_UPDATE_TYPE:
                        updateActivityDto(adActivityMessage);
                        break;
                    case AdActivityMessage.ACTION_DELETE_TYPE:
                        deleteActivityDto(adActivityMessage);
                        break;
                    default:
                        break;
                }
            }
            log.info("ActivityKafkaConsumer end message consume");
        } catch (Exception e) {
            log.error(String.format("ActivityKafkaConsumer error, key=[%s], offset=[%s], message=[%s]", key, offset, message), e);
        }
    }

    private void deleteActivityDto(AdActivityMessage adActivityMessage) throws TuiaMediaException {
        activityService.deleteActivity(adActivityMessage.getActivityId(), adActivityMessage.getActivityType());
        activitySortService.deleteActivity(adActivityMessage.getActivityId(), adActivityMessage.getActivityType(), null);
    }

    private void updateActivityDto(AdActivityMessage adActivityMessage) throws TuiaMediaException {
        ActivityDto activityDto = new ActivityDto();
        activityDto.setActivityId(adActivityMessage.getActivityId());
        activityDto.setActivityType(adActivityMessage.getActivityType());
        if (adActivityMessage.getActivityStatus() != null) {
            activityDto.setActivityStatus(adActivityMessage.getActivityStatus());
        }
        if (adActivityMessage.getActivityName() != null) {
            activityDto.setActivityName(adActivityMessage.getActivityName());
        }
        if (adActivityMessage.getTag() != null) {
            activityDto.setTag(adActivityMessage.getTag());
        }
        activityService.updateActivity(activityDto);
    }

    private void createActivityDto(AdActivityMessage adActivityMessage) throws TuiaMediaException {
        ActivityDto activityDto = new ActivityDto();
        activityDto.setActivityId(adActivityMessage.getActivityId());
        activityDto.setActivityStatus(adActivityMessage.getActivityStatus());
        activityDto.setActivityType(adActivityMessage.getActivityType());
        activityDto.setActivityName(adActivityMessage.getActivityName());
        activityDto.setTag(adActivityMessage.getTag());
        activityService.insertActivity(activityDto);
    }
}
