package cn.com.duiba.tuia.media.service.impl;

import java.util.*;

import cn.com.duiba.tuia.media.api.constant.MaterialSpecificationConstant;
import cn.com.duiba.tuia.media.api.dto.rsp.*;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import com.google.common.base.Joiner;

import cn.com.duiba.tuia.media.api.constant.ActivityConstant;
import cn.com.duiba.tuia.media.api.dto.req.ReqActivityDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqIdAndType;
import cn.com.duiba.tuia.media.api.dto.req.ReqMaterialSpecificationDto;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.dao.ActivityDAO;
import cn.com.duiba.tuia.media.dao.MaterialSpecificationItemContentDAO;
import cn.com.duiba.tuia.media.dataobject.MaterialSpecificationItemContentDO;
import cn.com.duiba.tuia.media.domain.ActivityDto;
import cn.com.duiba.tuia.media.parser.ActivityDtoParser;
import cn.com.duiba.tuia.media.service.ActivityService;
import cn.com.duiba.tuia.media.service.ActivitySortService;
import cn.com.duiba.tuia.media.service.MaterialSpecificationService;
import cn.com.duiba.tuia.media.service.cacheservice.ActivityCacheService;

/**
 * @author xuyenan
 * @createTime 2016/12/21
 */
@Service("activityNewService")
public class ActivityServiceImpl implements ActivityService {

    @Autowired
    private ActivityDAO                         activityDAO;

    @Autowired
    private MaterialSpecificationItemContentDAO materialSpecificationItemContentDAO;

    @Autowired
    private ActivityCacheService                activityCacheService;

    @Autowired
    private MaterialSpecificationService        materialSpecificationService;

    @Autowired
    private ActivitySortService                 activitySortService;

    @Override
    public List<RspActivityDto> getPlanList(ReqActivityDto req) throws TuiaMediaException {
        req.setSort("gmt_modified");
        req.setOrder("desc");
        req.setRowStart(req.getPageSize() * (req.getCurrentPage() - 1));
        List<ActivityDto> result = activityDAO.getList(req);
        List<RspActivityDto> list = new ArrayList<>();
        for (ActivityDto activityDto : result) {
            RspActivityDto rspActivityDto = ActivityDtoParser.parseToRspActivityDto(activityDto);
            // 查询素材数量
            rspActivityDto.setItemContentAmount(materialSpecificationItemContentDAO.itemContentAmount(rspActivityDto.getId(), rspActivityDto.getType()));
            list.add(rspActivityDto);
        }
        return list;
    }

    @Override
    public int getPlanListAmount(ReqActivityDto req) throws TuiaMediaException {
        return activityDAO.getListAmount(req);
    }

    @Transactional
    @Override
    public Boolean addActivityPlan(List<ReqIdAndType> activities) throws TuiaMediaException {
        for (ReqIdAndType idAndType : activities) {
            ActivityDto activityDto = new ActivityDto();
            activityDto.setActivityId(idAndType.getId());
            activityDto.setActivityType(idAndType.getType());
            activityDto.setDirectAdvertMode(ActivityConstant.DIRECT_ADVERT_MODE_ONLY);
            activityDto.setIsPublish(ActivityConstant.IS_PUBLISH);
            activityDAO.update(activityDto);
        }
        return true;
    }

    @Transactional
    @Override
    public Boolean updateActivityPlan(ReqActivityDto reqActivityDto) throws TuiaMediaException {
        // 更新活动关联规格
        List<Long> activityMsIdList = reqActivityDto.getMsIdList();
        ActivityDto activityDto = new ActivityDto();
        activityDto.setActivityId(reqActivityDto.getId());
        activityDto.setActivityType(reqActivityDto.getType());
        activityDto.setActivityMsList(Joiner.on(",").appendTo(new StringBuilder(), activityMsIdList).toString());
        activityDAO.update(activityDto);
        // 更新素材列表,先删除再添加
        List<RspMaterialSpecificationItemContentDto> itemContentDtoList = reqActivityDto.getItemContentList();
        materialSpecificationItemContentDAO.delete(reqActivityDto.getId(), reqActivityDto.getType());
        if (!CollectionUtils.isEmpty(itemContentDtoList)) {
            List<MaterialSpecificationItemContentDO> msItemContentList = new ArrayList<>();
            for (RspMaterialSpecificationItemContentDto msItemContent : itemContentDtoList) {
                MaterialSpecificationItemContentDO itemContentDO = new MaterialSpecificationItemContentDO();
                BeanUtils.copyProperties(msItemContent, itemContentDO);
                msItemContentList.add(itemContentDO);
            }
            materialSpecificationItemContentDAO.insertBatch(msItemContentList);
        }
        return true;
    }

    @Override
    public RspActivityDto getActivityPlanDetail(Long activityId, Integer activityType) throws TuiaMediaException {
        ActivityDto activityDto = activityCacheService.get(activityId, activityType);
        if (activityDto == null) {
            activityDto = activityDAO.getDetail(activityId, activityType);
        }
        if (activityDto == null) {
            throw new TuiaMediaException(ErrorCode.E0501004);
        }
        return ActivityDtoParser.parseToRspActivityDto(activityDto);
    }

    @Override
    public Boolean changeActivityPlanEnable(Long activityId, Integer activityType, Integer enableStatus) throws TuiaMediaException {
        if (ActivityConstant.IS_NOT_ENABLE != enableStatus && ActivityConstant.IS_ENABLE != enableStatus) {
            throw new TuiaMediaException(ErrorCode.E0001005);
        }
        ActivityDto activityDto = new ActivityDto();
        activityDto.setActivityId(activityId);
        activityDto.setActivityType(activityType);
        activityDto.setIsEnable(enableStatus);
        return activityDAO.update(activityDto) == 1;
    }

    @Transactional
    @Override
    public Boolean deleteActivityPlan(Long activityId, Integer activityType) throws TuiaMediaException {
        ActivityDto activityDto = new ActivityDto();
        activityDto.setActivityId(activityId);
        activityDto.setActivityType(activityType);
        activityDto.setIsPublish(ActivityConstant.IS_NOT_PUBLISH);
        // 删除所有定制列表中的该活动计划
        activitySortService.deleteActivity(activityId, activityType, null);
        return activityDAO.update(activityDto) == 1;
    }

    @Override
    public List<RspActivityDto> getActivityByIds(List<Long> ids) throws TuiaMediaException {
        List<ActivityDto> result = activityDAO.getByIds(ids);
        List<RspActivityDto> list = new ArrayList<>();
        for (ActivityDto activityDto : result) {
            list.add(ActivityDtoParser.parseToRspActivityDto(activityDto));
        }
        return list;
    }

    @Override
    public Boolean insertActivity(ActivityDto activityDto) throws TuiaMediaException {
        return activityDAO.insert(activityDto) == 1;
    }

    @Override
    public Boolean updateActivity(ActivityDto activityDto) throws TuiaMediaException {
        return activityDAO.update(activityDto) == 1;
    }

    @Override
    public Boolean deleteActivity(Long activityId, Integer activityType) throws TuiaMediaException {
        ActivityDto activityDto = new ActivityDto();
        activityDto.setActivityId(activityId);
        activityDto.setActivityType(activityType);
        activityDto.setIsDelete(ActivityConstant.IS_DELETE);
        return activityDAO.update(activityDto) == 1;
    }

    @Override
    public String getActivityName(Long activityId, Integer activityType) throws TuiaMediaException {
        ActivityDto activityDto = activityDAO.getDetail(activityId, activityType);
        if (activityDto != null) {
            return activityDto.getActivityName();
        }
        return null;
    }

    @Override
    public RspActivityDto getMsItemContent(RspActivityDto rspActivityDto) throws TuiaMediaException {
        if (rspActivityDto == null) {
            return null;
        }
        // 查询素材列表
        List<RspMaterialSpecificationDto> msItemContentList = new ArrayList<>();
        List<Long> msIds = rspActivityDto.getMsIdList();
        if (CollectionUtils.isEmpty(msIds)) {
            return rspActivityDto;
        }
        Iterator<Long> iterator = msIds.iterator();
        while (iterator.hasNext()) {
            Long msId = iterator.next();
            RspMaterialSpecificationDto rspMaterialSpecificationDto = materialSpecificationService.getMaterialSpecificationDetail(msId);
            if (MaterialSpecificationConstant.MS_DELETE == rspMaterialSpecificationDto.getIsDelete()) {
                iterator.remove();
            }
            List<MaterialSpecificationItemContentDO> list = materialSpecificationItemContentDAO.getByActivity(rspActivityDto.getId(), rspActivityDto.getType(), msId);
            // 构造规格素材列表MAP，给下面创意的一些字段赋值用
            List<RspMaterialSpecificationItemDto> msItemList = rspMaterialSpecificationDto.getItemDtoList();
            Map<Long, RspMaterialSpecificationItemDto> msItemMap = new HashMap<>();
            for (RspMaterialSpecificationItemDto msItem : msItemList) {
                msItemMap.put(msItem.getId(), msItem);
            }
            // 创意列表，根据位置分类
            if (!CollectionUtils.isEmpty(list)) {
                List<RspMaterialSpecificationAssortDto> sortItemContent = new ArrayList<>(list.get(list.size() - 1).getItemIndex());
                for (int i = 0; i < list.get(list.size() - 1).getItemIndex(); i++) {
                    RspMaterialSpecificationAssortDto msAssortDto = new RspMaterialSpecificationAssortDto();
                    msAssortDto.setId(i);
                    msAssortDto.setValue(new ArrayList<RspMaterialSpecificationItemContentDto>());
                    sortItemContent.add(msAssortDto);
                }
                for (MaterialSpecificationItemContentDO msItemContent : list) {
                    RspMaterialSpecificationItemContentDto rspMsItemContent = new RspMaterialSpecificationItemContentDto();
                    BeanUtils.copyProperties(msItemContent, rspMsItemContent);
                    // 附加字段赋值
                    RspMaterialSpecificationItemDto msItem = msItemMap.get(rspMsItemContent.getMsItemId());
                    if (msItem != null) {
                        rspMsItemContent.setItemType(msItem.getItemType());
                        rspMsItemContent.setDescription(msItem.getDescription());
                        rspMsItemContent.setImageHeight(msItem.getImageHeight());
                        rspMsItemContent.setImageWidth(msItem.getImageWidth());
                    }
                    // 图片优先级大于文案
                    if (MaterialSpecificationConstant.ITEM_TYPE_IMAGE == rspMsItemContent.getItemType()) {
                        sortItemContent.get(msItemContent.getItemIndex() - 1).getValue().add(0, rspMsItemContent);
                    } else {
                        sortItemContent.get(msItemContent.getItemIndex() - 1).getValue().add(sortItemContent.get(msItemContent.getItemIndex() - 1).getValue().size(), rspMsItemContent);
                    }

                }
                rspMaterialSpecificationDto.setItemContentList(sortItemContent);
            }
            msItemContentList.add(rspMaterialSpecificationDto);
        }
        rspActivityDto.setMsList(msItemContentList);
        return rspActivityDto;
    }

    @Override
    public List<Long> getIdsByName(String name) throws TuiaMediaException {
        return activityDAO.getIdsByName(name);
    }

    @Override
    public List<RspMaterialSpecificationDto> getMsItemList() throws TuiaMediaException {
        ReqMaterialSpecificationDto req = new ReqMaterialSpecificationDto();
        req.setSort("gmt_modified");
        req.setOrder("desc");
        req.setPageSize(-1);
        List<RspMaterialSpecificationDto> list = materialSpecificationService.getMaterialSpecificationList(req);
        List<RspMaterialSpecificationDto> result = new ArrayList<>();
        for (RspMaterialSpecificationDto ms : list) {
            result.add(materialSpecificationService.getMaterialSpecificationDetail(ms.getId()));
        }
        return result;
    }
}
