package cn.com.duiba.tuia.media.service.impl;

import java.util.*;

import cn.com.duiba.tuia.media.service.*;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import cn.com.duiba.tuia.media.api.dto.SlotDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqActivitySort;
import cn.com.duiba.tuia.media.api.dto.req.ReqIdAndType;
import cn.com.duiba.tuia.media.api.dto.rsp.RspActivityDto;
import cn.com.duiba.tuia.media.api.constant.ActivityConstant;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.dao.ActivitySortDAO;
import cn.com.duiba.tuia.media.dataobject.ActivitySlotDO;
import cn.com.duiba.tuia.media.domain.ActivitySortDto;

/**
 * Function: ActivitySortServiceImpl
 *
 * @author xuyenan
 * @createTime 2016/10/24
 */
@Service
public class ActivitySortServiceImpl implements ActivitySortService {

    protected Logger                logger    = LoggerFactory.getLogger(getClass());

    private static final long       SORT_STEP = 10000;

    @Autowired
    private ActivitySortDAO         activitySortDAO;
    @Autowired
    private SlotService             slotService;
    @Autowired
    private ActivityService         activityService;
    @Autowired
    private ActivitySlotService     activitySlotService;
    @Autowired
    private MediaAppActivityService mediaAppActivityService;

    @Override
    public int insertBatchActivitySort(List<ActivitySortDto> activitySortDtoList) throws TuiaMediaException {
        return activitySortDAO.insertBatch(activitySortDtoList);
    }

    @Override
    public List<ActivitySortDto> selectByCondition(ReqActivitySort reqActivitySort) throws TuiaMediaException {
        // 将名称模糊查询转换为ID列表
        if (!StringUtils.isEmpty(reqActivitySort.getName())) {
            List<Long> list = activityService.getIdsByName(reqActivitySort.getName());
            if (CollectionUtils.isEmpty(list)) {
                return new ArrayList<>();
            }
            reqActivitySort.setActivityIds(list);
        }
        return activitySortDAO.selectByCondition(reqActivitySort);
    }

    @Override
    public int selectByConditionAmount(ReqActivitySort reqActivitySort) throws TuiaMediaException {
        return activitySortDAO.selectByConditionAmount(reqActivitySort);
    }

    @Override
    @Transactional
    public boolean deleteActivity(Long activityId, Integer activityType, Long slotId) throws TuiaMediaException {
        if (activityId == null || activityType == null) {
            throw new TuiaMediaException(ErrorCode.E0001005);
        }
        return activitySortDAO.deleteActivity(slotId, activityId, activityType) > 0;
    }

    @Override
    public ActivitySortDto getMaxSortValueActivitySortDto(Long slotId) throws TuiaMediaException {
        if (slotId == null) {
            throw new TuiaMediaException(ErrorCode.E0001005);
        }
        ReqActivitySort reqActivitySort = new ReqActivitySort();
        reqActivitySort.setSlotId(slotId);
        reqActivitySort.setSort("activity_sort desc,gmt_modified asc");
        reqActivitySort.setRowStart(0);
        reqActivitySort.setPageSize(1);
        List<ActivitySortDto> result = activitySortDAO.selectByCondition(reqActivitySort);
        if (result.isEmpty()) {
            return null;
        }
        return result.get(0);
    }

    @Override
    public List<RspActivityDto> getActivityForRecommendByApp(Long appId) throws TuiaMediaException {
        if (appId == null) {
            throw new TuiaMediaException(ErrorCode.E0001005);
        }
        List<ActivitySortDto> sortList = activitySortDAO.getActivityForRecommendByApp(appId);
        List<RspActivityDto> result = new ArrayList<>();
        for (ActivitySortDto activitySortDto : sortList) {
            RspActivityDto rspActivityDto = activityService.getActivityPlanDetail(activitySortDto.getActivityId(), activitySortDto.getActivityType());
            result.add(rspActivityDto);
        }
        return result;
    }

    private List<ActivitySortDto> getActivityBySlot(Long slotId) throws TuiaMediaException {
        ActivitySlotDO activitySlot = activitySlotService.getBySlotId(slotId);
        if (activitySlot == null) {
            return activitySortDAO.getActivityForRecommendBySlot(0L);
        }
        // 若没有定制列表，则返回默认列表
        List<ActivitySortDto> customSortList = activitySortDAO.getActivityForRecommendBySlot(slotId);
        if (CollectionUtils.isEmpty(customSortList)) {
            return activitySortDAO.getActivityForRecommendBySlot(0L);
        }
        if (activitySlot.getDirectMode() == ActivityConstant.DIRECT_ADVERT_MODE_ONLY) {
            return customSortList;
        }

        if (activitySlot.getDirectMode() == ActivityConstant.DIRECT_ADVERT_MODE_PRIOR) {
            // 此处设置sort值是为了下面使用TreeSet,去重和排序
            for (int i = 0; i < customSortList.size(); i++) {
                customSortList.get(i).setActivitySort((long) i);
            }
            List<ActivitySortDto> defaultSortList = activitySortDAO.getActivityForRecommendBySlot(0L);
            for (int j = 0; j < defaultSortList.size(); j++) {
                defaultSortList.get(j).setActivitySort((long) (j + customSortList.size()));
            }
            // 开启定制并且优先出定制列表，再出默认列表
            TreeSet<ActivitySortDto> customSortSet = new TreeSet<>(customSortList);
            customSortSet.addAll(defaultSortList);
            return new ArrayList<>(customSortSet);
        }
        return customSortList;
    }

    private boolean avilableMediaApp(Long appId, RspActivityDto activity) throws TuiaMediaException {
        if (activity == null || appId == null) {
            return false;
        }

        if (ActivityConstant.IS_DIRECT_MEDIA == activity.getIsDirectMedia()) {
            List<Long> directMediaApp = mediaAppActivityService.getActivityDirectedMediaApp(activity.getId(), activity.getType());
            if (CollectionUtils.isEmpty(directMediaApp)) {
                return true;
            }
            return directMediaApp.contains(appId);
        }
        return true;
    }

    @Override
    public List<RspActivityDto> getActivityForRecommendBySlot(Long slotId) throws TuiaMediaException {
        if (slotId == null) {
            throw new TuiaMediaException(ErrorCode.E0001005);
        }

        try {
            SlotDto slot = slotService.selectById(slotId);
            if (slot == null) {
                logger.error("Not found slot, slotId=" + slotId);
                throw new TuiaMediaException(ErrorCode.E0001005);
            }

            List<ActivitySortDto> sortList = getActivityBySlot(slotId);
            List<RspActivityDto> result = new ArrayList<>();
            for (ActivitySortDto activitySortDto : sortList) {
                RspActivityDto rspActivityDto = activityService.getActivityPlanDetail(activitySortDto.getActivityId(), activitySortDto.getActivityType());
                if (rspActivityDto != null && rspActivityDto.getIsEnable() == ActivityConstant.IS_ENABLE && rspActivityDto.getIsPublish() == ActivityConstant.IS_PUBLISH) {
                    if (slot.getSlotType() == SlotDto.ADSENSE_TYPE_MATERIAL_SPEC && rspActivityDto.getMsIdList() != null && rspActivityDto.getMsIdList().contains(slot.getSlotMsId())) {
                        if (avilableMediaApp(slot.getAppId(), rspActivityDto)) {
                            // 广告位规格和活动规格匹配
                            result.add(rspActivityDto);
                        }
                    } else if (slot.getSlotType() == SlotDto.ADSENSE_TYPE_MANUAL) {
                        // 手动投放类型，不走SDK接口
                    } else if (slot.getSlotType() != SlotDto.ADSENSE_TYPE_MATERIAL_SPEC) {
                        if (avilableMediaApp(slot.getAppId(), rspActivityDto)) {
                            result.add(rspActivityDto);
                        }
                    }
                }
            }
            return result;
        } catch (Exception e) {
            logger.error("ActivitySortService.getActivityForRecommendBySlot is error");
            throw new TuiaMediaException(ErrorCode.E9999999, e);
        }
    }

    @Override
    public int addActivitySort(Long slotId, List<ReqIdAndType> idAndTypes) {
        try {
            // 需要将当前时间戳与数据库最大排序值比较，若时间戳小于数据库最大值，则需要处理
            Long sort = new Date().getTime();
            ActivitySortDto result = getMaxSortValueActivitySortDto(slotId);
            if (result != null && result.getActivitySort() >= sort) {
                sort = result.getActivitySort() + SORT_STEP;
            }
            // 查询已有活动列表
            ReqActivitySort reqActivitySort = new ReqActivitySort();
            reqActivitySort.setSlotId(slotId);
            List<ActivitySortDto> existList = selectByCondition(reqActivitySort);
            Set<String> existActivity = new HashSet<>(existList.size());
            for (ActivitySortDto activitySortDto : existList) {
                existActivity.add(activitySortDto.getActivityId() + "-" + activitySortDto.getActivityType());
            }
            List<ActivitySortDto> activitySortDtoList = new ArrayList<>();
            // 查询广告位所属媒体
            SlotDto slotDto = null;
            if (!slotId.equals(0L)) {
                slotDto = slotService.selectById(slotId);
            }
            for (ReqIdAndType idAndType : idAndTypes) {
                // 定向媒体的活动只能添加到定向的媒体中
                if (slotDto != null) {
                    // 检查活动是否开启媒体定向
                    RspActivityDto rspActivityDto = activityService.getActivityPlanDetail(idAndType.getId(), idAndType.getType());
                    if (!avilableMediaApp(slotDto.getAppId(), rspActivityDto)) {
                        continue;
                    }
                }
                // 如果已经添加过则不添加
                ReqIdAndType reqIdAndType = new ReqIdAndType();
                reqIdAndType.setId(idAndType.getId());
                reqIdAndType.setType(idAndType.getType());
                if (!existActivity.contains(idAndType.getId() + "-" + idAndType.getType())) {
                    ActivitySortDto activitySortDto = new ActivitySortDto();
                    activitySortDto.setActivitySlotId(slotId);
                    activitySortDto.setActivityId(idAndType.getId());
                    activitySortDto.setActivityType(idAndType.getType());
                    activitySortDto.setActivitySort(sort);
                    sort += SORT_STEP;
                    activitySortDtoList.add(activitySortDto);
                }
            }
            if (!activitySortDtoList.isEmpty()) {
                return insertBatchActivitySort(activitySortDtoList);
            }
            return 0;
        } catch (Exception e) {
            logger.error("ActivitySortService.addActivitySort is error", e);
            return 0;
        }
    }

    @Override
    public int addDefaultActivitySort(Long slotId) {
        try {
            // 查询默认活动列表
            List<ReqIdAndType> idAndTypes = new ArrayList<>();
            ReqActivitySort condition = new ReqActivitySort();
            condition.setSlotId(0L);
            condition.setSort("activity_sort asc,gmt_modified desc");
            List<ActivitySortDto> needList = selectByCondition(condition);
            for (ActivitySortDto activitySortDto : needList) {
                ReqIdAndType idAndType = new ReqIdAndType();
                idAndType.setId(activitySortDto.getActivityId());
                idAndType.setType(activitySortDto.getActivityType());
                idAndTypes.add(idAndType);
            }
            return addActivitySort(slotId, idAndTypes);
        } catch (Exception e) {
            logger.error("ActivitySortService.addDefaultActivitySort is error", e);
            return 0;
        }
    }

    private ActivitySortDto resetActivitySort(Long slotId, ActivitySortDto startSort) throws TuiaMediaException {
        activitySortDAO.updateSortBatch(slotId, startSort.getActivitySort(), SORT_STEP);
        return activitySortDAO.select(slotId, startSort.getActivityId(), startSort.getActivityType());
    }

    private int getActivitySortLocation(Long slotId, Long activityId, Integer activityType) throws TuiaMediaException {
        ActivitySortDto activitySort = activitySortDAO.select(slotId, activityId, activityType);
        return activitySortDAO.selectSortLocation(slotId, activitySort.getActivitySort());
    }

    private long getActivitySortValue(Long slotId, List<ActivitySortDto> aroundSortList, int offset) throws TuiaMediaException {
        if (aroundSortList.size() == 1) {
            ActivitySortDto aroundSortMin = aroundSortList.get(0);
            return aroundSortMin.getActivitySort() + (offset < 0 ? -SORT_STEP : SORT_STEP);
        } else {
            ActivitySortDto aroundSortMin = aroundSortList.get(0);
            ActivitySortDto aroundSortMax = aroundSortList.get(1);
            if (aroundSortMax.getActivitySort() - aroundSortMin.getActivitySort() < 2) { // 间隔太小，中间无法插入新排序记录，重置排序字段间隔
                aroundSortMax = resetActivitySort(slotId, aroundSortMax);
            }
            return (aroundSortMax.getActivitySort() + aroundSortMin.getActivitySort()) / 2;
        }
    }

    private Long getActivitySort(Long slotId, int destLocation, int offset) throws TuiaMediaException {
        ReqActivitySort reqActivitySort = new ReqActivitySort();
        reqActivitySort.setSlotId(slotId);
        reqActivitySort.setPageSize(destLocation == 1 ? 1 : 2);
        reqActivitySort.setRowStart((destLocation - 1 + offset) < 0 ? 0 : (destLocation - 1 + offset));
        reqActivitySort.setSort("activity_sort asc,gmt_modified desc");
        List<ActivitySortDto> aroundSortList = activitySortDAO.selectByCondition(reqActivitySort);
        if (aroundSortList.isEmpty()) {
            return null;
        }
        return getActivitySortValue(slotId, aroundSortList, offset);
    }

    @Override
    public boolean sortActivity(Long slotId, Long activityId, Integer activityType, int destLocation) throws TuiaMediaException {
        if (slotId == null || activityId == null || activityType == null || destLocation <= 0) {
            return false;
        }

        int origLocation = getActivitySortLocation(slotId, activityId, activityType);
        if (destLocation == origLocation) {
            return false;
        }

        int offset = destLocation < origLocation ? -1 : 0; // -1 往前排序, 0 往后排序
        Long sort = getActivitySort(slotId, destLocation, offset);
        if (sort == null) {
            return false;
        }
        return activitySortDAO.updateSort(slotId, activityId, activityType, sort) > 0;
    }

    @Override
    public List<RspActivityDto> getActivityForCenter(Long slotId) throws TuiaMediaException {
        List<ActivitySortDto> activitySortList = activitySortDAO.getActivityForRecommendBySlot(0L);
        List<RspActivityDto> rspActivityList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(activitySortList)) {
            for (ActivitySortDto activitySortDto : activitySortList) {
                rspActivityList.add(activityService.getActivityPlanDetail(activitySortDto.getActivityId(), activitySortDto.getActivityType()));
            }
        }
        return rspActivityList;
    }
}
