/**
 * Project Name:media-biz<br/>
 * File Name:AdsenseServiceImpl.java<br/>
 * Package Name:cn.com.duiba.tuia.media.service.impl<br/>
 * Date:2016年9月26日下午7:25:36<br/>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 */

package cn.com.duiba.tuia.media.service.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import cn.com.duiba.tuia.media.api.constant.MaterialSpecificationConstant;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.tuia.media.api.dto.PageResultDto;
import cn.com.duiba.tuia.media.api.dto.SlotDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqGetSlotsByPageDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqUpdateSlot;
import cn.com.duiba.tuia.media.api.dto.rsp.RspAdsenseSlotListDto;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.common.utils.CollectionUtil;
import cn.com.duiba.tuia.media.dao.MediaAppDao;
import cn.com.duiba.tuia.media.dao.SlotDAO;
import cn.com.duiba.tuia.media.dataobject.MaterialSpecificationDO;
import cn.com.duiba.tuia.media.model.IdAndName;
import cn.com.duiba.tuia.media.model.req.ReqCreateSlot;
import cn.com.duiba.tuia.media.model.req.ReqEditSlotEnableStatus;
import cn.com.duiba.tuia.media.model.req.ReqSlotList;
import cn.com.duiba.tuia.media.model.rsp.RspPageResult;
import cn.com.duiba.tuia.media.model.rsp.RspSlot;
import cn.com.duiba.tuia.media.service.MaterialSpecificationService;
import cn.com.duiba.tuia.media.service.SlotService;

/**
 * 广告位置service实现类 <br/>
 * Date: 2016年9月26日 下午7:25:36 <br/>
 * 
 * @author ZFZ
 * @version
 * @since JDK 1.6
 * @see
 */

@Service
public class SlotServiceImpl implements SlotService {

    @Autowired
    private SlotDAO                      slotDAO;

    @Autowired
    private SlotCacheService             slotCacheService;

    @Autowired
    private MediaAppDao                  mediaAppDao;

    @Autowired
    private MaterialSpecificationService materialSpecificationService;

    @Override
    public Boolean insert(ReqCreateSlot req) throws TuiaMediaException {

        SlotDto slotDto = new SlotDto();
        slotDto.setMediaId(req.getMediaId());
        slotDto.setSlotName(req.getSlotName());
        // 在广告位"去掉审核流程需求"中，创建广告位时广告位为默认开启状态。
        slotDto.setEnableStatus(SlotDto.ENABLE_STATUS_OPEN);
        slotDto.setSlotType(req.getSlotType());
        // 在广告位"去掉审核流程需求"中，创建广告位时默认为审核通过。
        slotDto.setCheckStatus(SlotDto.CHECK_STATUS_PASS);
        slotDto.setSlotMsId(req.getMsId());
        slotDto.setAppId(req.getAppId());
        slotDto.setStrategyId(req.getStrategyId());
        slotDto.setVisibleOfCloseButton(false);
        slotDto.setVisibleOfIco(false);

        slotDAO.insert(slotDto);

        // 新增缓存
        slotCacheService.updateSlotCache(slotDto.getId());
        // 通过广告ID更新广告ID和广告名称
        if (slotDto.getId() != null) {
            slotCacheService.updateSoltIdAndNameCache(slotDto.getId()); // 通过广告ID更新广告ID和广告名称
        }
        return true;
    }

    @Override
    public Boolean update(ReqUpdateSlot param) throws TuiaMediaException {

        SlotDto slotDto = new SlotDto();
        slotDto.setId(param.getSlotId());
        slotDto.setSlotName(param.getSlotName());

        // 在广告位"去掉审核流程需求"中，修改广告位时广告位为默认开启状态。
        slotDto.setEnableStatus(SlotDto.ENABLE_STATUS_OPEN);

        // 在广告位"去掉审核流程需求"中，修改广告位时默认为审核通过。
        slotDto.setCheckStatus(SlotDto.CHECK_STATUS_PASS);

        slotDto.setMediaId(param.getMediaId());
        slotDto.setStrategyId(param.getStrategyId());
        slotDAO.update(slotDto);
        // 更新缓存
        slotCacheService.updateSlotCache(slotDto.getId());
        slotCacheService.updateSoltIdAndNameCache(slotDto.getId()); // 通过广告ID更新广告ID和广告名称
        return true;
    }

    @Override
    public RspPageResult<RspSlot> selectListByPage(ReqSlotList param) throws TuiaMediaException {
        int count = slotDAO.selectAmountByPage(param);
        // 当总数查询结果有数据时，再查询具体数据
        List<SlotDto> slotDtos;
        List<RspSlot> rspSlots = null;
        List<Long> appIds;
        if (count > 0 && count >= param.getRowStart()) {
            slotDtos = slotDAO.selectListByPage(param);
            int size = slotDtos.size();
            appIds = new ArrayList<>(size);
            for (SlotDto slot : slotDtos) {
                appIds.add(slot.getAppId());
            }

            List<IdAndName> appIdAndNameList;

            // appId 与 appName Map
            Map<Long, String> appMap;
            appIdAndNameList = mediaAppDao.selectAppIdAndName(appIds);
            appMap = new HashMap<>(size);
            for (IdAndName appVO : appIdAndNameList) {
                appMap.put(appVO.getId(), appVO.getName());
            }

            rspSlots = new ArrayList<>(size);

            Map<Long, MaterialSpecificationDO> msMap = materialSpecificationService.getMapByIds(CollectionUtil.getFieldList(slotDtos, "slotMsId"));

            for (SlotDto slot : slotDtos) {
                RspSlot rspSlot = new RspSlot();
                rspSlot.setSlotId(slot.getId());
                rspSlot.setSlotName(slot.getSlotName());
                rspSlot.setAppId(slot.getAppId());
                rspSlot.setAppName(appMap.get(slot.getAppId()));
                rspSlot.setCheckStatus(slot.getCheckStatus());
                rspSlot.setModifiedDate(new DateTime(slot.getGmtModified()).toString("yyyy-MM-dd"));
                rspSlot.setSlotType(slot.getSlotType());
                rspSlot.setEnableStatus(slot.getEnableStatus());
                rspSlot.setActivityUrl(slot.getActivityUrl());
                rspSlot.setValid(true);
                rspSlot.setMsName(SlotDto.getSlotTypeNameBySlotType(slot.getSlotType()));
                // 规格若被删除，则状态置为无效
                if (slot.getSlotMsId() != null) {
                    MaterialSpecificationDO ms = msMap.get(slot.getSlotMsId());
                    if (ms != null) {
                        rspSlot.setMsName(ms.getTitle());
                        rspSlot.setValid(MaterialSpecificationConstant.MS_NOT_DELETE == ms.getIsDelete());
                    }
                }
                rspSlots.add(rspSlot);
            }
        }

        return new RspPageResult<>(count, rspSlots, param.getPageSize());
    }

    @Override
    public SlotDto selectById(Long slotId) throws TuiaMediaException {
        return slotDAO.selectById(slotId);
    }

    @Override
    public Boolean updateEnableStatus(ReqEditSlotEnableStatus param) throws TuiaMediaException {
        long slotId = param.getSlotId();
        SlotDto slotDto = selectById(slotId);
        Integer enableStatus = param.getEnableStatus() ? SlotDto.ENABLE_STATUS_OPEN : SlotDto.ENABLE_STATUS_CLOSE;
        if (checkAppAndMediaStatus(slotDto)) {
            slotDAO.updateEnableStatus(slotId, enableStatus);
            // 更新缓存
            slotCacheService.updateSlotCache(slotId);
        }

        return true;
    }

    /**
     * 判断是否可以更新广告位开启状态. <br/>
     * 1.如果是关闭广告位操作，可以直接更新. <br/>
     * 2.如果是开启广告位操作，需要先判断媒体及App状态. <br/>
     *
     * @author ZFZ
     * @param adsenseDto
     * @return
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    private boolean checkAppAndMediaStatus(SlotDto adsenseDto) throws TuiaMediaException {
        // 判断广告位审核状态，只有审核通过的可以开启或者关闭
        if (SlotDto.CHECK_STATUS_PASS != adsenseDto.getCheckStatus()) {
            throw new TuiaMediaException(ErrorCode.E0304006);
        }
        return true;
    }

    @Override
    public PageResultDto<RspAdsenseSlotListDto> selectAdsenseListByPage(ReqGetSlotsByPageDto req) throws TuiaMediaException {

        String appName = req.getAppName();
        List<Long> appIdsList = null;
        if (StringUtils.isNotBlank(appName)) {
            // 通过媒体名称模糊查询媒体ID
            appIdsList = mediaAppDao.selectIdsByName(appName);
            if (CollectionUtils.isEmpty(appIdsList)) {
                return new PageResultDto<>(0, null, req.getPageSize());
            }
        }

        int count = slotDAO.selectAdsenseAmountByPage(req, appIdsList);
        // 当总数查询结果有数据时，再查询具体数据
        List<SlotDto> slotDtos;
        List<RspAdsenseSlotListDto> rspSlots = null;
        List<Long> appIds;

        if (count > 0 && count >= req.getRowStart()) {
            req.setRowStart(req.getPageSize() * (req.getCurrentPage() - 1));
            slotDtos = slotDAO.selectListAdsenseByPage(req, appIdsList);
            int size = slotDtos.size();
            appIds = new ArrayList<>(size);
            for (SlotDto slot : slotDtos) {
                appIds.add(slot.getAppId());
            }

            List<IdAndName> appIdAndNameList;

            // appId 与 appName Map
            Map<Long, String> appMap;
            appIdAndNameList = mediaAppDao.selectAppIdAndName(appIds);
            appMap = new HashMap<>(size);
            for (IdAndName appVO : appIdAndNameList) {
                appMap.put(appVO.getId(), appVO.getName());
            }
            Map<Long, MaterialSpecificationDO> msMap = materialSpecificationService.getMapByIds(CollectionUtil.getFieldList(slotDtos, "slotMsId"));
            rspSlots = new ArrayList<>(size);
            for (SlotDto slotDto : slotDtos) {
                RspAdsenseSlotListDto rspDto = new RspAdsenseSlotListDto();
                rspDto.setAppId(slotDto.getAppId());
                rspDto.setAppName(appMap.get(slotDto.getAppId()));
                rspDto.setCheckStatus(slotDto.getCheckStatus());
                rspDto.setCreateDate(new DateTime(slotDto.getGmtCreate()).toString("yyyy-MM-dd"));
                rspDto.setSlotId(slotDto.getId());
                rspDto.setSlotName(slotDto.getSlotName());
                rspDto.setSlotType(slotDto.getSlotType());
                if (slotDto.getSlotMsId() == null) {
                    // 如果是老数据，则状态都是有效
                    rspDto.setValid(true);
                    rspDto.setMsName(SlotDto.getSlotTypeNameBySlotType(slotDto.getSlotType()));
                } else {
                    // 规格若被删除，则状态置为无效
                    MaterialSpecificationDO ms = msMap.get(slotDto.getSlotMsId());
                    if (ms != null) {
                        rspDto.setMsName(ms.getTitle());
                        rspDto.setValid(MaterialSpecificationConstant.MS_NOT_DELETE == ms.getIsDelete());
                    }
                }
                rspSlots.add(rspDto);
            }
        }

        return new PageResultDto<>(count, rspSlots, req.getPageSize());
    }

    @Override
    public boolean batchUpdateCheckStatus(List<Long> slotIds, Integer checkType) throws TuiaMediaException {
        slotDAO.batchUpdateCheckStatus(slotIds, checkType);

        if (checkType == 1) {
            slotDAO.batchUpdateEnableStatus(slotIds, SlotDto.ENABLE_STATUS_OPEN);
        }
        // 更新缓存
        slotCacheService.updateBatchSlotCache(slotIds);
        return true;
    }

    @Override
    public List<SlotDto> getListDetail(List<Long> slotIds) throws TuiaMediaException {
        return slotDAO.selectListDetail(slotIds);
    }

    @Override
    public List<Long> selectIdsByName(String slotName) throws TuiaMediaException {
        return slotDAO.selectIdsByName(slotName);
    }

    @Override
    public List<IdAndName> selectAppIdAndName(List<Long> slotIds) throws TuiaMediaException {
        return slotDAO.selectAppIdAndName(slotIds);
    }

    @Override
    public Boolean updateSlot(SlotDto slotDto) throws TuiaMediaException {
        slotDAO.update(slotDto);
        // 更新缓存
        slotCacheService.updateSlotCache(slotDto.getId());
        return true;
    }

    @Override
    public Boolean closeSlotByMsId(Long msId) throws TuiaMediaException {
        slotDAO.closeSlotByMsId(msId);
        List<Long> ids = slotDAO.getIdsByMsId(msId);
        slotCacheService.updateBatchSlotCache(ids);
        return true;
    }

    @Override
    public Boolean deleteSlotById(Long slotId, Long mediaId) throws TuiaMediaException {
        return slotDAO.updateDelStatus(slotId, mediaId) == 0;
    }

}
