/**
 * Project Name:media-biz<br>
 * File Name:StrategyServiceImpl.java<br>
 * Package Name:cn.com.duiba.tuia.media.service.impl<br>
 * Date:2016年9月27日下午2:02:06<br>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.<br>
 */

package cn.com.duiba.tuia.media.service.impl;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.dao.ShieldStrategyDAO;
import cn.com.duiba.tuia.media.domain.ShieldStrategyDto;
import cn.com.duiba.tuia.media.model.req.ReqShieldListStrategy;
import cn.com.duiba.tuia.media.model.req.ReqStrategyPageQuery;
import cn.com.duiba.tuia.media.model.rsp.RspIdAndValue;
import cn.com.duiba.tuia.media.model.rsp.RspPageResult;
import cn.com.duiba.tuia.media.model.rsp.RspShieldStrategy;
import cn.com.duiba.tuia.media.model.rsp.RspStrategyList;
import cn.com.duiba.tuia.media.service.StrategyService;
import cn.com.duiba.tuia.media.utils.RemoveDuplicateTool;
import cn.com.duiba.tuia.media.utils.StringTool;
import cn.com.duiba.wolf.redis.RedisClient;

import com.alibaba.dubbo.common.utils.CollectionUtils;

/**
 * ClassName: StrategyServiceImpl <br/>
 * Function: 屏蔽策略. <br/>
 * date: 2016年9月27日 下午2:02:06 <br/>
 *
 * @author leiliang
 * @version
 * @since JDK 1.6
 */
@Service
public class StrategyServiceImpl implements StrategyService {

    @Autowired
    private ShieldStrategyDAO shieldStrategyDAO;

    @Autowired
    private RedisClient       redisClient;

    @Autowired
    private SlotCacheService  slotCacheService;

    @Override
    public RspPageResult<RspStrategyList> getPageList(ReqStrategyPageQuery req) throws TuiaMediaException {
        int amount = shieldStrategyDAO.selectAmountByPage(req);
        List<RspStrategyList> strategies = null;
        if (amount > 0 && amount >= req.getRowStart()) {
            // 只有有记录时，才去库里查询
            req.setRowStart(req.getPageSize() * (req.getCurrentPage() - 1));
            List<ShieldStrategyDto> strategyDtos = shieldStrategyDAO.selectListByPage(req);
            strategies = new ArrayList<>(strategyDtos.size());
            for (ShieldStrategyDto strategyDto : strategyDtos) {
                RspStrategyList rspStrategy = new RspStrategyList();
                rspStrategy.setId(strategyDto.getId());
                rspStrategy.setStrategyName(strategyDto.getStrategyName());
                rspStrategy.setCreateDate(new DateTime(strategyDto.getGmtCreate()).toString("yyyy-MM-dd"));
                rspStrategy.setIndustryFlag(StringUtils.isNotBlank(strategyDto.getShieldIndustries()));
                rspStrategy.setAdvertTagNums(StringUtils.isNotBlank(strategyDto.getAdvertTagNums()));
                rspStrategy.setAdvertiserFlag(StringUtils.isNotBlank(strategyDto.getShieldUrls()));

                strategies.add(rspStrategy);
            }
        }
        return new RspPageResult<>(amount, strategies, req.getPageSize());
    }

    @Override
    public Boolean delete(Long mediaId, Long strategyId) throws TuiaMediaException {
        // 校验该屏蔽策略是否是该媒体用户的
        if (isBelongMedia(strategyId, mediaId)) {
            shieldStrategyDAO.delete(strategyId);
        }

        // 删除缓存
        slotCacheService.deleteStrategyCache(strategyId);
        return true;
    }

    /**
     * 检查屏蔽策略是否是媒体用户创建的.
     *
     * @param strategyId 屏蔽策略id
     * @param mediaId 媒体用户id
     * @return true：是
     * @throws TuiaMediaException the tuia media exception
     */
    private boolean isBelongMedia(Long strategyId, Long mediaId) throws TuiaMediaException {
        // 校验该屏蔽策略是否是该媒体用户的
        ShieldStrategyDto shieldStrategy = shieldStrategyDAO.selectById(strategyId);
        if (shieldStrategy == null) {
            throw new TuiaMediaException(ErrorCode.E0304002);
        }
        if (shieldStrategy.getMediaId() == null || mediaId.longValue() != shieldStrategy.getMediaId()) {
            throw new TuiaMediaException(ErrorCode.E0304001);
        }

        return true;
    }

    @Override
    public Boolean create(ReqShieldListStrategy param, Long mediaId) throws TuiaMediaException {
        ShieldStrategyDto req = new ShieldStrategyDto();
        req.setStrategyName(param.getStrategyName());
        req.setShieldIndustries(StringTool.getStringByList(param.getShieldIndustries()));
        req.setShieldUrls(getShieldUrls(param.getShieldAdvertisers()));
        req.setMediaId(mediaId);

        // 当前用户创建的屏蔽策略不可重名
        if (shieldStrategyDAO.isNotExist(null, mediaId, req.getStrategyName())) {
            shieldStrategyDAO.insert(req);

            // 新增缓存
            slotCacheService.updateStrategyById(req.getId());
        }

        return true;
    }

    @Override
    public Boolean update(ReqShieldListStrategy param, Long mediaId) throws TuiaMediaException {
        ShieldStrategyDto req = new ShieldStrategyDto();
        Long id = param.getId();
        req.setId(id);
        req.setStrategyName(param.getStrategyName());
        req.setShieldIndustries(StringTool.getStringByList(param.getShieldIndustries()));
        req.setShieldUrls(getShieldUrls(param.getShieldAdvertisers()));

        if (isBelongMedia(id, mediaId) && shieldStrategyDAO.isNotExist(id, mediaId, req.getStrategyName())) {
            shieldStrategyDAO.update(req);

            // 更新缓存
            slotCacheService.updateStrategyById(id);
        }

        return true;
    }

    /**
     * 得到广告主URL屏蔽列表.
     *
     * @param shieldAdvertisers the shield advertisers
     * @return the shield urls
     */
    private String getShieldUrls(List<String> shieldAdvertisers) {
        List<String> shieldUrls = null;

        // 去重去空
        if (CollectionUtils.isNotEmpty(shieldAdvertisers)) {
            shieldUrls = new ArrayList<>(shieldAdvertisers.size());
            for (String shieldAdvertiser : shieldAdvertisers) {
                if (StringUtils.isNotBlank(shieldAdvertiser)) {
                    shieldUrls.add(shieldAdvertiser.trim());
                }
            }
        }
        return StringTool.getStringByList(RemoveDuplicateTool.doRemoveDuplicateUrl(shieldUrls));
    }

    @Override
    public Boolean updateByManager(ShieldStrategyDto strategyDto) throws TuiaMediaException {

        shieldStrategyDAO.updateByManager(strategyDto);

        // 更新缓存
        slotCacheService.updateStrategyById(strategyDto.getId());

        return true;
    }

    @Override
    public List<RspIdAndValue> getListByMediaId(Long mediaId) throws TuiaMediaException {
        List<ShieldStrategyDto> strategyDtos = shieldStrategyDAO.selectListByMediaId(mediaId);
        List<RspIdAndValue> rspIdAndValue = new ArrayList<>(strategyDtos.size());
        for (ShieldStrategyDto strategyDto : strategyDtos) {
            RspIdAndValue dto = new RspIdAndValue();
            dto.setId(Long.toString(strategyDto.getId()));
            dto.setValue(strategyDto.getStrategyName());
            rspIdAndValue.add(dto);
        }

        return rspIdAndValue;
    }

    @Override
    public RspShieldStrategy getDetail(Long strategyId) throws TuiaMediaException {
        ShieldStrategyDto shield = shieldStrategyDAO.selectById(strategyId);
        RspShieldStrategy shieldStrategy = null;
        if (shield != null) {
            shieldStrategy = new RspShieldStrategy();
            shieldStrategy.setId(strategyId);
            shieldStrategy.setIndustryFlag(StringUtils.isNotBlank(shield.getShieldIndustries()));
            shieldStrategy.setAdvertiserFlag(StringUtils.isNotBlank(shield.getShieldUrls()));
            shieldStrategy.setAdvertTagNumsFlag(StringUtils.isNotBlank(shield.getAdvertTagNums()));
            shieldStrategy.setShieldIndustries(StringTool.getStringListByStr(shield.getShieldIndustries()));
            shieldStrategy.setAdvertTagNums(StringTool.getStringListByStr(shield.getAdvertTagNums()));
            shieldStrategy.setShieldAdvertisers(StringTool.getStringListByStr(shield.getShieldUrls()));
            shieldStrategy.setStrategyName(shield.getStrategyName());
        }
        return shieldStrategy;
    }
}
