package cn.com.duiba.tuia.media.common.tool;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Stack;

import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 
 * ClassName: TimeProfileTool <br/>
 * date: 2016年12月8日 下午7:00:43 <br/>
 *
 * @author ZFZ
 * @version 
 * @since JDK 1.6
 */
public class TimeProfileTool {

    private static Logger                               log       = LoggerFactory.getLogger(TimeProfileTool.class);

    private static ThreadLocal<Stack<TimeProfileStack>> stack     = new ThreadLocal<>();

    private static ThreadLocal<List<TimeProfileStack>>  logs      = new ThreadLocal<>();

    public static final ThreadLocal<HttpServletRequest> REQUEST   = new ThreadLocal<>();

    public static final int                             THRESHOLD = 500;                                           // 超时时间，单位毫秒

    private static final String                         LOGSTR    = "TimeProfile timeout ";

    private TimeProfileTool                             timeProTool;

    /**
     * 
     * getTimeProfileTool:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author ZFZ
     * @return
     * @since JDK 1.6
     */
    public TimeProfileTool getTimeProfileTool() {
        if (timeProTool == null) {
            timeProTool = new TimeProfileTool();
        }
        return timeProTool;
    }

    /**
     * 
     * enter:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author ZFZ
     * @since JDK 1.6
     */
    public static void enter() {
        enter("");
    }

    /**
     * 
     * enter:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author ZFZ
     * @param tag
     * @since JDK 1.6
     */
    public static void enter(String tag) {
        if (stack.get() == null) {
            return;
        }
        TimeProfileStack t = new TimeProfileStack();
        t.setEnterTime(new Date());
        t.setTag(tag);
        t.setDeep(stack.get().size());

        t.index = logs.get().size() + stack.get().size();
        stack.get().push(t);
    }

    /**
     * 
     * release:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author ZFZ
     * @since JDK 1.6
     */
    public static void release() {
        if (stack.get() == null) {
            return;
        }
        TimeProfileStack t = stack.get().pop();
        t.setReleaseTime(new Date());
        logs.get().add(t);
    }

    /**
     * 
     * start:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author ZFZ
     * @since JDK 1.6
     */
    public static void start() {
        stack.set(new Stack<TimeProfileTool.TimeProfileStack>());
        logs.set(new ArrayList<TimeProfileTool.TimeProfileStack>());
        enter();
    }

    /**
     * 
     * end:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author ZFZ
     * @since JDK 1.6
     */
    public static void end() {
        TimeProfileStack t = stack.get() == null ? null : stack.get().pop();
        if (t == null) {
            return;
        }
        t.setReleaseTime(new Date());
        logs.get().add(t);

        Long timeconsume = t.getReleaseTime().getTime() - t.getEnterTime().getTime();
        if (timeconsume > THRESHOLD) {
            // 输出日志
            StringBuilder sb = new StringBuilder();
            if (REQUEST.get() != null) {
                String url = RequestTool.getRequestUrl(REQUEST.get());
                sb.append(LOGSTR).append(timeconsume).append("ms >").append(THRESHOLD).append("ms, url=").append(url);
            } else {
                sb.append(LOGSTR).append(timeconsume).append("ms ");
            }
            List<TimeProfileStack> list = new ArrayList<>(logs.get());
            Collections.sort(list, new Comparator<TimeProfileStack>() {

                @Override
                public int compare(TimeProfileStack o1, TimeProfileStack o2) {
                    if (o1.index > o2.index) {
                        return 1;
                    } else if (o1.index < o2.index) {
                        return -1;
                    }
                    return 0;
                }
            });
            for (TimeProfileStack s : list) {
                sb.append("\r\n\t");
                for (int i = 0; i < s.getDeep(); i++) {
                    sb.append("-");
                }
                Long consume = s.getReleaseTime().getTime() - s.getEnterTime().getTime();
                sb.append(consume * 100 / timeconsume).append("% ").append(consume).append("ms");
                if (s.getTag() != null) {
                    sb.append("  ").append(s.getTag());
                }

                if (s.getStackTraceElement() != null) {
                    StackTraceElement ste = s.getStackTraceElement();
                    sb.append("  ").append(ste.getClassName()).append(".").append(ste.getMethodName()).append(" line:").append(ste.getLineNumber());
                }
            }
            log.info(sb.toString());
        }
        REQUEST.set(null);
        stack.set(null);
        logs.set(null);
    }

    /**
     * 
     * dump:(dump堆栈信息). <br/>
     *
     * @author ZFZ
     * @param ms 毫秒 ，超过此毫秒时才会dump
     * @return
     * @since JDK 1.6
     */
    public static String dump(long ms) {
        if (logs.get() == null) {
            return null;
        }
        TimeProfileStack t = stack.get().pop();
        t.setReleaseTime(new Date());
        logs.get().add(t);

        Long timeconsume = 0L;
        for (TimeProfileStack s : logs.get()) {
            if (s.deep == 0) {
                timeconsume += s.getReleaseTime().getTime() - s.getEnterTime().getTime();
            }
        }
        if (timeconsume < ms) {
            return null;
        }
        // 输出日志
        StringBuilder sb = new StringBuilder();
        sb.append(LOGSTR).append(timeconsume).append("ms ");
        List<TimeProfileStack> list = new ArrayList<>(logs.get());
        Collections.sort(list, new Comparator<TimeProfileStack>() {

            @Override
            public int compare(TimeProfileStack o1, TimeProfileStack o2) {
                if (o1.index > o2.index) {
                    return 1;
                } else if (o1.index < o2.index) {
                    return -1;
                }
                return 0;
            }
        });
        for (TimeProfileStack s : list) {
            Long consume = s.getReleaseTime().getTime() - s.getEnterTime().getTime();
            if (consume == 0) {
                continue;
            }
            sb.append("\r\n\t");
            for (int i = 0; i < s.getDeep(); i++) {
                sb.append("  ");
            }
            sb.append("-").append(consume * 100 / timeconsume).append("%").append("  ").append(consume).append("ms");
            if (s.getTag() != null) {
                sb.append("  ").append(s.getTag());
            }

            if (s.getStackTraceElement() != null) {
                StackTraceElement ste = s.getStackTraceElement();
                sb.append("  ").append(ste.getClassName()).append(".").append(ste.getMethodName()).append(" line:").append(ste.getLineNumber());
            }
        }
        REQUEST.set(null);
        stack.set(null);
        logs.set(null);

        return sb.toString();
    }

    /**
     * 
     * ClassName: TimeProfileStack <br/>
     * date: 2016年12月8日 下午6:03:29 <br/>
     *
     * @author ZFZ
     * @version TimeProfileTool
     * @since JDK 1.6
     */
    public static class TimeProfileStack {

        private Date              enterTime;

        private Date              releaseTime;

        private int               deep;

        private String            tag;

        int                       index;

        private StackTraceElement stackTraceElement;

        public Date getEnterTime() {
            return enterTime;
        }

        public void setEnterTime(Date enterTime) {
            this.enterTime = enterTime;
        }

        public Date getReleaseTime() {
            return releaseTime;
        }

        public void setReleaseTime(Date releaseTime) {
            this.releaseTime = releaseTime;
        }

        public int getDeep() {
            return deep;
        }

        public void setDeep(int deep) {
            this.deep = deep;
        }

        public String getTag() {
            return tag;
        }

        public void setTag(String tag) {
            this.tag = tag;
        }

        public StackTraceElement getStackTraceElement() {
            return stackTraceElement;
        }

        public void setStackTraceElement(StackTraceElement stackTraceElement) {
            this.stackTraceElement = stackTraceElement;
        }
    }
}
